// Torus.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Torus solid with the hole pointing towards the normal. The
// major radius is the radius of extrusion; the minor radius is the
// radius of the extruded circle.

#ifndef TOVERO_MATH_TORUS_HPP_
#define TOVERO_MATH_TORUS_HPP_

#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <string>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Torus : public Solid
    {
    public:
      // Torus defaults to centered at origin with the hole pointing
      // towards the X axis
      // constructors/copy
      Torus(const std::string& name = "");
      Torus(const Point& center,
            const Unit_vector& normal,
            const Distance& major_radius,
            const Distance& minor_radius,
            const std::string& name = "");
      virtual Solid* clone_solid() const { return new Torus(*this); }
      //   Note: let compiler generate the default copy constructor and operator=
      // accessors/mutators
      const Point& center() const { return m_center; }
      void set_center(const Point& center) { m_center = center; }
      const Unit_vector& normal() const { return m_normal; }
      void set_normal(const Unit_vector& normal) { m_normal = normal; }
      const Distance& major_radius() const { return m_major_radius; }
      void set_major_radius(const Distance& radius) { m_major_radius = radius; }
      const Distance& minor_radius() const { return m_minor_radius; }
      void set_minor_radius(const Distance& radius) {  m_minor_radius = radius; }
      // predicates
      virtual bool is_valid(const Geometric_tolerances& tolerances,
                            Roan_trail::Tovero_support::Error_param& return_error) const;
      // visitor
      virtual Solid_visitor::Visit_result accept(Solid_visitor& visitor) const { return visitor.visit(*this); }
      // other
      virtual std::string solid_class() const { return "Torus"; }
    protected:
      // destructor
      virtual ~Torus() {}
    private:
      // solid parameters
      Point m_center;
      Unit_vector m_normal;    // hole points in this direction
      Distance m_major_radius; // radius of the extrusion circle
      Distance m_minor_radius; // radius of the extruded circle
    };
  }
}

#endif // TOVERO_MATH_TORUS_HPP_
