// -*- C++ -*-
//
// math.i
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// SWIG interface file for the Tovero math library
// Notes:
//   %include <file> generates wrapper code for file
//   %import <file> just reads declarations, and does not generate wrapper code

%module "libtovero_math_rb_0"

%import "../support/support.i"

// Note: code between the %{ %} brackets is copied verbatim to the wrapper file
%{
  #include "../support/common.hpp"
  #include <sstream>
  #include "../support/error/Error.hpp"
  #include "../support/error/Math_error.hpp"
  //
  #include "../support/Reference_counting_base.hpp"
  #include "../support/Reference_counting_list.hpp"
  #include "geometry/Quantity.hpp"
  //
  #include "geometry/Angle.hpp"
  #include "geometry/Area.hpp"
  #include "geometry/Distance.hpp"
  #include "geometry/Geometric_tolerances.hpp"
  #include "geometry/Matrix.hpp"
  #include "geometry/Point.hpp"
  #include "geometry/Triangle_face.hpp"
  #include "geometry/Unit_vector.hpp"
  #include "geometry/Unitless.hpp"
  #include "geometry/Units_mapper.hpp"
  #include "geometry/Vector.hpp"
  //
  #include "geometry/Solid.hpp"
  #include "geometry/Solid_visitor.hpp"
  #include "geometry/Transformation.hpp"
  //
  #include "geometry/Solid_member.hpp"
  #include "geometry/Solid_operator.hpp"
  #include "geometry/Solid_operand.hpp"
  // composite solids
  #include "geometry/Solid_combination_base.hpp"
  #include "geometry/Solid_combination.hpp"
  #include "geometry/Solid_operation.hpp"
  // solid bases
  #include "geometry/Ellipsoid_base.hpp"
  #include "geometry/General_cone_base.hpp"
  #include "geometry/Polyhedron_base.hpp"
  #include "geometry/Triangle_mesh.hpp"
  // primitive solids
  #include "geometry/Axis_aligned_box.hpp"
  #include "geometry/Boundary_triangle_mesh.hpp"
  #include "geometry/Box.hpp"
  #include "geometry/Cone.hpp"
  #include "geometry/Cylinder.hpp"
  #include "geometry/Ellipsoid.hpp"
  #include "geometry/Elliptical_cone.hpp"
  #include "geometry/Elliptical_cylinder.hpp"
  #include "geometry/General_cone.hpp"
  #include "geometry/Half_space.hpp"
  #include "geometry/Oblique_cone.hpp"
  #include "geometry/Oblique_cylinder.hpp"
  #include "geometry/Plate_triangle_mesh.hpp"
  #include "geometry/Polyhedron.hpp"
  #include "geometry/Sphere.hpp"
  #include "geometry/Torus.hpp"
  #include "geometry/Wedge.hpp"

  using Roan_trail::Tovero_math::Solid_visitor;
  using Roan_trail::Tovero_math::Unitless;
%}

//
// SWIG macros
//

%define extend_get(Class, Value_type)
  %extend Class
  {
     Value_type get(size_t index)
     {
       return (*$self)[index];
     }
   }
%enddef

%define extend_set(Class, Value_type)
  %extend Class
  {
     void set(size_t index, Value_type value)
     {
       (*$self)[index] = value;
     }
   }
%enddef

//
// Ruby-specific interface
//

#ifdef SWIGRUBY

%ignore *::operator[];
%ignore *::operator=;
%ignore *::operator*=;
%ignore *::operator/=;
%ignore *::operator+=;
%ignore *::operator-=;
%ignore *::operator!=;

%alias Roan_trail::Tovero_math::Solid_combination::add_member "<<"

namespace Roan_trail
{
  namespace Tovero_math
  {
    // Solid
    %predicate Solid::is_primitive() const;
    %predicate Solid::is_valid(const Geometric_tolerances& tolerances,
                               Roan_trail::Tovero_support::Error_param& return_error) const;
    %rename("to_s") Solid::solid_class() const;
    // Solid_member
    %rename("operand=") Solid_member::set_operand(Solid_operand& operand);
    %rename("op=") Solid_member::set_op(Solid_operator::Solid_operator_type op);
    // Solid_operand
    %rename("solid=") Solid_operand::set_solid(Solid& solid);
    %rename("transformation=") Solid_operand::set_transformation(Transformation* transformation);
    // Transformation
    %rename("matrix=") Transformation::set_matrix(const Matrix_4x4& matrix);
    %rename("translation_units=") Transformation::set_translation_units(const Distance& units);
    %rename("translation=") Transformation::set_translation(const Vector& tv);
    // composite solids
    //   Solid_combination
    //     (none)
    //   Solid_operation
    %rename("left_operand=") Solid_operation::set_left_operand(Solid_operand& left_operand);
    %rename("right_operand=") Solid_operation::set_right_operand(Solid_operand& right_operand);
    %rename("op=") Solid_operation::set_op(Solid_operator::Solid_operator_type op);
    // primitive solids
    //   Axis_aligned_box
    %rename("minimum=") Axis_aligned_box::set_minimum(const Point& minimum);
    %rename("maximum=") Axis_aligned_box::set_maximum(const Point& maximum);
    //   Boundary_triangle_mesh
    //     (none)
    //   Box
    %rename("vertex=") Box::set_vertex(const Point& vertex);
    %rename("height=") Box::set_height(const Vector& height);
    %rename("width=") Box::set_width(const Vector& width);
    %rename("depth=") Box::set_depth(const Vector& depth);
    //   Cone
    %rename("base=") Cone::set_base(const Point& base);
    %rename("height=") Cone::set_height(const Vector& height);
    %rename("base_radius=") Cone::set_base_radius(const Distance& base_radius);
    %rename("top_radius=") Cone::set_top_radius(const Distance& top_radius);
    //   Cylinder
    %rename("base=") Cylinder::set_base(const Point& base);
    %rename("height=") Cylinder::set_height(const Vector& height);
    %rename("radius=") Cylinder::set_radius(const Distance& radius);
    //   Ellipsoid
    %rename("center=") Ellipsoid::set_center(const Point& center);
    %rename("axis_a=") Ellipsoid::set_axis_a(const Vector& axis_a);
    %rename("axis_b=") Ellipsoid::set_axis_b(const Vector& axis_b);
    %rename("axis_c=") Ellipsoid::set_axis_c(const Vector& axis_c);
    //   Elliptical cone
    %rename("base=") Elliptical_cone::set_base(const Point& base);
    %rename("base_a=") Elliptical_cone::set_base_a(const Vector& base_a);
    %rename("base_b=") Elliptical_cone::set_base_b(const Vector& base_b);
    %rename("height=") Elliptical_cone::set_height(const Distance& height);
    %rename("ratio=") Elliptical_cone::set_ratio(const Unitless& ratio);
    //   Elliptical cylinder
    %rename("base=") Elliptical_cylinder::set_base(const Point& base);
    %rename("base_a=") Elliptical_cylinder::set_base_a(const Vector& base_a);
    %rename("base_b=") Elliptical_cylinder::set_base_b(const Vector& base_b);
    %rename("height=") Elliptical_cylinder::set_height(const Distance& height);
    //   General_cone
    %rename("base=") General_cone::set_base(const Point& base);
    %rename("height=") General_cone::set_height(const Vector& height);
    %rename("base_a=") General_cone::set_base_a(const Vector& base_a);
    %rename("base_b=") General_cone::set_base_b(const Vector& base_b);
    %rename("top_c=") General_cone::set_top_c(const Vector& top_c);
    %rename("top_d=") General_cone::set_top_d(const Vector& top_d);
    //   Half_space
    %rename("normal=") Half_space::set_normal(const Unit_vector& normal);
    %rename("distance=") Half_space::set_distance(const Distance& distance);
    //   Oblique cone
    %rename("base=") Oblique_cone::set_base(const Point& base);
    %rename("height=") Oblique_cone::set_height(const Vector& height);
    %rename("base_r=") Oblique_cone::set_base_r(const Vector& v);
    %rename("ratio=") Oblique_cone::set_ratio(const Unitless& ratio);
    //   Oblique cylinder
    %rename("base=") Oblique_cylinder::set_base(const Point& base);
    %rename("height=") Oblique_cylinder::set_height(const Vector& height);
    %rename("base_r=") Oblique_cylinder::set_base_r(const Vector& base_r);
    //   Plate_triangle_mesh
    //     (none)
    //   Polyhedron
    //     (none)
    //   Sphere
    %rename("center=") Sphere::set_center(const Point& center);
    %rename("radius=") Sphere::set_radius(const Distance& radius);
    //   Torus
    %rename("center=") Torus::set_center(const Point& center);
    %rename("normal=") Torus::set_normal(const Unit_vector& normal);
    %rename("major_radius=") Torus::set_major_radius(const Distance& radius);
    %rename("minor_radius=") Torus::set_minor_radius(const Distance& radius);
    //   Triangle mesh
    %rename("triangle_direction=")
    Triangle_mesh::set_triangle_direction(Triangle_face::Triangle_direction_type triangle_direction);
    %predicate Triangle_mesh::has_surface_normals() const;
    %rename("has_surface_normals=") Triangle_mesh::set_has_surface_normals(bool has_surface_normals);
    %predicate Triangle_mesh::use_surface_normals() const;
    %rename("use_surface_normals=") Triangle_mesh::set_use_surface_normals(bool use_surface_normals);
    //     Wedge
    %rename("vertex=") Wedge::set_vertex(const Point& vertex);
    %rename("height=") Wedge::set_height(const Vector& height);
    %rename("width=") Wedge::set_width(const Vector& width);
    %rename("depth=") Wedge::set_depth(const Vector& depth);
  }
}

extend_get(Roan_trail::Tovero_math::Point, const Distance&)
extend_set(Roan_trail::Tovero_math::Point, const Distance&)
//
extend_get(Roan_trail::Tovero_math::Matrix, double)
extend_set(Roan_trail::Tovero_math::Matrix, double)
//
extend_get(Roan_trail::Tovero_math::Unit_vector, const Unitless&)
//
extend_get(Roan_trail::Tovero_math::Vector, const Distance&)
extend_set(Roan_trail::Tovero_math::Vector, const Distance&)

#endif // SWIGRUBY

//
// free functions defined in Vector.hpp
//

%extend Roan_trail::Tovero_math::Unitless
{
  Roan_trail::Tovero_math::Vector __mul__(const Roan_trail::Tovero_math::Vector& v)
  {
    return *$self * v;
  }
}

%extend Roan_trail::Tovero_math::Unit_vector
{
  Roan_trail::Tovero_math::Vector __mul__(const Roan_trail::Tovero_math::Distance& d)
  {
    return *$self * d;
  }
}

%extend Roan_trail::Tovero_math::Distance
{
  Roan_trail::Tovero_math::Vector __mul__(const Roan_trail::Tovero_math::Unit_vector& u)
  {
    return *$self * u;
  }
}

// add support for the STL

%include <stl.i>
%include <std_list.i>
%include <std_vector.i>
%import <std_iostream.i>

%warnfilter(503) operator <<;

// Note: the template parameter must be qualified with a namespace
//       so the appropriate conversion (e.g. value -> pointer) is used by SWIG
%template (BoolArray) std::vector<bool>;
%template (PointArray) std::vector<Roan_trail::Tovero_math::Point>;
%template (TriangleFaceArray) std::vector<Roan_trail::Tovero_math::Triangle_face>;
%template (VectorArray) std::vector<Roan_trail::Tovero_math::Vector>;

//
// Wrap classes
//

// TODO: the order of these is dependent, otherwise SWIG needs a fully qualified
//       namespace for a parameter or a return type.  How to handle? Is there a "forward declaration" in SWIG?
%include "geometry/Quantity.hpp"
%include "geometry/Unitless.hpp"
%include "geometry/Angle.hpp"
%include "geometry/Distance.hpp"
%include "geometry/Area.hpp"
%include "geometry/Geometric_tolerances.hpp"
%include "geometry/Matrix.hpp"
%include "geometry/Unit_vector.hpp"
%include "geometry/Vector.hpp"
%include "geometry/Point.hpp"
//
%include "geometry/Triangle_face.hpp"
%include "geometry/Units_mapper.hpp"
//
%include "geometry/Solid.hpp"
%include "geometry/Solid_visitor.hpp"
%include "geometry/Transformation.hpp"
//
// Note: %feature("notabstract") used to force wrapping of constructors
%include "geometry/Solid_operator.hpp"
%include "geometry/Solid_operand.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Solid_member;
%include "geometry/Solid_member.hpp"
// solid bases
%include "geometry/Ellipsoid_base.hpp"
%include "geometry/General_cone_base.hpp"
%include "geometry/Polyhedron_base.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Triangle_mesh;
%include "geometry/Triangle_mesh.hpp"
// composite solids
%feature("notabstract") Roan_trail::Tovero_math::Solid_combination_base;
%include "geometry/Solid_combination_base.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Solid_combination;
%include "geometry/Solid_combination.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Solid_operation;
%include "geometry/Solid_operation.hpp"
// primitive solids
%feature("notabstract") Roan_trail::Tovero_math::Axis_aligned_box;
%include "geometry/Axis_aligned_box.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Box;
%include "geometry/Box.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Boundary_triangle_mesh;
%include "geometry/Boundary_triangle_mesh.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Cone;
%include "geometry/Cone.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Cylinder;
%include "geometry/Cylinder.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Ellipsoid;
%include "geometry/Ellipsoid.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Elliptical_cone;
%include "geometry/Elliptical_cone.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Elliptical_cylinder;
%include "geometry/Elliptical_cylinder.hpp"
%feature("notabstract") Roan_trail::Tovero_math::General_cone;
%include "geometry/General_cone.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Half_space;
%include "geometry/Half_space.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Oblique_cone;
%include "geometry/Oblique_cone.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Oblique_cylinder;
%include "geometry/Oblique_cylinder.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Plate_triangle_mesh;
%include "geometry/Plate_triangle_mesh.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Polyhedron;
%include "geometry/Polyhedron.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Sphere;
%include "geometry/Sphere.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Torus;
%include "geometry/Torus.hpp"
%feature("notabstract") Roan_trail::Tovero_math::Wedge;
%include "geometry/Wedge.hpp"
