#!/usr/bin/ruby
#
# radio1.rb
#
# Copyright 2013-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Simple radio model for Tovero tutorial
#

# These load the extension libraries
require "libtovero_support_rb_1"
require "libtovero_math_rb_1"
require "libtovero_graphics_rb_1"

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

#
# Setup some constants
#

ZERO = Unitless.new(0.0)
CM = Distance::meter * 0.01
ZERO_CM = Distance::meter * 0.0
ORIGIN = Point.new(ZERO_CM,
                   ZERO_CM,
                   ZERO_CM)

#
# Create the radio
#

#
#   radio body
#
body = AxisAlignedBox.new(ORIGIN,
                          Point.new(CM * 16.0,
                                    CM * 32.0,
                                    CM * 48.0),
                          "body.s")

#
#   talk button
#
button = EllipticalCylinder.new(Point.new(CM * 8.0,
                                          CM * 33.0,
                                          CM * 36.0),
                                Vector.new(CM * 4.0,
                                           ZERO_CM,
                                           ZERO_CM),
                                Vector.new(ZERO_CM,
                                           ZERO_CM,
                                           CM * 2.0),
                                CM * 3.0,
                                "btn.s")
button2 = Ellipsoid.new(Point.new(CM * 8.0,
                                  CM * 33.0,
                                  CM * 36.0),
                        Vector.new(CM * 4.0,
                                   ZERO_CM,
                                   ZERO_CM),
                        CM * 2.0,
                        "btn2.s")

#
#   speaker
#
speaker = Torus.new(Point.new(CM * 16.0,
                              CM * 16.0,
                              CM * 16.0),
                    UnitVector.new(Unitless.new(1.0),
                                   ZERO,
                                   ZERO),
                    CM * 12.0,
                    CM * 1.0,
                    "spkr.s")

#
#   antenna
#
antenna = Cylinder.new(Point.new(CM * 2.0,
                                 CM * 2.0,
                                 CM * 46.0),
                       Vector.new(ZERO_CM,
                                  ZERO_CM,
                                  CM * 48.0),
                       CM * 1.0,
                       "ant.s")

#
#   volume knob
#
knob = Cylinder.new(Point.new(CM * 4.0,
                              CM * 4.0,
                              CM * 40.0),
                    Vector.new(CM * 8.0,
                               ZERO_CM,
                               ZERO_CM),
                    CM * 5.0,
                    "knob.s")

#
#   combine parts to make the radio
#
radio = SolidCombination.new("radio.c")
radio << body
radio << button
radio << button2
radio << speaker
radio << antenna
radio << knob

#
# Create a BRL-CAD ".g" database
#

database = BCDatabase.new

#
# Add the radio to the database
#

solid_list = database.top_solids;
solid_list << radio

#
# Write the database to a file
#

database_file = "radio1.g"
error = ErrorParam.new
success = database.write(database_file, true, error)

if (not success)
  puts("Could not write #{database_file}:")
  puts(error.base.to_s)
  abort()
end

#
# Render the database with BRL-CAD's tools
#

image_height = 512
image_width = 512
azimuth = -35
elevation = 25

#   display raytraced model
#   options to "rt":
#     -C [background color (R/G/B)]
#     -a azimuth
#     -e elevation
#     -o [output .pix file]
#     -w [width in pixels of rendering]
#     -n [height in pixels (number of lines) of rendering]
#     <first positional arg> [database]
#     <second positional arg> [entity in database to trace]

command = "rm -f #{database_file}.pix #{database_file}.png ; \
          rt -C255/255/255 -a #{azimuth} -e #{elevation} \
             -o #{database_file}.pix \
             -w#{image_width} -n#{image_height} #{database_file} radio.c"
%x[#{command}]

#   convert image and cleanup
command = "pix-png -w#{image_width} -n#{image_height} \
                   -o #{database_file}.png \
                   #{database_file}.pix ; \
          rm -f #{database_file}.pix"
%x[#{command}]
