#!/usr/bin/ruby
#
# radio3_cutaway.rb
#
# Copyright 2013-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Final radio model for Tovero tutorial (cutaway view)
#

# These load the extension libraries
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'
require 'libtovero_graphics_rb_1'

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

#
# Setup some constants
#

ZERO = Unitless.new(0.0)
CM = Distance::meter * 0.01
ZERO_CM = Distance::meter * 0.0
ORIGIN = Point.new(ZERO_CM,
                   ZERO_CM,
                   ZERO_CM)
Z_AXIS = UnitVector.new(ZERO,
                        ZERO,
                        Unitless.new(1.0))

#
# Start creating the radio
#

#
#   radio body and cavity
#

body = AxisAlignedBox.new(ORIGIN,
                          Point.new(CM * 16.0,
                                    CM * 32.0,
                                    CM * 48.0),
                          "body.s")
cavity_offset = Vector.new(CM * 1.0,
                           CM * 1.0,
                           CM * 1.0)
cavity = AxisAlignedBox.new(body.minimum + cavity_offset,
                            body.maximum - cavity_offset,
                            "cavity.s")
#
#   talk button
#

button_cylinder = EllipticalCylinder.new(Point.new(CM * 8.0,
                                                   CM * 33.0,
                                                   CM * 36),
                                         Vector.new(CM * 4.0,
                                                    ZERO_CM,
                                                    ZERO_CM),
                                         Vector.new(ZERO_CM,
                                                    ZERO_CM,
                                                    CM * 2.0),
                                         CM * 3.0,
                                         "btn.s")
button_ellipsoid = Ellipsoid.new(Point.new(CM * 8.0,
                                           CM * 33.0,
                                           CM * 36.0),
                                 Vector.new(CM * 4.0,
                                            ZERO_CM,
                                            ZERO_CM),
                                 CM * 2.0,
                                 "btn2.s")

button_color = Color.new(128, 128, 128, 0) # grey
button_attributes = CombinationAttributes.new
button_attributes.color = button_color
button_attributes.is_part = true

button = Combination.new(button_attributes, "button.r")
button << button_cylinder
button << button_ellipsoid

#
#   speaker
#

speaker = Torus.new(Point.new(CM * 16.0,
                              CM * 16.0,
                              CM * 16.0),
                    UnitVector.new(Unitless.new(1.0),
                                   ZERO,
                                   ZERO),
                    CM * 12.0,
                    CM * 1.0,
                    "spkr.s")

#
#   antenna
#

antenna_cylinder = Cylinder.new(Point.new(CM * 2.0,
                                          CM * 2.0,
                                          CM * 46.0),
                                Vector.new(ZERO_CM,
                                           ZERO_CM,
                                           CM * 48.0),
                                CM * 1.0,
                                "ant.s")
antenna_ellipsoid = Ellipsoid.new(Point.new(CM * 2.0,
                                            CM * 2.0,
                                            CM * 94.0),
                                  Vector.new(ZERO_CM,
                                             ZERO_CM,
                                             CM * 1.0),
                                  CM * 3.0,
                                  "ant2.s")

antenna_shader = PhongShader::mirror
antenna_color = Color.new(128, 128, 128, 0) # grey
antenna_attributes = CombinationAttributes.new
antenna_attributes.shader = antenna_shader
antenna_attributes.color = antenna_color
antenna_attributes.is_part = true

antenna = Combination.new(antenna_attributes, "antenna.r")
antenna << antenna_cylinder
antenna << antenna_ellipsoid

#
#   volume knob
#

knob_cylinder = Cylinder.new(Point.new(CM * 4.0,
                                       CM * 4.0,
                                       CM * 40.0),
                             Vector.new(CM * 8.0,
                                        ZERO_CM,
                                        ZERO_CM),
                             CM * 5.0,
                             "knob.s")

knob_shader = PhongShader::plastic
knob_shader.shine = 4
knob_color = Color.new(198, 198, 0, 0) # yellow
knob_attributes = CombinationAttributes.new
knob_attributes.shader = knob_shader
knob_attributes.color = knob_color
knob_attributes.is_part = true

knob = Combination.new(knob_attributes, "knob.r")
knob << knob_cylinder

#
# Radio case
#

case_color = Color.new(0, 0, 255, 0) # blue
case_attributes = CombinationAttributes.new
case_attributes.color = case_color
case_attributes.is_part = true

radio_case = Combination.new(case_attributes, "radio_case.r")
radio_case << body
radio_case << SolidMember.new(SolidOperand.new(cavity),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(antenna_cylinder),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(knob_cylinder),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(button_cylinder),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(speaker),
                              SolidOperator::Union_op)

#
#   circuit board
#

board_box = AxisAlignedBox.new(Point.new(CM * 3.0,
                                         CM * 1.0,
                                         CM * 1.0),
                               Point.new(CM * 4.0,
                                         CM * 31.0,
                                         CM * 47.0),
                               "board.s")

board_color = Color.new(0, 255, 0, 0) # green
board_attributes = CombinationAttributes.new
board_attributes.color = board_color
board_attributes.is_part = true

board = Combination.new(board_attributes, "board.r")
board << board_box

#
#   combine parts to make the radio
#

radio = Combination.new("radio.c")
radio << radio_case
radio << button
radio << knob
radio << antenna
radio << board

#
# Make a translucent view
#

#
#   make a copy of the attributes
#
radio_case_translucent_shader = PhongShader.new
radio_case_translucent_shader.transmitted = Unitless.new(0.6)
radio_case_translucent_attributes = CombinationAttributes.new(case_attributes)
radio_case_translucent_attributes.shader = radio_case_translucent_shader
#
#   make a copy of the case
#
radio_case_translucent = Combination.new(radio_case)
radio_case_translucent.name = "case_translucent.r"
radio_case_translucent.attributes = radio_case_translucent_attributes
#
#   make the translucent radio
#
radio_translucent = Combination.new("radio_translucent.c")
radio_translucent << radio_case_translucent
radio_translucent << button
radio_translucent << knob
radio_translucent << antenna
radio_translucent << board

#
# Make a cutaway view
#

cutaway_box = AxisAlignedBox.new(Point.new(CM * -16.0,
                                           CM * -16.0,
                                           CM * -1.0),
                                 Point.new(CM * 16.0,
                                           CM * 16.0,
                                           CM * 49.0),
                                 "cutaway_box.s")
cutaway_transform = Transformation.new
cutaway_transform.set_translation(CM * -8.0,
                                  CM * 36.0,
                                  ZERO_CM)
cutaway_transform.rotate(Z_AXIS, Angle::degree * -45.0)
cutaway = SolidCombination.new("cutaway.c")
cutaway << SolidMember.new(SolidOperand.new(cutaway_box, cutaway_transform))

radio_cutaway = Combination.new("radio_cutaway.c")
radio_cutaway << radio
radio_cutaway << SolidMember.new(cutaway, SolidOperator::Difference_op)

#
# Create a BRL-CAD ".g" database
#

database = BCDatabase.new

#
# Add the different radio views to the database
#

solid_list = database.top_solids;
solid_list << radio
solid_list << radio_translucent
solid_list << radio_cutaway

#
# Write the database to a file
#

database_file = "radio3.g"
error = ErrorParam.new
success = database.write(database_file, true, error)

if (not success)
  puts("Could not write #{database_file}:")
  puts(error.base.to_s)
  abort()
end

#
# Render the database with BRL-CAD's tools
#

image_height = 512
image_width = 512
azimuth = 35
elevation = 25

# display raytraced model
#   options to "rt":
#     -C [background color (R/G/B)]
#     -a azimuth
#     -e elevation
#     -o [output .pix file]
#     -w [width in pixels of rendering]
#     -n [height in pixels (number of lines) of rendering]
#     <first positional arg> [database]
#     <second positional arg> [entity in database to trace]
views = ["cutaway"]
views.each do |view|
  if view == ""
    image_file_name = "radio3"
    view_combination = "radio.c"
  else
    image_file_name = "radio3_#{view}"
    view_combination = "radio_#{view}.c"
  end
  command = "rm -f #{image_file_name}.pix #{image_file_name}.png ; \
             rt -C255/255/255 -a #{azimuth} -e #{elevation} -o #{image_file_name}.pix \
                -w#{image_width} -n#{image_height} #{database_file} #{view_combination}"
  %x[#{command}]

  # convert image and cleanup
  command = "pix-png -w#{image_width} -n#{image_height} \
                      -o #{image_file_name}.png #{image_file_name}.pix ; \
             rm -f #{image_file_name}.pix"
  %x[#{command}]
end
