#!/usr/bin/ruby
#
# radio_axes.rb
#
# Copyright 2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Radio axes model for the Tovero tutorial
#

# These load the extension libraries
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'
require 'libtovero_graphics_rb_1'

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

#
# Setup some constants
#
ZERO = Unitless.new(0.0)
CM = Distance::meter * 0.01
ZERO_CM = Distance::meter * 0.0
ORIGIN = Point.new(ZERO_CM,
                   ZERO_CM,
                   ZERO_CM)
Z_AXIS = UnitVector.new(ZERO,
                        ZERO,
                        Unitless.new(1.0))
LABEL_AXIS = UnitVector.new(Unitless.new(1.0),
                            ZERO,
                            ZERO)
LABEL_SCALE = Unitless.new(0.7)

#
# Create the shapes for the radio
#
body = AxisAlignedBox.new(ORIGIN,
                          Point.new(CM * 16.0,
                                    CM * 32.0,
                                    CM * 48.0),
                          "body.s")

#
# create axes
#

AXIS_LENGTH = CM * 60.0
AXIS_RADIUS = CM * 0.5
ARROW_HEIGHT = CM * 4.0
ARROW_RADIUS = CM * 1.5
LABEL_OFFSET = CM * 12.0

xaxis = EllipticalCylinder.new(ORIGIN,
                               Vector.new(ZERO_CM,
                                          AXIS_RADIUS,
                                          ZERO_CM),
                               Vector.new(ZERO_CM,
                                          ZERO_CM,
                                          AXIS_RADIUS),
                               AXIS_LENGTH,
                               "xax.s")
xarrow = Cone.new(Point.new(AXIS_LENGTH,
                            ZERO_CM,
                            ZERO_CM),
                  Vector.new(ARROW_HEIGHT,
                             ZERO_CM,
                             ZERO_CM),
                  ARROW_RADIUS,
                  ZERO_CM,
                  "xar.s")

yaxis = EllipticalCylinder.new(ORIGIN,
                               Vector.new(ZERO_CM,
                                          ZERO_CM,
                                          AXIS_RADIUS),
                               Vector.new(AXIS_RADIUS,
                                          ZERO_CM,
                                          ZERO_CM),
                               AXIS_LENGTH,
                               "yax.s")
yarrow = Cone.new(Point.new(ZERO_CM,
                            AXIS_LENGTH,
                            ZERO_CM),
                  Vector.new(ZERO_CM,
                             ARROW_HEIGHT,
                             ZERO_CM),
                  ARROW_RADIUS,
                  ZERO_CM,
                  "yar.s")

zaxis = EllipticalCylinder.new(ORIGIN,
                               Vector.new(AXIS_RADIUS,
                                          ZERO_CM,
                                          ZERO_CM),
                               Vector.new(ZERO_CM,
                                          AXIS_RADIUS,
                                          ZERO_CM),
                               AXIS_LENGTH,
                               "zax.s")
zarrow = Cone.new(Point.new(ZERO_CM,
                            ZERO_CM,
                            AXIS_LENGTH),
                  Vector.new(ZERO_CM,
                             ZERO_CM,
                             ARROW_HEIGHT),
                  ARROW_RADIUS,
                  ZERO_CM,
                  "zar.s")

axes_color = Color.new(80, 80, 80, 0)  # dark grey
axes_attributes = CombinationAttributes.new
axes_attributes.color = axes_color
axes_attributes.is_part = true

red_color = Color.new(150, 0, 0, 0)  # dark red
red_axis_attributes = CombinationAttributes.new
red_axis_attributes.color = red_color
red_axis_attributes.is_part = true
dkred_color = Color.new(50, 0, 0, 0)  # darker red
dkred_label_attributes = CombinationAttributes.new
dkred_label_attributes.color = dkred_color
dkred_label_attributes.is_part = true

green_color = Color.new(0, 150, 0, 0)  # dark green
green_axis_attributes = CombinationAttributes.new
green_axis_attributes.color = green_color
green_axis_attributes.is_part = true
dkgreen_color = Color.new(0, 50, 0, 0)  # darker green
dkgreen_label_attributes = CombinationAttributes.new
dkgreen_label_attributes.color = dkgreen_color
dkgreen_label_attributes.is_part = true

blue_color = Color.new(0, 0, 150, 0)  # dark blue
blue_axis_attributes = CombinationAttributes.new
blue_axis_attributes.color = blue_color
blue_axis_attributes.is_part = true
dkblue_color = Color.new(0, 0, 50, 0)  # darker blue
dkblue_label_attributes = CombinationAttributes.new
dkblue_label_attributes.color = dkblue_color
dkblue_label_attributes.is_part = true

#
# create axis labels
#

x_box1 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 10.0,
                                      CM * 10.0),
                            "x1.s")
x_box2 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 10.0,
                                      CM * 10.0),
                            "x2.s")

x1_transform = Transformation.new
x1_transform.set_translation(CM * 0.0,
                             CM * 0.0,
                             ZERO_CM)
x1_transform.rotate(Z_AXIS, Angle::degree * -30.0)
x2_transform = Transformation.new
x2_transform.set_translation(CM * 4.0,
                             CM * -3.0,
                             ZERO_CM)
x2_transform.rotate(Z_AXIS, Angle::degree * 30.0)

x_letter = SolidCombination.new("xletter.c")
x_letter << SolidMember.new(SolidOperand.new(x_box1, x1_transform))
x_letter << SolidMember.new(SolidOperand.new(x_box2, x2_transform))

y_box1 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 5.0,
                                      CM * 10.0),
                            "y1.s")
y_box2 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 5.0,
                                      CM * 10.0),
                            "y2.s")
y_box3 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 5.0,
                                      CM * 10.0),
                            "y3.s")

y1_transform = Transformation.new
y1_transform.set_translation(CM * 0.0,
                             CM * 0.0,
                             ZERO_CM)
y1_transform.rotate(Z_AXIS, Angle::degree * -30.0)
y2_transform = Transformation.new
y2_transform.set_translation(CM * 4.0,
                             CM * -3.0,
                             ZERO_CM)
y2_transform.rotate(Z_AXIS, Angle::degree * 30.0)
y3_transform = Transformation.new
y3_transform.set_translation(CM * 2.3,
                             CM * 3.8,
                             ZERO_CM)
y3_transform.rotate(Z_AXIS, Angle::degree * 0.0)

y_letter = SolidCombination.new("yletter.c")
y_letter << SolidMember.new(SolidOperand.new(y_box1, y1_transform))
y_letter << SolidMember.new(SolidOperand.new(y_box2, y2_transform))
y_letter << SolidMember.new(SolidOperand.new(y_box3, y3_transform))

z_box1 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 1.0,
                                      CM * 10.0,
                                      CM * 10.0),
                            "z1.s")
z_box2 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 5.0,
                                      CM * 1.0,
                                      CM * 10.0),
                            "z2.s")
z_box3 = AxisAlignedBox.new(Point.new(CM * 0.0,
                                      CM * 0.0,
                                      CM * 0.0),
                            Point.new(CM * 5.0,
                                      CM * 1.0,
                                      CM * 10.0),
                            "z3.s")

z1_transform = Transformation.new
z1_transform.set_translation(CM * 0.0,
                             CM * 0.0,
                             ZERO_CM)
z1_transform.rotate(Z_AXIS, Angle::degree * -30.0)
z2_transform = Transformation.new
z2_transform.set_translation(CM * 0.0,
                             CM * -0.5,
                             ZERO_CM)
z2_transform.rotate(Z_AXIS, Angle::degree * 0.0)
z3_transform = Transformation.new
z3_transform.set_translation(CM * 0.0,
                             CM * 8.0,
                             ZERO_CM)
z3_transform.rotate(Z_AXIS, Angle::degree * 0.0)

z_letter = SolidCombination.new("yletter.c")
z_letter << SolidMember.new(SolidOperand.new(z_box1, z1_transform))
z_letter << SolidMember.new(SolidOperand.new(z_box2, z2_transform))
z_letter << SolidMember.new(SolidOperand.new(z_box3, z3_transform))

letter_rotate = Transformation.new
letter_rotate.set_translation(ZERO_CM,
                              ZERO_CM,
                              ZERO_CM)
letter_rotate.rotate(LABEL_AXIS, Angle::degree * -65.0)
letter_rotate.rotate(Z_AXIS, Angle::degree * -128.0)       # -125 here gives completely flat, no depth letters
letter_rotate.scale(LABEL_SCALE, LABEL_SCALE, LABEL_SCALE)
x_rletter = SolidCombination.new("xrletter.c")
x_rletter << SolidMember.new(SolidOperand.new(x_letter, letter_rotate))
y_rletter = SolidCombination.new("yrletter.c")
y_rletter << SolidMember.new(SolidOperand.new(y_letter, letter_rotate))
z_rletter = SolidCombination.new("xrletter.c")
z_rletter << SolidMember.new(SolidOperand.new(z_letter, letter_rotate))

# place labels
x_place = Transformation.new
x_place.set_translation(AXIS_LENGTH + LABEL_OFFSET,
                        ZERO_CM,
                        ZERO_CM)
x_place.rotate(Z_AXIS, Angle::degree * 0.0)
x_label = SolidCombination.new("xlabel.c")
x_label << SolidMember.new(SolidOperand.new(x_rletter, x_place))

y_place = Transformation.new
y_place.set_translation(ZERO_CM,
                        AXIS_LENGTH + LABEL_OFFSET,
                        ZERO_CM)
y_place.rotate(Z_AXIS, Angle::degree * 0.0)
y_label = SolidCombination.new("ylabel.c")
y_label << SolidMember.new(SolidOperand.new(y_rletter, y_place))

z_place = Transformation.new
z_place.set_translation(ZERO_CM,
                        ZERO_CM,
                        AXIS_LENGTH + LABEL_OFFSET)
z_place.rotate(Z_AXIS, Angle::degree * 0.0)
z_label = SolidCombination.new("zlabel.c")
z_label << SolidMember.new(SolidOperand.new(z_rletter, z_place))

red_axis = Combination.new(red_axis_attributes, "redaxis.r")
red_axis << xaxis
red_axis << xarrow
red_label = Combination.new(dkred_label_attributes, "redlabel.r")
red_label << x_label

green_axis = Combination.new(green_axis_attributes, "greenaxis.r")
green_axis << yaxis
green_axis << yarrow
green_axis << SolidMember.new(SolidOperand.new(xaxis),
                              SolidOperator::Difference_op)
green_axis << SolidMember.new(SolidOperand.new(zaxis),
                              SolidOperator::Difference_op)
green_label = Combination.new(dkgreen_label_attributes, "greenlabel.r")
green_label << y_label

blue_axis = Combination.new(blue_axis_attributes, "blueaxis.r")
blue_axis << zaxis
blue_axis << zarrow
blue_axis << SolidMember.new(SolidOperand.new(xaxis),
                             SolidOperator::Difference_op)
blue_label = Combination.new(dkblue_label_attributes, "bluelabel.r")
blue_label << z_label

#
# Radio case
#
case_color = Color.new(230, 230, 230, 0) # light grey
case_attributes = CombinationAttributes.new
case_attributes.color = case_color
case_attributes.is_part = true

radio_case = Combination.new(case_attributes, "radio_case.r")
radio_case << body
radio_case << SolidMember.new(SolidOperand.new(xaxis),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(yaxis),
                              SolidOperator::Difference_op)
radio_case << SolidMember.new(SolidOperand.new(zaxis),
                              SolidOperator::Difference_op)

#
# make a copy of the attributes
#
radio_case_translucent_shader = PhongShader.new
radio_case_translucent_shader.transmitted = Unitless.new(0.3)
radio_case_translucent_attributes = CombinationAttributes.new(case_attributes)
radio_case_translucent_attributes.shader = radio_case_translucent_shader
radio_case_translucent = Combination.new(radio_case)
radio_case_translucent.name = "case_translucent.r"
radio_case_translucent.attributes = radio_case_translucent_attributes

#
# Combine parts to make the radio
#
radio = Combination.new("radio.c")
radio << radio_case_translucent
radio << red_axis
radio << red_label
radio << green_axis
radio << green_label
radio << blue_axis
radio << blue_label

#
# Create a BRL-CAD ".g" database
#
database = BCDatabase.new

#
# Add the shapes to the database
#
solid_list = database.top_solids();
solid_list << radio

#
# Write the database to a file
#
database_file = "radio_axes.g"
error = ErrorParam.new
success = database.write(database_file, true, error)
if (not success)
  puts("Could not write #{database_file}:")
  puts(error.base.to_s)
  abort()
end

#
# Render the database with BRL-CAD's tools
#

image_height = 512
image_width = 512
azimuth = -35
elevation = 25

# display raytraced model
#   options to "rt":
#     -a azimuth
#     -e elevation
#     -w [width in pixels of rendering]
#     -n [height in pixels (number of lines) of rendering]
#     <first positional arg> [database]
#     <second positional arg> [entity in database to trace]

image_file = "radio_axes"

command = "rm -f #{image_file}.pix #{image_file}.png ; \
           rt -C255/255/255 -a #{azimuth} -e #{elevation} -o #{image_file}.pix \
             -w#{image_width} -n#{image_height} #{database_file} radio.c\n"
%x[#{command}]

# convert image and cleanup
command = "pix-png -w#{image_width} -n#{image_height} -o #{image_file}.png \
                   #{image_file}.pix ; \
            rm -f #{image_file}.pix"
%x[#{command}]

