// BC_database_reader.hpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// BRL-CAD database read functions

#ifndef TOVERO_GRAPHICS_BC_DATABASE_READER_HPP_
#define TOVERO_GRAPHICS_BC_DATABASE_READER_HPP_

#include <tovero/graphics/base/Database.hpp>
#include <string>
#include <vector>

struct rt_wdb;
typedef void* genptr_t;

namespace Roan_trail
{
  namespace Tovero_support
  {
    template <class Value_type> class Reference_counting_list;
    class Error;
    class Error_param;
    class Logger;
  }

  namespace Tovero_math
  {
    class Distance;
    class Geometric_tolerances;
    class Solid;
    class Units_mapper;
  }

  typedef Roan_trail::Tovero_support::Reference_counting_list<Tovero_math::Solid> Solid_list;

  namespace Tovero_graphics
  {
    template <class Node_type> class Node_mapper;

    class BC_database_reader
    {
    public:
      // constructor/destructor/initialization
      BC_database_reader(bool simplify_solids,
                         bool tolerate_database_errors,
                         const Roan_trail::Tovero_math::Geometric_tolerances& tolerances);
      ~BC_database_reader();
      // accessors/mutators
      const std::string& title() const { return m_title; }
      // read a vector of objects from the BRL-CAD
      // database, return a vector of solids
      bool read(const std::string& path,
                const std::vector<std::string>& objects,
                Roan_trail::Solid_list& return_solids,
                Roan_trail::Tovero_math::Units_mapper& return_units_mapper,
                Roan_trail::Tovero_support::Error_param& return_error);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      std::string m_title;
      struct Conversion_struct
      {
        void* solid;
        int solid_ID;
        const std::string& solid_name;
      };
      //
      struct File_struct
      {
        File_struct();
        //
        std::string path;
        struct db_i *DB_instance;
      } m_file;
      std::vector<std::string> m_objects;
      Node_mapper<Roan_trail::Tovero_math::Solid*>* m_node_mapper;
      bool m_simplify_solids;
      bool m_tolerate_database_errors;
      const Roan_trail::Tovero_math::Geometric_tolerances& m_tolerances;
      Roan_trail::Tovero_support::Error* m_IO_error;
      Roan_trail::Tovero_support::Logger& m_logger;
      Roan_trail::Tovero_math::Distance* m_distance_conversion_factor;
      //
      bool mf_open_for_read(const Roan_trail::Tovero_math::Units_mapper& units_mapper,
                            std::string& return_length_units,
                            Roan_trail::Tovero_support::Error_param& return_error);
      bool mf_read_database(Roan_trail::Solid_list& return_solids,
                            Roan_trail::Tovero_support::Error_param& return_error);
      // setup/cleanup
      void mf_setup_for_read(const std::string& path,
                             const std::vector<std::string>& objects,
                             const Roan_trail::Tovero_math::Units_mapper& units_mapper);
      void mf_cleanup_after_read();
      void mf_cleanup_nodes();
      // tree walk callbacks
      static void mf_do_combination(struct db_i *database,
                                    struct directory *dir,
                                    genptr_t client_data);
      static void mf_do_leaf(struct db_i *database,
                             struct directory *dir,
                             genptr_t client_data);
      // BRL-CAD conversion functions
      //   handles Triangle_mesh
      bool mf_make_BOT(const Conversion_struct& conversion,
                       Roan_trail::Tovero_support::Error_param& error);
      //   handles Half_space
      bool mf_make_half_space(const Conversion_struct& conversion,
                              Roan_trail::Tovero_support::Error_param& error);
      //   handles Ellipsoid, Sphere
      bool mf_make_ellipsoid(const Conversion_struct& conversion, Roan_trail::Tovero_support::Error_param& error);
      //   handles Polyhedron, Wedge, Box, Axis_aligned_box
      bool mf_make_polyhedron(const Conversion_struct& conversion,
                              Roan_trail::Tovero_support::Error_param& return_error);
      bool mf_make_torus(const Conversion_struct& conversion,
                         Roan_trail::Tovero_support::Error_param& error);
      //   handles General_cone, Cone, Cylinder, Elliptical_cone, Elliptical_cylinder,
      //   Oblique_cone, Oblique_cylinder
      bool mf_make_general_cone(const Conversion_struct& conversion,
                                Roan_trail::Tovero_support::Error_param& error);
      //   handles Pipe
      bool mf_make_pipe(const Conversion_struct& conversion,
                        Roan_trail::Tovero_support::Error_param& error);
      //
      void mf_add_solid(const std::string& node_name, Roan_trail::Tovero_math::Solid* solid);
      // prevent compiler from generating
      BC_database_reader(const BC_database_reader& database);
      BC_database_reader& operator=(const BC_database_reader& database);
    };
  }
}

#endif // TOVERO_GRAPHICS_BC_DATABASE_READER_HPP_
