// BC_database_writer.hpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// BRL-CAD database write functions

#ifndef TOVERO_GRAPHICS_BC_DATABASE_WRITER_HPP_
#define TOVERO_GRAPHICS_BC_DATABASE_WRITER_HPP_

#include <tovero/math/geometry/Solid_visitor.hpp>
#include <tovero/graphics/base/Database.hpp>

#include <string>

struct rt_wdb;

namespace Roan_trail
{
  namespace Tovero_support
  {
    template <class Value_type> class Reference_counting_list;
    class Error;
    class Error_param;
    class Logger;
  }

  namespace Tovero_math
  {
    class Distance;
    class Geometric_tolerances;
    //
    class Solid_combination;
    class Solid_operation;
    //
    class Axis_aligned_box;
    class Box;
    class Boundary_triangle_mesh;
    class Cone;
    class Cylinder;
    class Ellipsoid;
    class Ellipsoid_base;
    class Elliptical_cone;
    class Elliptical_cylinder;
    class General_cone;
    class General_cone_base;
    class Half_space;
    class Oblique_cone;
    class Oblique_cylinder;
    class Pipe;
    class Plate_triangle_mesh;
    class Polyhedron;
    class Polyhedron_base;
    class Solid;
    class Sphere;
    class Torus;
    class Triangle_mesh;
    class Units_mapper;
    class Wedge;
  }

  typedef Roan_trail::Tovero_support::Reference_counting_list<Tovero_math::Solid> Solid_list;

  namespace Tovero_graphics
  {
    template <class Node_type> class Node_mapper;

    typedef Roan_trail::Tovero_math::Solid_visitor::Visit_result Visit_result_type;

    class BC_database_writer : public Roan_trail::Tovero_math::Solid_visitor
    {
    public:
      // constructor/destructor/initialization
      BC_database_writer(const std::string& title,
                         bool tolerate_database_errors,
                         const Roan_trail::Tovero_math::Geometric_tolerances& tolerances,
                         const Roan_trail::Solid_list& solids);
      virtual ~BC_database_writer();
      //
      bool write(const std::string& path,
                 bool overwrite,
                 const Roan_trail::Tovero_math::Units_mapper& units_mapper,
                 Roan_trail::Tovero_support::Error_param& return_error);
    protected:
      // invariant check
      bool mf_invariant(bool check_base_class = true) const;
    private:
      //
      std::string m_title;
      bool m_tolerate_database_errors;
      const Roan_trail::Tovero_math::Geometric_tolerances& m_tolerances;
      const Roan_trail::Solid_list* m_solids;
      struct File_struct
      {
        File_struct();
        //
        std::string path;
        bool overwrite;
        struct rt_wdb* database;
      } m_file;
      //
      Node_mapper<const Roan_trail::Tovero_math::Solid*>* m_node_mapper;
      Roan_trail::Tovero_support::Error* m_IO_error;
      Roan_trail::Tovero_support::Logger& m_logger;
      Roan_trail::Tovero_math::Distance* m_distance_conversion_factor;
      //
      bool mf_open_for_write(const Roan_trail::Tovero_math::Units_mapper& units_mapper,
                             Roan_trail::Tovero_support::Error_param& return_error);
      // setup/cleanup
      void mf_setup_for_write(const std::string& path,
                              bool overwrite,
                              const Roan_trail::Tovero_math::Units_mapper& units_mapper);
      void mf_cleanup_after_write();
      // visitor
      //   composites
      virtual Visit_result_type visit_exit(const Roan_trail::Tovero_math::Solid_combination& combination);
      virtual Visit_result_type visit_exit(const Roan_trail::Tovero_math::Solid_operation& operation);
      //   primitives
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Axis_aligned_box& box);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Boundary_triangle_mesh& mesh);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Box& box);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Cone& cone);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Cylinder& cylinder);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Ellipsoid& ellipsoid);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Elliptical_cone& cone);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Elliptical_cylinder& cylinder);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::General_cone& cone);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Half_space& space);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Oblique_cone& cone);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Oblique_cylinder& cylinder);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Pipe& pipe);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Plate_triangle_mesh& mesh);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Polyhedron& polyhedron);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Sphere& sphere);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Torus& torus);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Triangle_mesh& mesh);
      virtual Visit_result_type visit(const Roan_trail::Tovero_math::Wedge& wedge);
      //   base
      Visit_result_type mf_visit_base(const Roan_trail::Tovero_math::Ellipsoid_base& ellipsoid);
      Visit_result_type mf_visit_base(const Roan_trail::Tovero_math::General_cone_base& cone);
      Visit_result_type mf_visit_base(const Roan_trail::Tovero_math::Polyhedron_base& polyhedron);
      Visit_result_type mf_visit_base(const Roan_trail::Tovero_math::Triangle_mesh& mesh,
                                      bool is_solid,
                                      bool has_thicknesses,
                                      const std::vector<Roan_trail::Tovero_math::Distance>& thicknesses,
                                      const std::vector<bool>& append_thicknesses);
      //
      bool mf_add_and_validate_solid(const Roan_trail::Tovero_math::Solid& solid,
                                     std::string& return_solid_name,
                                     Roan_trail::Tovero_support::Error_param& return_error);
      // prevent compiler from generating
      BC_database_writer(const BC_database_writer& database);
      BC_database_writer& operator=(const BC_database_writer& database);
    };
  }
}

#endif // TOVERO_GRAPHICS_BC_DATABASE_WRITER_HPP_
