// BC_shader.cpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.
//
// Based on the following files from BRL-CAD (version 7.20.4) source:
//
//   ./src/liboptical/sh_cook.c
//     Copyright (c) 1985-2011 United States Government as represented by
//     the U.S. Army Research Laboratory.
//
//   ./src/liboptical/sh_light.c
//     Copyright (c) 1998-2011 United States Government as represented by
//     the U.S. Army Research Laboratory.
//
//   ./src/liboptical/sh_plastic.c
//     Copyright (c) 1998-2011 United States Government as represented by
//     the U.S. Army Research Laboratory.
//
// under the following the license:
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

// TODO: add is_valid function to shaders and do validity checking

// include these first to avoid conflicts
#include <bu.h>
#include <light.h>
#include <plastic.h>
//
#include <tovero/graphics/brlcad/BC_shader.hpp>
#include <tovero/graphics/base/Shader.hpp>
#include <tovero/graphics/base/Cook_shader.hpp>
#include <tovero/graphics/base/Light_shader.hpp>
#include <tovero/graphics/base/Phong_shader.hpp>
#include <tovero/graphics/base/Toon_shader.hpp>
#include <tovero/math/base/math.hpp>
#include <tovero/math/geometry/Angle.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <map>
#include <sstream>
#include <string>

using std::map;
using std::string;
using std::stringstream;
using Roan_trail::Tovero_math::exact_equal_fp;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Angle;
using namespace Roan_trail::Tovero_graphics;

namespace
{
  //
  // shader specific structures (when not available in BRL-CAD headers)
  //

  //   from BRL-CAD ./src/liboptical/sh_cook.c
  struct Ih_cook_specific
  {
    double m; // rms slope
    int shine;
    double wgt_specular;
    double wgt_diffuse;
    double transmit;
    double reflect;
    double refrac_index;
    double extinction;
  };

  //
  // default parameters
  //

  const struct Ih_cook_specific ic_default_cook_parameters =
    {
      0.2, // slope
      10,  // shine
      0.7, // wgt_specular
      0.3, // wgt_diffuse
      0.0, // transmit
      0.0, // reflect
      1.0, // refrac_index (air)
      0.0  // extinction
    };

  const struct light_specific ic_default_light_parameters =
    {
      { LIGHT_MAGIC, 0, 0 }, // doublely linked list
      { 0.0, 0.0, 0.0 },     // target
      1.0,                   // intensity
      180.0,                 // angle
      1.0,                   // fraction
      1,                     // shadows
      0,                     // infinite
      1,                     // visible
      0,                     // invisible
      0,                     // explicit aim
      0.0                    // light obscuration
    };

  const struct phong_specific ic_default_phong_parameters =
    {
      PL_MAGIC,        // magic
      10,              // shine
      0.7,             // wgt_specular
      0.3,             // wgt_diffuse
      0.0,             // transmit
      0.0,             // reflect
      1.0,             // refrac_index (air)
      0.0,             // extinction
      {0.0, 0.0, 0.0}, // emission
      0                // mpf
    };

  //
  //  parsing structures (when not available in BRL-CAD headers)
  //

  //   from BRL-CAD ./src/liboptical/sh_light.c
  void ih_light_convert_visible(const struct bu_structparse *sdp,
                                const char *name,
                                char *base,
                                char *value)
  {
    // the reinterpret cast is OK because bu_struct_parse supplies the struct as char *
    struct light_specific *lsp = reinterpret_cast<struct light_specific *>(base);

    switch (sdp->sp_offset)
    {
    case bu_offsetof(struct light_specific, lt_invisible): // check_code_ignore
      lsp->lt_visible = !lsp->lt_invisible;
    break;
    case bu_offsetof(struct light_specific, lt_visible): // check_code_ignore
      lsp->lt_invisible = !lsp->lt_visible;
      break;
    default:
      break;
    }
  }

  //   from BRL-CAD ./src/liboptical/sh_light.c
  // TODO: find a safer way to register the callback without reinterpret casts
  const struct bu_structparse ic_light_parse[] = {
    {"%f", 1, "bright", bu_offsetof(struct light_specific, lt_intensity), 0, 0, 0 },  // check_code_ignore
    {"%f", 1, "b", bu_offsetof(struct light_specific, lt_intensity), 0, 0, 0 },  // check_code_ignore
    {"%f", 1, "inten", bu_offsetof(struct light_specific, lt_intensity), 0, 0, 0 },  // check_code_ignore
    //
    {"%f", 1, "angle", bu_offsetof(struct light_specific, lt_angle), 0, 0, 0 },  // check_code_ignore
    {"%f", 1, "a", bu_offsetof(struct light_specific, lt_angle), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "fract", bu_offsetof(struct light_specific, lt_fraction), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "f", bu_offsetof(struct light_specific, lt_fraction), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 3, "target", bu_offsetof(light_specific, lt_target), 0, 0, 0 }, // check_code_ignore
    {"%f", 3, "t", bu_offsetof(light_specific, lt_target), 0, 0, 0 }, // check_code_ignore
    // TODO: this is not in sh_light.c, but mged only accepts it (and "d")
    {"%f", 3, "direct", bu_offsetof(light_specific, lt_target), 0, 0, 0 }, // check_code_ignore
    {"%f", 3, "aim", bu_offsetof(light_specific, lt_target), 0, 0, 0 }, // check_code_ignore

    //
    {"%d", 1, "shadows", bu_offsetof(struct light_specific, lt_shadows), 0, 0, 0 }, // check_code_ignore
    {"%d", 1, "s", bu_offsetof(struct light_specific, lt_shadows), 0, 0, 0 }, // check_code_ignore
    //
    {"%d", 1, "infinite", bu_offsetof(struct light_specific, lt_infinite), 0, 0, 0 }, // check_code_ignore
    {"%d", 1, "i", bu_offsetof(struct light_specific, lt_infinite), 0, 0, 0 }, // check_code_ignore
    //
    {"%d", 1, "visible", bu_offsetof(struct light_specific, lt_visible), // check_code_ignore
     reinterpret_cast<void (*)()>(ih_light_convert_visible), 0, 0 },
    {"%d", 1, "v", bu_offsetof(struct light_specific, lt_visible), // check_code_ignore
     reinterpret_cast<void (*)()>(ih_light_convert_visible), 0, 0 },
    //
    {"%d", 1, "invisible", bu_offsetof(struct light_specific, lt_invisible), // check_code_ignore
     reinterpret_cast<void (*)()>(ih_light_convert_visible), 0, 0 },
    //
    {"", 0, 0, 0, 0, 0, 0 }
  };

  //   from BRL-CAD ./src/liboptical/sh_cook.c
  struct bu_structparse cook_parse[] = {
    {"%f", 1, "m", bu_offsetof(struct Ih_cook_specific, m), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "specular", bu_offsetof(struct Ih_cook_specific, wgt_specular), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "sp", bu_offsetof(struct Ih_cook_specific, wgt_specular), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "diffuse", bu_offsetof(struct Ih_cook_specific, wgt_diffuse), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "di", bu_offsetof(struct Ih_cook_specific, wgt_diffuse), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "transmit", bu_offsetof(struct Ih_cook_specific, transmit), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "tr", bu_offsetof(struct Ih_cook_specific, transmit), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "reflect", bu_offsetof(struct Ih_cook_specific, reflect), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "re", bu_offsetof(struct Ih_cook_specific, reflect), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "ri", bu_offsetof(struct Ih_cook_specific, refrac_index), 0, 0, 0 }, // check_code_ignore
    //
    {"%f", 1, "extinction", bu_offsetof(struct Ih_cook_specific, extinction), 0, 0, 0 }, // check_code_ignore
    {"%f", 1, "ex", bu_offsetof(struct Ih_cook_specific, extinction), 0, 0, 0 }, // check_code_ignore
    {"",   0, 0, 0, 0, 0, 0 }
  };

  //
  // shader map
  //

  enum Ih_shader_type
    {
      ic_shader_light = 0,
      ic_shader_phong = 1,
      ic_shader_toon =  2
    };

  typedef map<string, Ih_shader_type> Ih_shader_map_type;

  Ih_shader_map_type iv_shader_map;

  void ih_setup_shader_map()
  {
    iv_shader_map["light"] = ic_shader_light;
    iv_shader_map["phong"] = ic_shader_phong;
    iv_shader_map["toon"]  = ic_shader_toon;
  }
}

//
// Constructors
//

BC_shader::BC_shader(Shader& shader)
  : m_shader_name(),
    m_shader_args(),
    m_shader(&shader)
{
  // shader setup (using visitor pattern)
  shader.accept(*this); // TODO: check return value?

  m_shader->reference();
}

BC_shader::BC_shader(const string& shader)
  : m_shader_name(),
    m_shader_args(),
    m_shader(0)
{
  // setup shader map
  if (!iv_shader_map.size())
  {
    ih_setup_shader_map();
  }
  assert(iv_shader_map.size() && "invalid shader map");

  struct bu_vls shader_string;
  bu_vls_init(&shader_string);
  if (!bu_shader_to_key_eq(shader.c_str(), &shader_string)
      && shader_string.vls_str
      && ("" != string(shader_string.vls_str)))
  {
    stringstream shader_stream(string(shader_string.vls_str));
    string shader_name;
    shader_stream >> shader_name;
    stringstream shader_args_stream;
    char c;
    while (shader_stream.get(c))
    {
      shader_args_stream << c;
    }

    Ih_shader_map_type::const_iterator i = iv_shader_map.find(shader_name);
    if (i != iv_shader_map.end())
    {
      switch (i->second)
      {
      case ic_shader_phong:
        mf_parse_phong_shader(shader_args_stream.str());
        break;
      case ic_shader_toon:
        mf_parse_toon_shader(shader_args_stream.str());
        break;
      default:
        assert("invalid shader type");
      }
    }

    if (m_shader)
    {
      m_shader->reference();
    }
  }

  bu_vls_free(&shader_string);
}

BC_shader::~BC_shader()
{
  if (m_shader)
  {
    m_shader->unreference();
  }
}

//
// Private member functions
//

void BC_shader::mf_parse_cook_shader(const string& shader_args)
{
  precondition(!m_shader);

  struct Ih_cook_specific* BC_cook_parameters;
  BC_cook_parameters = static_cast<struct Ih_cook_specific*>(bu_calloc(1,
                                                                      sizeof(struct Ih_cook_specific),
                                                                      "Ih_cook_specific " BU_FLSTR));
  *BC_cook_parameters = ic_default_cook_parameters;
  struct bu_vls parameter_string;
  bu_vls_init(&parameter_string);
  bu_vls_strcpy(&parameter_string, shader_args.c_str());

  // the reinterpret cast is OK because bu_struct_parse expects a struct as char *
  if (bu_struct_parse(&parameter_string,
                      cook_parse,
                      reinterpret_cast<char *>(BC_cook_parameters)) < 0)
  {
    goto exit_point;
  }
  else
  {
    Cook_shader* shader = new Cook_shader;

    // fill in the shader
    shader->set_slope(Unitless(BC_cook_parameters->m));
    shader->set_shine(BC_cook_parameters->shine);
    shader->set_specular(Unitless(BC_cook_parameters->wgt_specular));
    shader->set_diffuse(Unitless(BC_cook_parameters->wgt_diffuse));
    shader->set_transmitted(Unitless(BC_cook_parameters->transmit));
    shader->set_reflected(Unitless(BC_cook_parameters->reflect));
    shader->set_refractive_index(Unitless(BC_cook_parameters->refrac_index));
    shader->set_extinction(Unitless(BC_cook_parameters->extinction));

    m_shader = shader;
  }
 exit_point:
  bu_vls_free(&parameter_string);
  bu_free((genptr_t)BC_cook_parameters, "Ih_cook_specific");
}

void BC_shader::mf_parse_light_shader(const string& shader_args)
{
  precondition(!m_shader);

  struct light_specific* BC_light_parameters;
  BC_light_parameters = static_cast<struct light_specific*>(bu_calloc(1,
                                                                      sizeof(struct light_specific),
                                                                      "light_specific " BU_FLSTR));
  *BC_light_parameters = ic_default_light_parameters;
  struct bu_vls parameter_string;
  bu_vls_init(&parameter_string);
  bu_vls_strcpy(&parameter_string, shader_args.c_str());

  // the reinterpret cast is OK because bu_struct_parse expects a struct as char *
  if (bu_struct_parse(&parameter_string,
                      ic_light_parse,
                      reinterpret_cast<char *>(BC_light_parameters)) < 0)
  {
    goto exit_point;
  }
  else
  {
    Light_shader* shader = new Light_shader;

    // fill in the shader
    shader->set_target_direction(Unit_vector(Unitless(BC_light_parameters->lt_target[0]),
                                             Unitless(BC_light_parameters->lt_target[1]),
                                             Unitless(BC_light_parameters->lt_target[2])));
    shader->set_intensity(Unitless(BC_light_parameters->lt_intensity));
    shader->set_beam_dispersion_angle(BC_light_parameters->lt_angle * Angle::degree);
    shader->set_fraction(Unitless(BC_light_parameters->lt_fraction));
    shader->set_shadow_ray_count(BC_light_parameters->lt_shadows);
    shader->set_is_infinite(BC_light_parameters->lt_infinite);
    shader->set_is_visible(BC_light_parameters->lt_visible);

    m_shader = shader;
  }
 exit_point:
  bu_vls_free(&parameter_string);
  bu_free((genptr_t)BC_light_parameters, "light_specific");
}

void BC_shader::mf_parse_phong_shader(const string& shader_args)
{
  precondition(!m_shader);

  struct phong_specific* BC_phong_parameters;
  BC_phong_parameters = static_cast<struct phong_specific*>(bu_calloc(1,
                                                                      sizeof(struct phong_specific),
                                                                      "phong_specific " BU_FLSTR));
  *BC_phong_parameters = ic_default_phong_parameters;
  struct bu_vls parameter_string;
  bu_vls_init(&parameter_string);
  bu_vls_strcpy(&parameter_string, shader_args.c_str());

  // the reinterpret cast is OK because bu_struct_parse expects a struct as char *
  if (bu_struct_parse(&parameter_string,
                      phong_parse,
                      reinterpret_cast<char *>(BC_phong_parameters)) < 0)
  {
    goto exit_point;
  }
  else
  {
    Phong_shader* shader = new Phong_shader;

    // fill in the shader
    shader->set_shine(BC_phong_parameters->shine);
    shader->set_specular(Unitless(BC_phong_parameters->wgt_specular));
    shader->set_diffuse(Unitless(BC_phong_parameters->wgt_diffuse));
    shader->set_transmitted(Unitless(BC_phong_parameters->transmit));
    shader->set_reflected(Unitless(BC_phong_parameters->reflect));
    shader->set_refractive_index(Unitless(BC_phong_parameters->refrac_index));
    shader->set_extinction(Unitless(BC_phong_parameters->extinction));
    shader->set_emission_red(Unitless(BC_phong_parameters->emission[0]));
    shader->set_emission_green(Unitless(BC_phong_parameters->emission[1]));
    shader->set_emission_blue(Unitless(BC_phong_parameters->emission[2]));

    m_shader = shader;
  }
 exit_point:
  bu_vls_free(&parameter_string);
  bu_free((genptr_t)BC_phong_parameters, "phong_specific");
}

void BC_shader::mf_parse_toon_shader(const string& shader_args)
{
  precondition(!m_shader);

  m_shader = new Toon_shader;
}

//
//   Visitor
//

Shader_visitor::Visit_result BC_shader::visit(const Cook_shader& shader)
{
  m_shader_name = "cook";

  stringstream parameters;

  // if any parameter differs from the default, specify it

  if (!exact_equal_fp(ic_default_cook_parameters.m, shader.slope().value()))
  {
    parameters << " m " << shader.slope().value();
  }

  // TODO: what about shine, doesn't seem to be needed for Cook-Torrence shader

  if (!exact_equal_fp(ic_default_cook_parameters.wgt_specular, shader.specular().value()))
  {
    parameters << " specular " << shader.specular().value();
  }

  if (!exact_equal_fp(ic_default_cook_parameters.wgt_diffuse, shader.diffuse().value()))
  {
    parameters << " diffuse " << shader.diffuse().value();
  }

  if (!exact_equal_fp(ic_default_cook_parameters.transmit, shader.transmitted().value()))
  {
    parameters << " transmit " << shader.transmitted().value();
  }

  if (!exact_equal_fp(ic_default_cook_parameters.reflect, shader.reflected().value()))
  {
    parameters << " reflect " << shader.reflected().value();
  }

  if (!exact_equal_fp(ic_default_cook_parameters.refrac_index, shader.refractive_index().value()))
  {
    parameters << " ri " << shader.refractive_index().value();
  }

  if (!exact_equal_fp(ic_default_cook_parameters.extinction, shader.extinction().value()))
  {
    parameters << " extinction " << shader.extinction().value();
  }

  m_shader_args = string("{") + parameters.str() + string(" }");

  return Shader_visitor::success;
}

Shader_visitor::Visit_result BC_shader::visit(const Light_shader& shader)
{
  m_shader_name = "light";

  stringstream parameters;

  // if any parameter differs from the default, specify it
  const Unit_vector& target_direction = shader.target_direction();
  if (!exact_equal_fp(ic_default_light_parameters.lt_target[0], target_direction[0].value())
      || !exact_equal_fp(ic_default_light_parameters.lt_target[1], target_direction[1].value())
      || !exact_equal_fp(ic_default_light_parameters.lt_target[2], target_direction[2].value()))
  {
    // TODO: using "d" for parameter name, "target" and "t" do not seem to import into mged, why?
    parameters << " d { ";
    parameters << target_direction[0].value() << " ";
    parameters << target_direction[1].value() << " ";
    parameters << target_direction[2].value() << " }";
  }

  if (!exact_equal_fp(ic_default_light_parameters.lt_intensity, shader.intensity().value()))
  {
    parameters << " bright " << shader.intensity().value();
  }

  if (!exact_equal_fp(ic_default_light_parameters.lt_angle,
                      (shader.beam_dispersion_angle() / Angle::degree).value()))
  {
    parameters << " angle " << (shader.beam_dispersion_angle() / Angle::degree).value();
  }

  // TODO: using "f" for parameter name, "fract" does not seem to import into mged, why?
  if (!exact_equal_fp(ic_default_light_parameters.lt_fraction, shader.fraction().value()))
  {
    parameters << " f " << shader.fraction().value();
  }

  if (ic_default_light_parameters.lt_shadows != shader.shadow_ray_count())
  {
    parameters << " shadows " << shader.shadow_ray_count();
  }

  if (ic_default_light_parameters.lt_infinite != (shader.is_infinite() ? 1 : 0))
  {
    parameters << " infinite " << (shader.is_infinite() ? 1 : 0);
  }

  if (ic_default_light_parameters.lt_visible != (shader.is_visible() ? 1 : 0))
  {
    parameters << " visible " << (shader.is_visible() ? 1 : 0);
  }

  m_shader_args = string("{") + parameters.str() + string(" }");

  return Shader_visitor::success;
}

Shader_visitor::Visit_result BC_shader::visit(const Phong_shader& shader)
{
  m_shader_name = "phong";

  stringstream parameters;

  // if any parameter differs from the default, specify it

  if (ic_default_phong_parameters.shine != shader.shine())
  {
    parameters << " shine " << shader.shine();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.wgt_specular, shader.specular().value()))
  {
    parameters << " specular " << shader.specular().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.wgt_diffuse, shader.diffuse().value()))
  {
    parameters << " diffuse " << shader.diffuse().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.transmit, shader.transmitted().value()))
  {
    parameters << " transmit " << shader.transmitted().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.reflect, shader.reflected().value()))
  {
    parameters << " reflect " << shader.reflected().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.refrac_index, shader.refractive_index().value()))
  {
    parameters << " ri " << shader.refractive_index().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.extinction, shader.extinction().value()))
  {
    parameters << " extinction_per_meter " << shader.extinction().value();
  }

  if (!exact_equal_fp(ic_default_phong_parameters.emission[0], shader.emission_red().value())
      || !exact_equal_fp(ic_default_phong_parameters.emission[1], shader.emission_green().value())
      || !exact_equal_fp(ic_default_phong_parameters.emission[2], shader.emission_blue().value()))
  {
    parameters << " emission { ";
    parameters << shader.emission_red().value() << " ";
    parameters << shader.emission_green().value() << " ";
    parameters << shader.emission_blue().value() << " }";
  }

  m_shader_args = string("{") + parameters.str() + string(" }");

  return Shader_visitor::success;
}

Shader_visitor::Visit_result BC_shader::visit(const Toon_shader& shader)
{
  m_shader_name = "toon";

  // toon shader has no arguments
  m_shader_args = string("{}");

  return Shader_visitor::success;
}
