// Tovero_medal.cpp
//
// Copyright 2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// (1) Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// (2) Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in
// the documentation and/or other materials provided with the
// distribution.
//
// (3) The name of the author may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
// IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

// Note: in order to comply with Tovero and BRL-CAD licensing terms (LGPL 2.1),
//       this program should not be statically linked to those libraries.

// This program generates a Tovero ASCII art medal using a hybrid
// of OpenInventor (Coin) and Tovero.  It uses OpenInventor to
// generate tesselated 3D text for the medal face.

// ../compile_tovero_graphics_cpp Tovero_medal.cpp
// .libs/tovero_medal
// rm tovero_medal.g
// <BRL-CAD_install_dir>/bin/g-stl -a 0.025 -n 0.025 -o tovero_medal.stl tovero_medal.g medal.c

// Tovero
#include <tovero/graphics/brlcad/BC_database.hpp>
#include <tovero/graphics/base/Combination.hpp>
#include <tovero/graphics/base/Database.hpp>
#include <tovero/graphics/base/Color.hpp>
#include <tovero/graphics/base/Combination_attributes.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <tovero/math/geometry/Angle.hpp>
#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_combination.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Solid_operand.hpp>
#include <tovero/math/geometry/Solid_operation.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Torus.hpp>
#include <tovero/math/geometry/Transformation.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/common.hpp>
// C++ library
#include <iostream>
#include <string>
#include <vector>
#include <cstddef>
#include <fstream>
#include <sstream>
// Inventor
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/SoOutput.h>
#include <Inventor/SbViewportRegion.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/VRMLnodes/SoVRMLCoordinate.h>
#include <Inventor/VRMLnodes/SoVRMLNormal.h>
#include <Inventor/VRMLnodes/SoVRMLIndexedFaceSet.h>
#include <Inventor/actions/SoToVRML2Action.h>

using std::cerr;
using std::cout;
using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Angle;
using Roan_trail::Tovero_math::Cone;
using Roan_trail::Tovero_math::Cylinder;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_combination;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Solid_operand;
using Roan_trail::Tovero_math::Solid_operation;
using Roan_trail::Tovero_math::Solid_operator;
using Roan_trail::Tovero_math::Torus;
using Roan_trail::Tovero_math::Transformation;
using Roan_trail::Tovero_math::Triangle_face;
using Roan_trail::Tovero_math::Triangle_mesh;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Vector;

using namespace Roan_trail::Tovero_graphics;

// TODO: protect "reinterpret_cast" assumptions (if <is of type> then <cast>)
namespace
{
  const Distance m = Distance::meter;
  const Distance mm = Distance::millimeter;
  typedef std::vector<Roan_trail::Tovero_math::Point> Point_array;
  typedef std::vector<Roan_trail::Tovero_math::Triangle_face> Triangle_face_array;
  typedef std::vector<Roan_trail::Tovero_math::Vector> Vector_array;
  typedef std::vector<Roan_trail::Tovero_math::Unit_vector> Unit_vector_array;

  SoCallbackAction::Response output_indexed_face_set_callback(void* data, // Solid Combination
                                                              SoCallbackAction* action,
                                                              const SoNode* node)
  {
    static int shape_number = 0;
    shape_number++;
    stringstream node_name_stream;
    node_name_stream << "shape_" << shape_number;
    const string& node_name = node_name_stream.str();

    Solid_combination& letters = *reinterpret_cast<Solid_combination*>(data);

    Point_array vertices;

    const SoVRMLIndexedFaceSet* indexed_face_set = reinterpret_cast<const SoVRMLIndexedFaceSet*>(node);
    const SoVRMLCoordinate* coord = reinterpret_cast<const SoVRMLCoordinate*>(indexed_face_set->coord.getValue());
    const int number_vertices = coord->point.getNum();
    for (int vertex_number = 0; vertex_number < number_vertices; ++vertex_number)
    {
      vertices.push_back(Point(coord->point[vertex_number][0],
                               coord->point[vertex_number][1],
                               coord->point[vertex_number][2],
                               mm));
    }
    Triangle_face_array faces;
    const int number_indices = indexed_face_set->coordIndex.getNum();
    int index_count = 0;
    bool skip_over_index = false;
    int32_t face_index[3];
    int total_faces = 0;
    for (int index_number = 0; index_number < number_indices; ++index_number)
    {
      if (index_count > 2)
      {
        if (!skip_over_index)
        {
          if (indexed_face_set->coordIndex[index_number] == -1)
          {
            faces.push_back(Triangle_face(face_index[0],
                                          face_index[1],
                                          face_index[2]));
            index_count = 0;
            ++total_faces;
          }
          else
          {
            skip_over_index = true;
            cerr << "Unsupported face for " << node_name << " (face " << index_number << ", ";
            cerr << "only triangles allowed)" << endl;
            ++index_count;
          }
        }
      }
      else
      {
        if (indexed_face_set->coordIndex[index_number] != -1)
        {
          face_index[index_count] = indexed_face_set->coordIndex[index_number];
          ++index_count;
        }
        else
        {
          cerr << "Unsupported face for " << node_name << " (face " << index_number << ", ";
          cerr << "only triangles allowed)" << endl;
          index_count = 0;
        }
      }
    }
    Unit_vector_array normals;
#if 1
    const SoVRMLNormal* normal = reinterpret_cast<const SoVRMLNormal*>(indexed_face_set->normal.getValue());
    const int number_normals = normal->vector.getNum();
    for (int normal_number = 0; normal_number < number_normals; ++normal_number)
    {
      normals.push_back(Unit_vector(Unitless(normal->vector[normal_number][0]),
                                    Unitless(normal->vector[normal_number][1]),
                                    Unitless(normal->vector[normal_number][2])));
    }
#endif
    Triangle_face_array face_normals;
#if 1
    const int number_normal_indices = indexed_face_set->coordIndex.getNum();
    int normal_index_count = 0;
    bool skip_over_normal_index = false;
    int32_t normal_index[3];
    int total_normals = 0;
    for (int normal_index_number = 0; normal_index_number < number_normal_indices; ++normal_index_number)
    {
      if (normal_index_count > 2)
      {
        if (!skip_over_index)
        {
          if (indexed_face_set->normalIndex[normal_index_number] == -1)
          {
            face_normals.push_back(Triangle_face(normal_index[0],
                                                 normal_index[1],
                                                 normal_index[2]));
            ++total_normals;
            normal_index_count = 0;
          }
          else
          {
            skip_over_index = true;
            cerr << "Unsupported normal " << node_name << " (normal " << normal_index_count << ", ";
            cerr << "(only triangles allowed)" << endl;
            ++normal_index_count;
          }
        }
      }
      else
      {
        if (indexed_face_set->normalIndex[normal_index_number] != -1)
        {
          normal_index[normal_index_count] = indexed_face_set->coordIndex[normal_index_number];
          ++normal_index_count;
        }
        else
        {
          cerr << "Unsupported normal " << node_name << " (normal " << normal_index_count << ", ";
          cerr << "(only triangles allowed)" << endl;
          normal_index_count = 0;
        }
      }
    }
#endif
    stringstream triangle_mesh_name_stream;
    triangle_mesh_name_stream << node_name << "_triangle_mesh.s";
    const string& triangle_mesh_name = triangle_mesh_name_stream.str();

    Triangle_mesh* text = new Triangle_mesh(Triangle_face::direction_none,
                                            vertices,
                                            faces,
                                            normals,
                                            face_normals,
                                            false,
                                            false,
                                            triangle_mesh_name);
    letters.add_member(*text);

    return SoCallbackAction::CONTINUE;
  }

  void output_model(const SoNode* root_node)
  {
    Distance zero_mm(0.0, Distance::meter);
    Unitless zero(0.0);
    Point origin(zero_mm,
                 zero_mm,
                 zero_mm);
    const Unit_vector x_axis(Unitless(1.0),
                             zero,
                             zero);
    const Unit_vector y_axis(zero,
                             Unitless(1.0),
                             zero);
    const Unit_vector z_axis(zero,
                             zero,
                             Unitless(1.0));

    // dimensions
    const Point medal_base_point;
    const Distance medal_base_radius = mm * 38.0;
    const Unitless sizing_factor(medal_base_radius / (mm * 20.0));
    const Distance medal_top_radius = medal_base_radius - mm * sizing_factor;
    const Distance medal_height = mm * 5.0 * sizing_factor;
    const Distance medal_rim_width = mm * sizing_factor;
    const Distance medal_cutout_depth = mm * 1.5 * sizing_factor;

    Cone* medal_cone = new Cone(medal_base_point,
                                Vector(zero_mm, medal_height, zero_mm),
                                medal_base_radius,
                                medal_top_radius,
                                "medal_cone.s");

    Cylinder* medal_cutout = new Cylinder(medal_base_point
                                          + Vector(zero_mm,
                                                   (medal_height - medal_cutout_depth),
                                                   zero_mm),
                                          Vector(zero_mm,
                                                 medal_cutout_depth * 2.0,
                                                 zero_mm),
                                          medal_top_radius - medal_rim_width,
                                          "medal_cutout.s");

    Solid_operation* basic_medal = new Solid_operation(*(new Solid_operand(*medal_cone)),
                                                       *(new Solid_operand(*medal_cutout)),
                                                       Solid_operator::difference_op,
                                                       "basic_medal.c");

    Angle medal_rotation_angle(Angle::degree * 90.0);
    Transformation* medal_transform = new Transformation;
    medal_transform->set_rotation(x_axis, medal_rotation_angle);

    Solid_combination* letters = new Solid_combination("letters.c");

    // Use OpenInventor to render text for Tovero's triangle mesh
    SoCallbackAction output_action;

    output_action.addPostCallback(SoVRMLIndexedFaceSet::getClassTypeId(),
                                  output_indexed_face_set_callback,
                                  letters);
    output_action.apply(const_cast<SoNode*>(root_node));

    SbViewportRegion text_viewport;
    SoGetBoundingBoxAction text_bounding_box_action(text_viewport);
    text_bounding_box_action.apply(const_cast<SoNode*>(root_node));
    SbBox3f text_bounding_box = text_bounding_box_action.getBoundingBox();
    float xmin;
    float ymin;
    float zmin;
    float xmax;
    float ymax;
    float zmax;
    text_bounding_box.getBounds(xmin,
                                ymin,
                                zmin,
                                xmax,
                                ymax,
                                zmax);

    float letter_scale = 0.28 * sizing_factor.value();
    float yoff = (ymax - ymin);
    float zoff = (zmax - zmin);


    // raised lettering version

    Transformation* letter_raised_transform = new Transformation;
    letter_raised_transform->set_translation(zero_mm,
                                             mm * yoff / 2.0 - mm * yoff / 11.0,
                                             zero_mm);
    letter_raised_transform->scale(Unitless(letter_scale),
                                   Unitless(letter_scale),
                                   sizing_factor);
    letter_raised_transform->translate(zero_mm,
                                       zero_mm,
                                       (medal_height - medal_cutout_depth
                                        + mm * zoff * sizing_factor
                                        - mm * .01 * sizing_factor.value()));

    Solid_combination* medal_raised_combo = new Solid_combination("medal.c");
    medal_raised_combo->add_member(*(new Solid_member(*(new Solid_operand(*basic_medal, medal_transform)))));
    medal_raised_combo->add_member(*(new Solid_member(*(new Solid_operand(*letters, letter_raised_transform)),
                                                      Solid_operator::union_op)));

    BC_database BC_DB_raised("Tovero 2014 Medal");
    Reference_counting_list<Solid>& out_raised_solids = BC_DB_raised.top_solids();
    out_raised_solids.push_back(*medal_raised_combo);

    Error_param error_raised;
    const bool wrote_DB_raised = BC_DB_raised.write("tovero_medal.g",
                                                    true,
                                                    error_raised);
    if (!wrote_DB_raised)
    {
      cerr << endl << "Error writing tovero_medal.g" << endl;
      cerr << "Diagnostic: " << error_raised()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cerr << *error_raised() << endl;
    }
  }
}


int main(int argc, char **argv)
{
  // create text using OpenInventor
  SoDB::init();
  SoGroup *root = new SoGroup;
  root->ref();

  // Control fineness of text mesh
  SoComplexity* complexity = new SoComplexity;
  complexity->value = 1.0;
  complexity->textureQuality = 1.0;

  root->addChild(complexity);

  SoFont* font = new SoFont;
  font->name.setValue("Courier:Bold");
  root->addChild(font);

  // Add the text
  SoText3* text_node = new SoText3;
  const char* lines[] =
    {
      "Tovero",
      "Freedom Medal",
      "(\\ \"\"\" /)    ",
      "% ^---^ %    ",
      "%{ @ @ }%%   ",
      " ( *** ) \\%% ",
      " / ** /   \\%%",
      "/ ** /|      ",
      "\\oo / | sjm  ",
      " ~~'         ",
      "2014",
    };
  text_node->string.setValues(0, 11, lines);
  text_node->justification.setValue(SoText3::CENTER);
  text_node->parts.setValue(SoText3::ALL);

  root->addChild(text_node);

  SoToVRML2Action* VRML = new SoToVRML2Action;
  VRML->apply(root);

  SoNode *converted_root = reinterpret_cast<SoNode *>(VRML->getVRML2SceneGraph());
  converted_root->ref();

#if 0
  // output a VRML version
  SoOutput out;
  out.openFile("tovero_medal.wrl");
  out.setBinary(FALSE);
  out.setHeaderString("#VRML V2.0 utf8");

  SoWriteAction writeAction(&out);
  writeAction.apply(converted_root);
  out.closeFile();
#endif

  // now output ".g" model using Tovero shapes and tesselated VRML text
  output_model(converted_root);
}
