// Triangle_mesh_demo
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// (1) Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// (2) Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in
// the documentation and/or other materials provided with the
// distribution.
//
// (3) The name of the author may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
// IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

// Note: in order to comply with Tovero and BRL-CAD licensing terms (LGPL 2.1),
//       this program should not be statically linked to those libraries.

// Sample model using a triangle mesh

#include <tovero/graphics/brlcad/BC_database.hpp>
#include <tovero/math/geometry/Area.hpp>
#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Plate_triangle_mesh.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_combination.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/Reference_counting_list.hpp>

#include <iostream>
#include <vector>

using Roan_trail::Tovero_math::Area;
using Roan_trail::Tovero_math::Cone;
using Roan_trail::Tovero_math::Cylinder;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::General_cone;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Polyhedron;
using Roan_trail::Tovero_math::Plate_triangle_mesh;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_combination;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Triangle_face;
using Roan_trail::Tovero_math::Triangle_mesh;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Vector;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Reference_counting_list;

using std::cout;
using std::endl;
using std::vector;

namespace
{
  const Distance m = Distance::meter;
  typedef std::vector<bool> Bool_array;
  typedef std::vector<Roan_trail::Tovero_math::Distance> Distance_array;
  typedef std::vector<Roan_trail::Tovero_math::Point> Point_array;
  typedef std::vector<Roan_trail::Tovero_math::Triangle_face> Triangle_face_array;
  typedef std::vector<Roan_trail::Tovero_math::Unit_vector> Unit_vector_array;
}

using namespace Roan_trail::Tovero_graphics;

void make_test();

int main(int argc, char* argv[])
{
  make_test();

  return 0;
}

void make_test()
{
  Point v0(0.0, 0.0, 0.0, m);
  Point v1(1.0, 0.0, 0.0, m);
  Point v2(0.0, 1.0, 0.0, m);
  Point v3(0.0, 0.0, -1.0, m);

  Point_array vertices;
  vertices.push_back(v0);
  vertices.push_back(v1);
  vertices.push_back(v2);
  vertices.push_back(v3);

  Triangle_face f0(0, 1, 2);
  Triangle_face f1(2, 1, 3);
  Triangle_face f2(3, 0, 2);
  Triangle_face f3(1, 0, 3);

  Triangle_face_array faces;
  faces.push_back(f0);
  faces.push_back(f1);
  faces.push_back(f2);
  faces.push_back(f3);

  Area cross_area;
  Unit_vector n0 = (v1 - v0).cross(v2 - v0, cross_area);
  Unit_vector n1 = (v1 - v2).cross(v3 - v2, cross_area);
  Unit_vector n2 = (v0 - v3).cross(v2 - v3, cross_area);
  Unit_vector n3 = (v0 - v1).cross(v3 - v1, cross_area);

  Unit_vector_array normals;
  normals.push_back(n0);
  normals.push_back(n1);
  normals.push_back(n2);
  normals.push_back(n3);

  Triangle_face f0_normals(0, 0, 0);
  Triangle_face f1_normals(1, 1, 1);
  Triangle_face f2_normals(2, 2, 2);
  Triangle_face f3_normals(3, 3, 3);

  Triangle_face_array face_normals;
  face_normals.push_back(f0_normals);
  face_normals.push_back(f1_normals);
  face_normals.push_back(f2_normals);
  face_normals.push_back(f3_normals);

  Distance_array thicknesses;
  thicknesses.push_back(Distance(1.0, m));
  Bool_array append_thicknesses;
  append_thicknesses.push_back(true);

  Triangle_mesh* mesh = new Triangle_mesh(Triangle_face::direction_counterclockwise,
                                          vertices,
                                          faces,
                                          normals,
                                          face_normals,
                                          true,
                                          true,
                                          "mesh.s");
  Cylinder* normal0_line = new Cylinder(v0,
                                        0.5 * m * n0,
                                        0.05 * m,
                                        "normal0_line.s");
  Cone* normal0_hat = new Cone(v0 + 0.5 * m * n0,
                               0.1 * m * n0,
                               0.1 * m,
                               0.001 * m,
                               "normal0_hat.s");
  Cylinder* normal1_line = new Cylinder(v1,
                                        0.5 * m * n1,
                                        0.05 * m,
                                        "normal1_line.s");
  Cone* normal1_hat = new Cone(v1 + 0.5 * m * n1,
                               0.1 * m * n1,
                               0.1 * m,
                               0.001 * m,
                               "normal1_hat.s");
  Cylinder* normal2_line = new Cylinder(v2,
                                        0.5 * m * n2,
                                        0.05 * m,
                                        "normal2_line.s");
  const Point& base = normal2_line->base();
  const Vector& height = normal2_line->height();
  const Distance& radius = normal2_line->radius();
  General_cone& general_cone = normal2_line->generalize();
  const Vector& base_a = general_cone.base_a();
  const Vector& base_b = general_cone.base_b();
  Cone* normal2_hat = new Cone(v2 + 0.5 * m * n2,
                               0.1 * m * n2,
                               0.1 * m,
                               0.001 * m,
                               "normal2_hat.s");
  Cylinder* normal3_line = new Cylinder(v3,
                                        0.5 * m * n3,
                                        0.05 * m,
                                        "normal3_line.s");
  Cone* normal3_hat = new Cone(v3 + 0.5 * m * n3,
                               0.1 * m * n3,
                               0.1 * m,
                               0.001 * m,
                               "normal3_hat.s");

  Solid_combination* normal0 = new Solid_combination("normal0.c");
  *normal0 << *normal0_hat;
  *normal0 << *normal0_line;
  Solid_combination* normal1 = new Solid_combination("normal1.c");
  *normal1 << *normal1_hat;
  *normal1 << *normal1_line;
  Solid_combination* normal2 = new Solid_combination("normal2.c");
  *normal2 << *normal2_hat;
  *normal2 << *normal2_line;
  Solid_combination* normal3 = new Solid_combination("normal3.c");
  *normal3 << *normal3_hat;
  *normal3 << *normal3_line;

  Solid_combination* normal_viz = new Solid_combination("normal_viz.c");
  *normal_viz << *normal0;
  *normal_viz << *normal1;
  *normal_viz << *normal2;
  *normal_viz << *normal3;

  Polyhedron* poly = new Polyhedron(vertices,
                                    "poly.s");

  BC_database BC_DB("Test Normals");
  Reference_counting_list<Solid>& out_solids = BC_DB.top_solids();
  out_solids.push_back(*normal_viz);
  out_solids.push_back(*mesh);
  out_solids.push_back(*poly);

  Error_param error;
  const bool wrote_DB = BC_DB.write("test_normals.g",
                                    true,
                                    error);
  if (!wrote_DB)
  {
    cout << endl << "Error writing test_normals.g" << endl;
    cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
    cout << *error() << endl;
  }
}
