#!/usr/bin/ruby
#
# bus.rb
#
# Copyright 2013-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

# TODO: cleanup specifications, dimensions, accessors
# TODO: BRL-CAD naming scheme

#
# Command line argument processing first
#

mode = ARGV[0]

if mode == 'edit'
  raytrace = false
elsif mode == 'render' or mode == nil
  raytrace = true
else
  abort "error, if an argument is supplied, it must be edit or render"
end

# These load the extension libraries
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'
require 'libtovero_graphics_rb_1'

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

# TODO: separate out as external Ruby module

module ToveroRuby

  #
  # Geometric constants
  #

  M = Distance::meter
  ZERO_M = Distance::meter * 0.0
  ZERO = Unitless.new(0.0)
  ORIGIN = Point.new(ZERO_M,
                     ZERO_M,
                     ZERO_M)
  X_AXIS = UnitVector.new(Unitless.new(1.0),
                          ZERO,
                          ZERO)
  Y_AXIS = UnitVector.new(ZERO,
                          Unitless.new(1.0),
                          ZERO)
  Z_AXIS = UnitVector.new(ZERO,
                          ZERO,
                          Unitless.new(1.0))

end

include ToveroRuby

# TODO: could be in a material library:

#
# Materials
#

#   rubber
rubber_shader = PhongShader.new
rubber_shader.shine = 1
rubber_shader.specular = Unitless.new(0.2)
rubber_shader.diffuse = Unitless.new(0.6)
rubber_shader.reflected = Unitless.new(0.1)
rubber_color = Color.new(10, 10, 10, 0)
$rubber_attributes = CombinationAttributes.new
$rubber_attributes.shader = rubber_shader
$rubber_attributes.color = rubber_color
$rubber_attributes.is_part = true
#   trim
trim_color = Color.new(255, 255, 255, 0)
$trim_attributes = CombinationAttributes.new
$trim_attributes.color = trim_color
$trim_attributes.is_part = true
#   glass
glass_shader = PhongShader::glass
glass_color = Color.new(200, 220, 220, 0)
$glass_attributes = CombinationAttributes.new
$glass_attributes.shader = glass_shader
$glass_attributes.color = glass_color
$glass_attributes.is_part = true
#   body
body_shader = PhongShader::plastic
body_shader.reflected = Unitless.new(0.1)
body_color = Color.new(0, 160, 255, 0)
$body_attributes = CombinationAttributes.new
$body_attributes.shader = body_shader
$body_attributes.color = body_color
$body_attributes.is_part = true
#   headlight glass
headlight_glass_shader = PhongShader::glass
headlight_glass_shader.transmitted = Unitless.new(0.1)
headlight_glass_color = Color.new(130, 200, 210, 0)
$headlight_glass_attributes = CombinationAttributes.new
$headlight_glass_attributes.shader = headlight_glass_shader
$headlight_glass_attributes.color = headlight_glass_color
$headlight_glass_attributes.is_part = true
#   wheel rim
wheel_rim_color = Color.new(180, 180, 180, 0)
$wheel_rim_attributes = CombinationAttributes.new
$wheel_rim_attributes.color = wheel_rim_color
$wheel_rim_attributes.is_part = true
#   light
light_color = Color.new(255, 255, 255, 0)
light_shader = LightShader.new
light_shader.is_visible = false
light_shader.intensity = Unitless.new(0.7)
light_shader.beam_dispersion_angle = Angle::degree * 120.0
$light_attributes = CombinationAttributes.new
$light_attributes.shader = light_shader
$light_attributes.color = light_color
$light_attributes.is_part = true
#   background color
background_color = Color.new(224, 212, 190, 0)

#
$separator = '.'

#
# Part classes
#

class BusPart
  # instance variables
  attr_accessor :parent
  attr_reader :name

  # constructor
  def initialize(parent, name)
    @parent = parent
    @name = name
  end

  def full_name
    return_name = ""
    ancestor = self
    while ancestor != nil
      return_name = "#{$separator}#{ancestor.name()}" + return_name
      ancestor = ancestor.parent
    end

    return return_name
  end
end

class Dish < BusPart
  # instance variables
  #   specifications
  attr_accessor :curvature_radius
  attr_accessor :disk_radius
  attr_accessor :direction
  attr_accessor :up
  attr_accessor :offset
  #
  attr_reader :shape

  # constructor
  def initialize(curvature_radius,
                 disk_radius,
                 direction,
                 up,
                 offset,
                 parent,
                 name)
    super(parent, name)
    @curvature_radius = curvature_radius
    @disk_radius = disk_radius
    @direction = direction
    @up = up
    @offset = offset
  end

  def render
    x = sqrt(@curvature_radius * @curvature_radius - @disk_radius * @disk_radius)
    basis = @direction.cross(@up)
    center_offset = @direction * x
    sphere = Sphere.new(@offset - center_offset,
                        @curvature_radius,
                        "#{full_name}#{$separator}sphere.s")
    direction_offset = @direction * @curvature_radius
    up_offset = @up * @curvature_radius
    basis_offset = basis * @curvature_radius
    radius_offset = direction_offset + up_offset + basis_offset
    box = Box.new(@offset - center_offset - radius_offset,
                  direction_offset + center_offset,
                  up_offset * Unitless.new(2.0),
                  basis_offset * Unitless.new(2.0),
                  "#{full_name}#{$separator}sphere.s")
    @shape = SolidCombination.new("#{full_name}")
    @shape << sphere
    @shape << SolidMember.new(box, SolidOperator::Difference_op)
  end
end

class BusBody < BusPart
  # instance variables
  #   specifications
  attr_accessor :radiusing_value
  attr_accessor :shell_thickness
  attr_accessor :panel_gap
  #
  attr_reader :solid
  attr_reader :front_solid
  attr_reader :rear_solid
  attr_reader :windshield_box
  attr_reader :rear_window_boxes
  #
  attr_reader :assembly


  def initialize(parent, name)
    super(parent, name)

    # default values for instance variables
    #   specifications
    @radiusing_value = M * 0.20
    @shell_thickness = M * 0.05
    @panel_gap = M * 0.02
  end

  def calculate_dimensions
    radiusing_diameter = @radiusing_value * 2.0
    @radiused_length = @parent.length() - radiusing_diameter
    @top_radiused_length = @parent.top_length() - radiusing_diameter
    @radiused_width = @parent.width() - radiusing_diameter
    @radiused_height = @parent.height() - radiusing_diameter
  end

  def render_frame
    radiusing_base = Point.new(@radiusing_value,
                               @radiusing_value,
                               @radiusing_value)
    frame_radiusing_sphere = Sphere.new(radiusing_base,
                                        @radiusing_value,
                                        "#{full_name()}#{$separator}frame_radiusing_sphere.s")
    lower_frame_length_cylinder = Cylinder.new(radiusing_base,
                                               Vector.new(@radiused_length,
                                                          ZERO_M,
                                                          ZERO_M),
                                               @radiusing_value,
                                               "#{full_name()}#{$separator}lower_frame_length_cylinder.s")
    upper_frame_length_cylinder = Cylinder.new(radiusing_base,
                                               Vector.new(@top_radiused_length,
                                                          ZERO_M,
                                                          ZERO_M),
                                               @radiusing_value,
                                               "#{full_name()}#{$separator}upper_frame_length_cylinder.s")
    frame_width_cylinder = Cylinder.new(radiusing_base,
                                        Vector.new(ZERO_M,
                                                   ZERO_M,
                                                   @radiused_width),
                                        @radiusing_value,
                                        "#{full_name()}#{$separator}frame_width_cylinder.s")
    rear_frame_height_cylinder = Cylinder.new(radiusing_base,
                                              Vector.new(ZERO_M,
                                                         @radiused_height,
                                                         ZERO_M),
                                              @radiusing_value,
                                              "#{full_name()}#{$separator}rear_frame_height_cylinder.s")
    front_frame_height = @radiused_height - (@parent.height() \
                                             - (@parent.belt_line_height()  + @radiusing_value))
    front_frame_height_cylinder = Cylinder.new(radiusing_base,
                                               Vector.new(ZERO_M,
                                                          front_frame_height,
                                                          ZERO_M),
                                               @radiusing_value,
                                               "#{full_name()}#{$separator}front_frame_height_cylinder.s")
    frame_width = SolidCombination.new("#{full_name()}#{$separator}frame_width.c")
    frame_width << frame_width_cylinder
    frame_width << frame_radiusing_sphere
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              @radiused_width)
    frame_width << SolidMember.new(SolidOperand.new(frame_radiusing_sphere, transform))
    # front rake
    @front_rake_start_point = Point.new(parent().length - @radiusing_value,
                                        @parent.belt_line_height(),
                                        @radiusing_value)
    front_rake_end_point = Point.new(@parent.top_length() - @radiusing_value,
                                     @parent.height() - @radiusing_value,
                                     @radiusing_value)
    @front_rake_height_vector = front_rake_end_point - @front_rake_start_point
    front_rake_cylinder = Cylinder.new(@front_rake_start_point,
                                       @front_rake_height_vector,
                                       @radiusing_value,
                                       "#{full_name()}#{$separator}front_rake_cylinder.s")
    front_rake = SolidCombination.new("#{full_name()}#{$separator}frame_front_rake.c")
    transform = Transformation.new
    transform.set_translation(@front_rake_start_point.get(0) - @radiusing_value,
                              @front_rake_start_point.get(1) - @radiusing_value,
                              @front_rake_start_point.get(2) - @radiusing_value)
    front_rake << front_rake_cylinder

    # setup rear frame
    @rear_frame = SolidCombination.new("#{full_name()}#{$separator}rear_frame.c")
    #   add rear frame width members
    @rear_frame << frame_width
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              @radiused_height,
                              ZERO_M)
    @rear_frame << SolidMember.new(SolidOperand.new(frame_width, transform))
    #   add rear frame height cylinders
    @rear_frame << rear_frame_height_cylinder
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              @radiused_width)
    @rear_frame << SolidMember.new(SolidOperand.new(rear_frame_height_cylinder, transform))
    #   add lower frame length cylinders
    @rear_frame << lower_frame_length_cylinder
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              @radiused_width)
    @rear_frame << SolidMember.new(SolidOperand.new(lower_frame_length_cylinder, transform))
    #   add upper frame length cylinders
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              @radiused_height,
                              ZERO_M)
    @rear_frame << SolidMember.new(SolidOperand.new(upper_frame_length_cylinder, transform))
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              @radiused_height,
                              @radiused_width)
    @rear_frame << SolidMember.new(SolidOperand.new(upper_frame_length_cylinder, transform))

    # setup front frame
    @front_frame = SolidCombination.new("#{full_name()}#{$separator}front_frame.c")
    #   add front frame width members
    transform = Transformation.new
    transform.set_translation(@radiused_length,
                              ZERO_M,
                              ZERO_M)
    @front_frame << SolidMember.new(SolidOperand.new(frame_width, transform))
    transform = Transformation.new
    transform.set_translation(@radiused_length,
                              front_frame_height,
                              ZERO_M)
    @front_frame << SolidMember.new(SolidOperand.new(frame_width, transform))
    transform = Transformation.new
    transform.set_translation(@top_radiused_length,
                              @radiused_height,
                              ZERO_M)
    @front_frame << SolidMember.new(SolidOperand.new(frame_width, transform))
    #   add front frame height cylinders
    transform = Transformation.new
    transform.set_translation(@radiused_length,
                              ZERO_M,
                              ZERO_M)
    @front_frame << SolidMember.new(SolidOperand.new(front_frame_height_cylinder, transform))
    transform = Transformation.new
    transform.set_translation(@radiused_length,
                              ZERO_M,
                              @radiused_width)
    @front_frame << SolidMember.new(SolidOperand.new(front_frame_height_cylinder, transform))
    #   add front rake pillars
    @front_frame << front_rake
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              @radiused_width)
    @front_frame << SolidMember.new(SolidOperand.new(front_rake, transform))

    #   add front and back frames to the main frame
    @frame = SolidCombination.new("#{full_name()}#{$separator}frame.c")
    @frame << @front_frame
    @frame << @rear_frame
  end

  def render_fill
    body_rear_length = @parent.top_length() - @radiusing_value
    body_width = @parent.width() - @radiusing_value
    body_rear_height = @parent.height() - @radiusing_value
    # rear boxes
    rear_length_box = AxisAlignedBox.new(Point.new(ZERO_M,
                                                   @radiusing_value,
                                                   @radiusing_value),
                                         Point.new(@parent.top_length(),
                                                   body_rear_height,
                                                   body_width),
                                         "#{full_name()}#{$separator}rear_length_box.s")
    rear_width_box = AxisAlignedBox.new(Point.new(@radiusing_value,
                                                  @radiusing_value,
                                                  ZERO_M),
                                        Point.new(body_rear_length,
                                                  body_rear_height,
                                                  @parent.width()),
                                        "#{full_name()}#{$separator}rear_width_box.s")
    rear_height_box = AxisAlignedBox.new(Point.new(@radiusing_value,
                                                   ZERO_M,
                                                   @radiusing_value),
                                         Point.new(body_rear_length,
                                                   @parent.height(),
                                                   body_width),
                                         "#{full_name()}#{$separator}rear_height_box.s")

    # front boxes
    front_start_length = @parent.top_length() - @radiusing_value
    front_length = @parent.length() - @radiusing_value
    front_height = @parent.belt_line_height()
    front_length_box = AxisAlignedBox.new(Point.new(front_start_length,
                                                    @radiusing_value,
                                                    @radiusing_value),
                                          Point.new(@parent.length(),
                                                    front_height,
                                                    body_width),
                                          "#{full_name()}#{$separator}front_length_box.s")
    front_width_box = AxisAlignedBox.new(Point.new(front_start_length,
                                                   @radiusing_value,
                                                   ZERO_M),
                                         Point.new(front_length,
                                                   front_height,
                                                   @parent.width()),
                                         "#{full_name()}#{$separator}front_width_box.s")
    front_height_box = AxisAlignedBox.new(Point.new(front_start_length,
                                                    ZERO_M,
                                                    @radiusing_value),
                                          Point.new(front_length,
                                                    @parent.belt_line_height() + @radiusing_value,
                                                    body_width),
                                          "#{full_name()}#{$separator}front_height_box.s")
    #   rake
    front_rake_width = Vector.new(ZERO_M,
                                  ZERO_M,
                                  @radiused_width)
    front_rake_u = UnitVector.new
    front_rake_width.normalize(front_rake_u)
    front_rake_v = UnitVector.new
    @front_rake_height_vector.normalize(front_rake_v)
    w = front_rake_v.cross(front_rake_u)
    front_rake_length = w * @radiusing_value
    front_rake_box = Box.new(@front_rake_start_point,
                             front_rake_width,
                             @front_rake_height_vector,
                             front_rake_length,
                             "#{full_name()}#{$separator}front_rake_box.s")
    # wedge
    wedge = Wedge.new(Point.new(@parent.top_length() - @radiusing_value,
                                @parent.belt_line_height(),
                                ZERO_M),
                      Vector.new(@parent.length() - @parent.top_length(),
                                 ZERO_M,
                                 ZERO_M),
                      Vector.new(ZERO_M,
                                 @parent.height() - (@parent.belt_line_height() + @radiusing_value),
                                 ZERO_M),
                      Vector.new(ZERO_M,
                                 ZERO_M,
                                 @parent.width()),
                      "#{full_name()}#{$separator}wedge.s")

    # setup the fill combinations
    #   rear
    @rear_fill = SolidCombination.new("#{full_name()}#{$separator}rear_fill.c")
    @rear_fill << rear_length_box
    @rear_fill << rear_width_box
    @rear_fill << rear_height_box
    #   front
    @front_fill = SolidCombination.new("#{full_name()}#{$separator}front_fill.c")
    @front_fill << front_length_box
    @front_fill << front_width_box
    @front_fill << front_height_box
    @front_fill << front_rake_box
    @front_fill << wedge
    #   main fill
    @fill = SolidCombination.new("#{full_name()}#{$separator}fill.c")
    #     add the front and rear
    @fill << @rear_fill
    @fill << @front_fill
  end

  def render_wheel_wells
    wheel_well_radius = @parent.wheel().height() / 2.0 + @parent.wheel_to_body_clearance()
    wheel_well_width = @parent.width() + @parent.wheel_to_body_clearance() * 2.0
    wheel_well_solid = Cylinder.new(ORIGIN,
                                    Vector.new(ZERO_M,
                                               ZERO_M,
                                               wheel_well_width),
                                    wheel_well_radius,
                                    "#{full_name()}#{$separator}wheel_well_solid.s")

    @wheel_wells_solid = SolidCombination.new("#{full_name()}#{$separator}wheel_wells_solid.c")
    transform = Transformation.new
    transform.set_translation(@parent.rear_wheel_x(),
                              ZERO_M,
                              ZERO_M)
    @wheel_wells_solid << SolidMember.new(SolidOperand.new(wheel_well_solid, transform))
    transform = Transformation.new
    transform.set_translation(@parent.rear_wheel_x() + @parent.wheel_base(),
                              ZERO_M,
                              ZERO_M)
    @wheel_wells_solid << SolidMember.new(SolidOperand.new(wheel_well_solid, transform))

    wheel_well = SolidCombination.new("#{full_name()}#{$separator}wheel_well.c")
    transform = Transformation.new
    transform.set_scale((wheel_well_radius - @shell_thickness) / wheel_well_radius,
                        (wheel_well_radius - @shell_thickness) / wheel_well_radius,
                        Unitless.new(1.0))
    transform.translate(ZERO_M,
                        ZERO_M,
                        ZERO_M)
    wheel_well << wheel_well_solid
    wheel_well << SolidMember.new(SolidOperand.new(wheel_well_solid, transform), SolidOperator::Difference_op)


    @wheel_wells = SolidCombination.new("#{full_name()}#{$separator}wheel_wells.c")
    transform = Transformation.new
    transform.set_translation(@parent.rear_wheel_x(),
                              ZERO_M,
                              ZERO_M)
    @wheel_wells << SolidMember.new(SolidOperand.new(wheel_well, transform))
    transform = Transformation.new
    transform.set_translation(@parent.rear_wheel_x() + @parent.wheel_base(),
                              ZERO_M,
                              ZERO_M)
    @wheel_wells << SolidMember.new(SolidOperand.new(wheel_well, transform))
    @wheel_wells << SolidMember.new(@solid, SolidOperator::Intersection_op)
  end

  def render_window_boxes
    @rear_window_boxes = SolidCombination.new("#{full_name()}#{$separator}rear_window_boxes.c")

    #     windshield box
    windshield_rear_x = @parent.top_length() - @radiusing_value
    window_base_y = @parent.belt_line_height() + @radiusing_value / 2.0
    window_top_y = @parent.height() - @radiusing_value
    @windshield_box = AxisAlignedBox.new(Point.new(windshield_rear_x,
                                                   window_base_y,
                                                   -@radiusing_value),
                                         Point.new(@parent.length() + @radiusing_value,
                                                   window_top_y,
                                                   @parent.width() + @radiusing_value),
                                         "#{full_name()}#{$separator}windshield_box.s")
    #     driver window box
    front_window_rear_x = windshield_rear_x - @parent.front_window_width - @radiusing_value
    left_window_inner_z = @shell_thickness + @radiusing_value
    driver_window_box = AxisAlignedBox.new(Point.new(front_window_rear_x,
                                                     window_base_y,
                                                     -@radiusing_value),
                                           Point.new(front_window_rear_x + @parent.front_window_width(),
                                                     window_top_y,
                                                     left_window_inner_z),
                                           "#{full_name()}#{$separator}driver_window_box.s")
    #     passenger window box
    right_window_inner_z = @parent.width() - @shell_thickness - @radiusing_value
    passenger_window_box = AxisAlignedBox.new(Point.new(front_window_rear_x,
                                                        window_base_y,
                                                        right_window_inner_z),
                                              Point.new(front_window_rear_x + @parent.front_window_width(),
                                                        window_top_y,
                                                        @parent.width() + @radiusing_value),
                                              "#{full_name()}#{$separator}passenger_window_box.s")
    #     back left window box
    back_window_rear_x = @radiusing_value * 2.0
    back_left_window_box = AxisAlignedBox.new(Point.new(back_window_rear_x,
                                                        window_base_y,
                                                        -@radiusing_value),
                                              Point.new(back_window_rear_x + @parent.back_window_width(),
                                                        window_top_y,
                                                        left_window_inner_z),
                                              "#{full_name()}#{$separator}back_left_window_box.s")
    #     back right window box
    back_right_window_box = AxisAlignedBox.new(Point.new(back_window_rear_x,
                                                         window_base_y,
                                                         right_window_inner_z),
                                               Point.new(back_window_rear_x + @parent.back_window_width(),
                                                         window_top_y,
                                                         @parent.width() + @radiusing_value),
                                               "#{full_name()}#{$separator}back_right_window_box.s")
    #     middle left window box
    middle_window_rear_x = back_window_rear_x + @parent.back_window_width() + @radiusing_value
    middle_left_window_box = AxisAlignedBox.new(Point.new(middle_window_rear_x,
                                                          window_base_y,
                                                          -@radiusing_value),
                                                Point.new(middle_window_rear_x + @parent.back_window_width(),
                                                          window_top_y,
                                                          left_window_inner_z),
                                                "#{full_name()}#{$separator}middle_left_window_box.s")
    #     middle right window box
    middle_right_window_box = AxisAlignedBox.new(Point.new(middle_window_rear_x,
                                                           window_base_y,
                                                           right_window_inner_z),
                                                 Point.new(middle_window_rear_x + parent.back_window_width(),
                                                           window_top_y,
                                                           @parent.width() + @radiusing_value),
                                                 "#{full_name()}#{$separator}middle_right_window_box.s")
    #     rear window box
    rear_window_box = AxisAlignedBox.new(Point.new(-@radiusing_value,
                                                   window_base_y,
                                                   @radiusing_value),
                                         Point.new(@shell_thickness + @radiusing_value,
                                                   window_top_y,
                                                   @parent.width() - @radiusing_value),
                                         "#{full_name()}#{$separator}rear_window_box.s")
    @rear_window_boxes << rear_window_box
    @rear_window_boxes << driver_window_box
    @rear_window_boxes << passenger_window_box
    @rear_window_boxes << middle_left_window_box
    @rear_window_boxes << middle_right_window_box
    @rear_window_boxes << back_left_window_box
    @rear_window_boxes << back_right_window_box
  end

  def render_solid
    # body
    @solid = SolidCombination.new("#{full_name()}#{$separator}solid.c")
    @solid << @frame
    @solid << @fill

    # front
    @front_solid = SolidCombination.new("#{full_name()}#{$separator}front_solid.c")
    @front_solid << @front_frame
    @front_solid << @front_fill

    # rear
    @rear_solid = SolidCombination.new("#{full_name()}#{$separator}rear_solid.c")
    @rear_solid << @rear_frame
    @rear_solid << @rear_fill
  end

  def render
    combo = SolidCombination.new("#{full_name()}.c")

    render_frame
    render_fill
    render_window_boxes
    render_solid
    render_wheel_wells

    transform = Transformation.new
    transform.set_scale((@parent.length() - @shell_thickness * 2.0) / @parent.length(),
                        (@parent.height() - @shell_thickness * 2.0) / @parent.height(),
                        (@parent.width() - @shell_thickness * 2.0) / @parent.width())
    transform.translate(@shell_thickness,
                        @shell_thickness,
                        @shell_thickness)

    combo << @solid
    combo << SolidMember.new(SolidOperand.new(@solid, transform),
                             SolidOperator::Difference_op)
    combo << SolidMember.new(@windshield_box, SolidOperator::Difference_op)
    combo << SolidMember.new(@rear_window_boxes, SolidOperator::Difference_op)
    combo << SolidMember.new(@wheel_wells_solid, SolidOperator::Difference_op)
    combo << @wheel_wells

    @assembly = Combination.new($body_attributes, "#{full_name()}.a")
    @assembly << combo
  end
end

class BusWindow < BusPart
  # instance variables
  #   specifications
  attr_accessor :glass_thickness

  def initialize(parent, name)
    super(parent, name)
    # default values for instance variables
    #   specifications
    @glass_thickness = M * 0.00635
    #
  end
end

class BusBumper < BusPart
  # instance variables
  #   specifications
  attr_accessor :radius
  attr_accessor :body_gap
  #
  attr_reader :flattening_offset
  #
  attr_accessor :part

  # constructor
  def initialize(parent, name)
    super(parent, name)
    # default values for instance variables
    @radius = M * 0.125
    @body_gap = M * 0.03
    #
  end

  def calculate_dimensions
    @flattening_offset = @radius / 4.0
  end

  def render
    @part = SolidCombination.new("#{full_name()}#{$separator}bus_bumper.c")

    cross_member_length = @parent.width() + @body_gap * 2.0 - @flattening_offset * 2.0
    cross_member = Cylinder.new(ORIGIN,
                                Vector.new(ZERO_M,
                                           ZERO_M,
                                           cross_member_length),
                                @radius,
                                "#{full_name()}#{$separator}cross_member.s")
    sphere = Sphere.new(ORIGIN,
                        @radius,
                        "#{full_name()}#{$separator}sphere.s")
    extension_length = @radius * 1.0
    extension = Cylinder.new(ORIGIN,
                             Vector.new(extension_length,
                                        ZERO_M,
                                        ZERO_M),
                             @radius,
                             "#{full_name()}#{$separator}sphere.s")

    box = AxisAlignedBox.new(Point.new(-@flattening_offset,
                                       -@radius * 2.0,
                                       -@flattening_offset),
                             Point.new(extension_length + @flattening_offset \
                                       + @radius,
                                       @radius * 2.0,
                                       cross_member_length + @flattening_offset),
                             "#{full_name()}#{$separator}box.s")
    @part << cross_member
    @part << sphere
    @part << extension
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              cross_member_length)
    @part << SolidMember.new(SolidOperand.new(sphere, transform))
    transform = Transformation.new
    transform.set_translation(ZERO_M,
                              ZERO_M,
                              cross_member_length)
    @part << SolidMember.new(SolidOperand.new(extension, transform))
    @part << SolidMember.new(box, SolidOperator::Difference_op)
  end
end

class BusHubCap < BusPart
  # 
  attr_accessor :curvature_radius
  attr_accessor :disk_radius
  attr_accessor :direction
  attr_accessor :up
  attr_accessor :base

  attr_reader :part

  # constructor
  def initialize(parent, name)
    super(parent, name)
  end

  def render
    dish = Dish.new(@curvature_radius,
                    @disk_radius,
                    @direction,
                    @up,
                    @base,
                    self,
                    "dish")
    dish.render()
    @part = Combination.new($trim_attributes, "#{full_name()}.r")
    @part << dish.shape()
  end
end

class BusTire < BusPart
  # instance variables
  #   specifications
  attr_accessor :width
  attr_accessor :sidewall_ratio
  attr_accessor :height
  attr_accessor :center_diameter
  #   dimensions
  attr_reader :sidewall_height
  attr_reader :inner_center_cylinder
  attr_reader :outer_center_cylinder
  #
  attr_reader :part


  def initialize(parent, name)
    super(parent, name)

    # default values for instance variables
    #  specifications (185/80R14 tires)
    @width = M * 0.185
    @sidewall_ratio = 0.8
    #   specifications
    @center_diameter = M * 0.356
  end

  def calculate_dimensions
    @sidewall_height = @width * @sidewall_ratio
    @height = @center_diameter + @sidewall_height * 2.0
    @outer_center_cylinder = Cylinder.new(Point.new(ZERO_M,
                                                    ZERO_M,
                                                    -@width),
                                          Vector.new(ZERO_M,
                                                     ZERO_M,
                                                     @width * 2.0),
                                          @height / 2.0,
                                          "#{full_name()}#{$separator}outer_center_cylinder.s")
    @inner_center_cylinder = Cylinder.new(Point.new(ZERO_M,
                                                    ZERO_M,
                                                    -@width),
                                          Vector.new(ZERO_M,
                                                     ZERO_M,
                                                     @width * 2.0),
                                          @center_diameter / 2.0,
                                          "#{full_name()}#{$separator}inner_center_cylinder.s")
  end

  def render
    combo = SolidCombination.new("#{full_name()}.c")

    torus_inner_radius = (@width * 1.1) / 2.0
    torus_outer_radius = (@height * 1.05) / 2.0 - torus_inner_radius
    torus = Torus.new(Point.new,
                      Z_AXIS,
                      torus_outer_radius,
                      torus_inner_radius,
                      "#{full_name()}#{$separator}torus.s")
    combo << torus
    combo << SolidMember.new(outer_center_cylinder, SolidOperator::Intersection_op)
    combo << SolidMember.new(inner_center_cylinder, SolidOperator::Difference_op)

    @part = Combination.new($rubber_attributes, "#{full_name()}.r")
    @part << combo
  end
end

class BusWheelRim < BusPart
  # instance variables
  #   dimensions
  attr_accessor :diameter
  attr_accessor :width
  #
  attr_accessor :rim_offset
  #
  attr_reader :part

  def initialize(parent, name)
    super(parent, name)
  end

  def render
    @part = Combination.new($wheel_rim_attributes, "#{full_name()}.r")

    wheel_rim_depth = @width / 1.1
    @rim_offset = Vector.new(ZERO_M,
                             ZERO_M,
                             wheel_rim_depth)
    wheel_rim_cylinder = Cylinder.new(Point.new(ZERO_M,
                                                ZERO_M,
                                                -wheel_rim_depth / 2.0),
                                      @rim_offset,
                                      @diameter / 2.0,
                                      "#{full_name()}#{$separator}wheel_rim_cylinder.s")
    @part << wheel_rim_cylinder
  end
end

class BusWheel < BusPart
  # instance variables
  #   dependent dimensions
  attr_reader :height
  #
  attr_reader :tire
  #
  attr_reader :assembly

  def initialize(parent, name)
    super(parent, name)

    # parts
    @tire = BusTire.new(self, "tire")
    @wheel_rim = BusWheelRim.new(self, "wheel_rim")
    @hub_cap = BusHubCap.new(self, "hub_cap")
  end

  def calculate_dimensions
    # update part dimensions
    @tire.calculate_dimensions()
    # update dependent dimensions
    @wheel_rim.diameter = @tire.center_diameter()
    @wheel_rim.width = @tire.width()
    @height = @tire.height()
    @hub_cap.disk_radius = @tire.center_diameter() / 2.0
    @hub_cap.disk_radius = @hub_cap.disk_radius() - @hub_cap.disk_radius() / 10.0
    @hub_cap.curvature_radius = @hub_cap.disk_radius() * 1.8
  end

  def render
    @tire.render
    @wheel_rim.render
    # update dependent geometry
    @hub_cap.base = ORIGIN + @wheel_rim.rim_offset() / Unitless.new(2.0)
    direction = Z_AXIS
    @wheel_rim.rim_offset().normalize(direction)
    @hub_cap.direction = direction
    @hub_cap.up = Y_AXIS
    @hub_cap.render
    @assembly = Combination.new("#{full_name()}.a")
    @assembly << @tire.part()
    @assembly << @wheel_rim.part()
    @assembly << @hub_cap.part()
  end
end

class BusHeadlights < BusPart
  # instance variables
  #   specifications
  attr_accessor :curvature_radius
  attr_accessor :disk_radius
  attr_accessor :direction
  attr_accessor :up
  #
  attr_reader :assembly

  # constructor
  def initialize(parent, name)
    super(parent, name)

    # default values for instance variables
    @curvature_radius = M * 0.300
    @disk_radius = M * 0.15
    @direction = X_AXIS
    @up = Y_AXIS
  end

  def render
    offset_z = (@parent.width() - @parent.headlight_spacing) / 2.0
    left_base = Point.new(@parent.length(),
                          @parent.headlight_height() - @parent.wheel().height() / 2.0,
                          offset_z)
    left_dish = Dish.new(@curvature_radius,
                         @disk_radius,
                         @direction,
                         @up,
                         left_base,
                         self,
                         "left_dish.s")
    left_dish.render()
    glass_offset = Vector.new(M * 0.01,
                              ZERO_M,
                              ZERO_M)
    left_glass = Dish.new(@curvature_radius * 0.6,
                          @disk_radius - M * 0.03,
                          @direction,
                          @up,
                          left_base + glass_offset,
                          self,
                         "left_glass.s")
    left_glass.render()
    left_trim = Combination.new($trim_attributes, "#{full_name()}#{$separator}left_trim.c")
    left_trim << left_dish.shape()
    left_trim << SolidMember.new(left_glass.shape(), SolidOperator::Difference_op)
    left_glass_lens = Combination.new($headlight_glass_attributes, "#{full_name()}#{$separator}left_glass_lens.c")
    left_glass_lens << left_glass.shape()
    left_light = Combination.new("#{full_name()}#{$separator}left_light.c")
    left_light << left_trim
    left_light << left_glass_lens

    right_base = Point.new(@parent.length(),
                           @parent.headlight_height() - @parent.wheel().height() / 2.0,
                           @parent.width() - offset_z)
    right_dish = Dish.new(@curvature_radius,
                           @disk_radius,
                           @direction,
                           @up,
                           right_base,
                           self,
                          "right_dish.s")
    right_dish.render()
    right_glass = Dish.new(@curvature_radius * 0.6,
                           @disk_radius - M * 0.03,
                           @direction,
                           @up,
                           right_base + glass_offset,
                           self,
                           "right_glass.s")
    right_glass.render()
    right_trim = Combination.new($trim_attributes, "#{full_name()}#{$separator}right_trim.c")
    right_trim << right_dish.shape()
    right_trim << SolidMember.new(right_glass.shape(), SolidOperator::Difference_op)
    right_glass_lens = Combination.new($headlight_glass_attributes, "#{full_name()}#{$separator}right_glass_lens.c")
    right_glass_lens << right_glass.shape()
    right_light = Combination.new("#{full_name()}#{$separator}right_light.c")
    right_light << right_trim
    right_light << right_glass_lens

    @assembly = Combination.new("#{full_name()}#{$separator}headlights.c")
    @assembly << left_light
    @assembly << right_light
  end
end

class Bus < BusPart
  # instance variables
  #   specifications
  #     overall
  attr_accessor :width
  attr_accessor :length
  attr_accessor :height
  attr_accessor :ride_height
  attr_accessor :belt_line_height
  #     wheel related
  attr_accessor :track
  attr_accessor :wheel_base
  attr_accessor :wheel_to_body_clearance
  attr_accessor :rear_overhang
  attr_accessor :wheel_turn_angle
  #     headlight related
  attr_accessor :headlight_height
  attr_accessor :headlight_spacing
  #     window related
  attr_accessor :front_window_width
  attr_accessor :back_window_width
  #   dimensions
  attr_reader :top_length
  attr_reader :rear_wheel_x
  attr_reader :wheel_height
  #   parts
  attr_reader :wheel # prototype wheel
  attr_reader :body
  attr_reader :windows
  attr_reader :wheels
  attr_reader :bumpers
  #
  attr_reader :assembly
  #
  attr_reader :database

  # constructor
  def initialize(parent, name)
    super(parent, name)

    # default values for instance variables
    #   specifications
    @width = M * 1.727
    @length = M * 4.505
    @height = M * 2.040
    #
    @ride_height = M * 0.20
    @track = M * 1.374
    @wheel_base = M * 2.400
    @wheel_to_body_clearance = M * 0.10
    @rear_overhang = M * 0.686
    @wheel_turn_angle = Angle::degree * 15.0
    #
    @headlight_height = M * 0.9
    @headlight_spacing = M * 0.9
    #
    @front_window_width = M * 0.6
    @back_window_width = M * 1.2
    #
    @belt_line_height = M * 1.143

    # create a BRL-CAD database
    @database = BCDatabase.new
  end

  def render_body
    @body.render

    @assembly << @body.assembly()
  end

  def render_wheels
    @wheel.render
    #     wheels
    front_wheel_x = @rear_wheel_x + @wheel_base
    left_wheel_z = (@width - @track - @wheel.tire().width()) / 2.0
    right_wheel_z = @width - left_wheel_z
    left_wheel_rotation_angle = Angle::degree * 180.0
    #       rear left
    rear_left_wheel = SolidCombination.new("#{full_name()}#{$separator}rear_left_wheel.c")
    wheel_transform = Transformation.new
    wheel_transform.set_rotation(Y_AXIS, left_wheel_rotation_angle)
    wheel_transform.translate(@rear_wheel_x,
                              ZERO_M,
                              left_wheel_z)
    rear_left_wheel << SolidMember.new(SolidOperand.new(@wheel.assembly(), wheel_transform))

    @assembly << rear_left_wheel

    #       front left
    front_left_wheel = SolidCombination.new("#{full_name()}#{$separator}front_left_wheel.c")
    wheel_transform = Transformation.new
    wheel_transform.set_rotation(Y_AXIS, left_wheel_rotation_angle - @wheel_turn_angle)
    wheel_transform.translate(front_wheel_x,
                              ZERO_M,
                              left_wheel_z)
    front_left_wheel << SolidMember.new(SolidOperand.new(@wheel.assembly(), wheel_transform))

    @assembly << front_left_wheel

    #       rear right
    rear_right_wheel = SolidCombination.new("#{full_name()}#{$separator}rear_right_wheel.c")
    wheel_transform = Transformation.new
    wheel_transform.set_translation(@rear_wheel_x,
                                    ZERO_M,
                                    right_wheel_z)
    rear_right_wheel << SolidMember.new(SolidOperand.new(@wheel.assembly(), wheel_transform))

    @assembly << rear_right_wheel

    #       front right
    front_right_wheel = SolidCombination.new("#{full_name()}#{$separator}front_right_wheel.c")
    wheel_transform = Transformation.new
    wheel_transform.set_rotation(Y_AXIS, -@wheel_turn_angle)
    wheel_transform.translate(front_wheel_x,
                              ZERO_M,
                              right_wheel_z)
    front_right_wheel << SolidMember.new(SolidOperand.new(@wheel.assembly(), wheel_transform))

    @assembly << front_right_wheel
  end

  def render_windows
    window = BusWindow.new(self, "window")

    windshield_shell = SolidCombination.new("#{full_name()}#{$separator}windshield_shell.c")
    windshield_shell << @body.front_solid()
    transform = Transformation.new
    transform.set_scale((@length - window.glass_thickness() * 2.0) / @length,
                        (@height - window.glass_thickness() * 2.0) / @height,
                        (@width - window.glass_thickness() * 2.0) / @width)
    transform.translate(window.glass_thickness(),
                        window.glass_thickness(),
                        window.glass_thickness())
    windshield_shell << SolidMember.new(SolidOperand.new(@body.front_solid(), transform),
                                        SolidOperator::Difference_op)
    windshield_shell << SolidMember.new(@body.windshield_box(), SolidOperator::Intersection_op)
    windshield = Combination.new($glass_attributes, "#{full_name()}#{$separator}windshield.r")
    windshield << windshield_shell

    @assembly << windshield

    #       the rest of the windows
    rear_window_shell = SolidCombination.new("#{full_name()}#{$separator}rear_window_shell.c")
    rear_window_shell << @body.rear_solid()
    transform = Transformation.new
    transform.set_scale((@length - window.glass_thickness() * 2.0) / @length,
                        (@height - window.glass_thickness() * 2.0) / @height,
                        (@width - window.glass_thickness() * 2.0) / @width)
    transform.translate(window.glass_thickness(),
                        window.glass_thickness(),
                        window.glass_thickness())
    rear_window_shell << SolidMember.new(SolidOperand.new(@body.rear_solid(), transform),
                                         SolidOperator::Difference_op)
    rear_window_shell << SolidMember.new(@body.rear_window_boxes(), SolidOperator::Intersection_op)

    rear_windows = Combination.new($glass_attributes, "#{full_name()}#{$separator}rear_windows.r")
    rear_windows << rear_window_shell

    @assembly << rear_windows
  end

  def render_bumpers
    @bumper.render

    rear_bumper = Combination.new($trim_attributes, "#{full_name()}#{$separator}rear_bumper.r")
    bumper_transform = Transformation.new
    bumper_transform.set_translation(- @bumper.body_gap() + @bumper.flattening_offset(),
                                     @bumper.radius() + @bumper.body_gap(),
                                     - @bumper.body_gap() + @bumper.flattening_offset())
    rear_bumper << SolidMember.new(SolidOperand.new(@bumper.part(), bumper_transform))

    @assembly << rear_bumper

    front_bumper = Combination.new($trim_attributes, "#{full_name()}#{$separator}front_bumper.r")
    bumper_transform = Transformation.new
    bumper_transform.set_rotation(Z_AXIS, Angle::degree * 180.0)
    bumper_transform.translate(@length + @bumper.body_gap() - @bumper.flattening_offset(),
                               @bumper.radius() + @bumper.body_gap() - @bumper.flattening_offset(),
                               -@bumper.body_gap() + @bumper.flattening_offset())
    front_bumper << SolidMember.new(SolidOperand.new(@bumper.part(), bumper_transform))

    @assembly << front_bumper
  end

  def render_headlights
    @headlights = BusHeadlights.new(self, "headlights")
    @headlights.render
    @assembly << @headlights.assembly()
  end

  def render
    # create parts
    @body = BusBody.new(self, "body")
    #   prototypes
    @wheel = BusWheel.new(self, "wheel")
    @bumper = BusBumper.new(self, "bumper")

    # calculate dimensions
    @top_length = @length - M * 0.305
    #   update part dimensions
    @wheel.calculate_dimensions()
    @body.calculate_dimensions()
    @bumper.calculate_dimensions()
    #   update dependent dimensions
    @rear_wheel_x = @rear_overhang + @wheel.height() / 2.0 + wheel_to_body_clearance

    # clear out any existing parts
    solid_list = @database.top_solids
    solid_list.clear()

    # make a top level bounding box for the body as a reference
    body_box = AxisAlignedBox.new(ORIGIN,
                                  Point.new(@length,
                                            @height,
                                            @width),
                                  "#{$separator}body_box.s")
    solid_list << body_box

    @assembly = Combination.new("#{full_name()}.a")
    solid_list << @assembly

    # render the parts
    # (these are order dependent)
    render_body
    render_wheels
    render_windows
    render_headlights
    render_bumpers
    # TODO: antenna
    # TODO: doors and rear lid
    # TODO: tail lights

    # create a light
    light_shape = Sphere.new(ORIGIN,
                             M * 0.10,
                             "#{$separator}light.s")
    light = Combination.new($light_attributes, "#{$separator}light.r")
    #   adjust the direction of the light (points in the -Z direction by default) by rotating
    light_transform = Transformation.new
    light_transform.set_rotation(X_AXIS, Angle::degree * -90.0)
    light_transform.translate(Vector.new(@length / 2.0,
                                         @height + M * 4.0,
                                         @width / 2.0))
    light << SolidMember.new(SolidOperand.new(light_shape, light_transform))

    solid_list << light
  end
end

#
# "main" function
#

#  render the bus
bus = Bus.new(nil, "bus")
bus.render

database_file = "bus.g"
error = ErrorParam.new
success = bus.database().write(database_file,
                               true,
                               error)

if (not success)
  puts("Could not write #{database_file}:")
  puts(error.base.to_s)
else
  #
  # edit or raytrace model
  #
  if (raytrace)
    image_height = 4096
    image_width = 4096
    background = "#{background_color.red}/#{background_color.green}/#{background_color.blue}"

    # TODO: specify parameters in Ruby and generate string
    # raytrace
    #   options to "rt":
    #     -A [ambient light fraction (0.0 - 1.0)]
    #     -J [jitter (1 = spatial)]
    #     -R [do not report overlaps]
    #     -C [background color (R/G/B)]
    #     -w [width in pixels of rendering]
    #     -n [height in pixels (number of lines) of rendering]
    #     -M [read model/view info from stdin]
    #     <first positional arg> [database]
    #     <second positional arg> [entity in database to trace]
    #     (model/view info can be saved in mged with the "saveview" command)

    command = "rm -f #{database_file}.pix #{database_file}.png ; \
              rt -A0.5 -J1 -R -C#{background} -o #{database_file}.pix \
                -w#{image_width} -n#{image_height} -M #{database_file} \
                #{bus.assembly().name()} #{$separator}light.r <<EOF\n\
                viewsize 9540;\n\
                orientation -0.119 0.331 0.00625 0.936;\n\
                eye_pt 12070 4500 12900;\n\
                start 0; clean; end;\n\EOF"
    result = %x[#{command}]

    if (result)
      # convert image and cleanup
      command = "pix-png -w#{image_width} -n#{image_height} -o #{database_file}.png #{database_file}.pix ; \
              rm -f #{database_file}.pix"
      %x[#{command}]
    end
  else
    # load the interactive editor
    command = "mged #{database_file}"
    %x[#{command}]
  end
end
