#!/usr/bin/ruby
#
# light.rb
#
# Copyright 2012-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Command line argument processing first
#

mode = ARGV[0]

if mode == 'edit'
  raytrace = false
elsif mode == 'render' or mode == nil
  raytrace = true
else
  abort "error, if an argument is supplied, it must be edit or render"
end

# These load the extension libraries
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'
require 'libtovero_graphics_rb_1'

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

ZERO = Unitless.new(0.0)
M = Distance::meter * 1.0
ZERO_M = Distance::meter * 0.0
ORIGIN = Point.new(ZERO_M,
                   ZERO_M,
                   ZERO_M)
X_AXIS = UnitVector.new(Unitless.new(1.0),
                        ZERO,
                        ZERO)
background_color = Color.new(0, 0, 50, 0)

scene = Combination.new("scene.a")

box = AxisAlignedBox.new(Point.new(-M, -M, -M),
                         Point.new(M, M, M),
                         "box.s")
box_transform = Transformation.new
box_transform.set_translation(ZERO_M,
                              M * 3.0,
                              ZERO_M)
scene << SolidMember.new(SolidOperand.new(box, box_transform))
#
box_transform = Transformation.new
box_transform.set_translation(ZERO_M,
                              M * -3.0,
                              ZERO_M)
scene << SolidMember.new(SolidOperand.new(box, box_transform))
#
box_transform = Transformation.new
box_transform.set_translation(M * 3.0,
                              ZERO_M,
                              ZERO_M)
scene << SolidMember.new(SolidOperand.new(box, box_transform))
#
box_transform = Transformation.new
box_transform.set_translation(M * -3.0,
                              ZERO_M,
                              ZERO_M)
scene << SolidMember.new(SolidOperand.new(box, box_transform))
#
box_transform = Transformation.new
box_transform.set_translation(ZERO_M,
                              ZERO_M,
                              M * 3.0)
scene << SolidMember.new(SolidOperand.new(box, box_transform))
#
box_transform = Transformation.new
box_transform.set_translation(ZERO_M,
                              ZERO_M,
                              M * -3.0)
scene << SolidMember.new(SolidOperand.new(box, box_transform))

light_sphere = Sphere.new(ORIGIN,
                          M * 0.25,
                          "light.s")
light_color = Color.new(255, 255, 255, 0)
light_shader = LightShader.new
light_shader.beam_dispersion_angle = Angle::degree * 20.0
light_attributes = CombinationAttributes.new
light_attributes.shader = light_shader
light_attributes.color = light_color
light_attributes.is_part = true
light = Combination.new(light_attributes, "light.r")
light_transform = Transformation.new
light_transform.set_rotation(X_AXIS, Angle::degree * -90.0)
light << SolidMember.new(SolidOperand.new(light_sphere, light_transform))

database = BCDatabase.new
solid_list = database.top_solids
solid_list << scene
solid_list << light

database_file = "light.g"
error = ErrorParam.new
success = database.write(database_file, true, error)
if (not success)
  puts("Could not write #{database_file}:")
  puts(error.base.to_s)
else
  if (raytrace)
    image_height = 512
    image_width = 512
    background = "#{background_color.red}/#{background_color.green}/#{background_color.blue}"

    # raytrace
    #   options to "rt":
    #     -A [ambient light fraction (0.0 - 1.0)]
    #     -J [jitter (1 = spatial)]
    #     -R [do not report overlaps]
    #     -C [background color (R/G/B)]
    #     -o [output .pix file]
    #     -w [width in pixels of rendering]
    #     -n [height in pixels (number of lines) of rendering]
    #     -M [read model/view info from stdin]
    #     <first positional arg> [database]
    #     <second positional arg> [entity in database to trace]
    #     (model/view info can be saved in mged with the "saveview" command)
    command = "rm -f #{database_file}.pix #{database_file}.png ; \
              rt -A0.5 -J1 -R -C#{background} -o #{database_file}.pix \
                -w#{image_width} -n#{image_height} -M #{database_file} \
                scene.a light.r <<EOF\n\
                viewsize 9540;\n\
                orientation -0.119 0.331 0.00625 0.936;\n\
                eye_pt 12070 4500 12900;\n\
                start 0; clean; end;\n\
                EOF"
    result = %x[#{command}]

    if (result)
      # convert image and cleanup
      command = "pix-png -w#{image_width} -n#{image_height} -o #{database_file}.png #{database_file}.pix ; \
              rm -f #{database_file}.pix"
      %x[#{command}]
    end
  else
    # load the interactive editor
    command = "mged #{database_file}"
    %x[#{command}]
  end
end
