#!/usr/bin/ruby
#
# pipe.rb
#
# Copyright 2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Command line argument processing first
#

mode = ARGV[0]

if mode == 'edit'
  raytrace = false
elsif mode == 'render' or mode == nil
  raytrace = true
else
  abort "error, if an argument is supplied, it must be edit or render"
end

# These load the extension libraries
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'
require 'libtovero_graphics_rb_1'

# These are like "using namespace" in C++
include Libtovero_support_rb_1
include Libtovero_math_rb_1
include Libtovero_graphics_rb_1

m = Distance::meter
background_color = Color.new(224, 212, 190, 0)

control_points = PipeControlPointArray.new
control_points << PipeControlPoint.new(Point.new,
                                       m * 0.1,
                                       m * 0.15,
                                       m * 0.20);
control_points << PipeControlPoint.new(Point.new(m * 0.5,
                                                 m * 0.5,
                                                 m * 0.0),
                                       m * 0.1,
                                       m * 0.15,
                                       m * 0.20);
control_points << PipeControlPoint.new(Point.new(m * 0.5,
                                                 m * 0.0,
                                                 m * 0.0),
                                       m * 0.1,
                                       m * 0.15,
                                       m * 0.20);
# Note: consistently setting the name of graphical entities makes
# it easier to work with the model in BRL-CAD, if necessary
pipe = Pipe.new(control_points, "pipe.s")

main_shader = PhongShader::plastic
main_shader.shine = 5
main_color = Color.new(86, 59, 39, 0)
main_attributes = CombinationAttributes.new
main_attributes.shader = main_shader
main_attributes.color = main_color
main_attributes.is_part = true

main_combo = Combination.new(main_attributes, "main_combo.r")
main_combo << pipe

BC_DB = BCDatabase.new
solid_list = BC_DB.top_solids
solid_list << main_combo

database_file = "pipe.g"
error = ErrorParam.new
success = BC_DB.write(database_file, true, error)
if (not success)
  puts("Could not write #{database_file}:")
  puts error.base.to_s
else
  if raytrace
    image_height = 4096
    image_width = 4096
    background = "#{background_color.red}/#{background_color.green}/#{background_color.blue}"

    # raytrace
    #   options to "rt":
    #     -A [ambient light fraction (0.0 - 1.0)]
    #     -J [jitter (1 = spatial)]
    #     -R [do not report overlaps]
    #     -C [background color (R/G/B)]
    #     -o [output .pix file]
    #     -w [width in pixels of rendering]
    #     -n [height in pixels (number of lines) of rendering]
    #     -M [read model/view info from stdin]
    #     <first positional arg> [database]
    #     <second positional arg> [entity in database to trace]
    #     (model/view info can be saved in mged with the "saveview" command)

    command = "rm -f #{database_file}.pix #{database_file}.png ; \
              rt -A0.5 -J1 -R -C#{background} -o #{database_file}.pix \
              -w#{image_width} -n#{image_height} #{database_file} main_combo.r -M <<EOF\n\
              viewsize 4100;\n\
              orientation -0.0912 0.0935 0.00465 0.991;\n\
              eye_pt 1550 1930 7997;\n\
              start 0; clean; end;\n\
              EOF"
    result = %x[#{command}]

    if (result)
      # convert image and cleanup
      command = "pix-png -w#{image_width} -n#{image_height} -o #{database_file}.png #{database_file}.pix ; \
              rm -f #{database_file}.pix"
      %x[#{command}]
    end
  else
    # [or] load the interactive editor
    command = "mged #{database_file}"
    %x[#{command}]
  end
end
