// Area.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#ifndef TOVERO_MATH_AREA_HPP_
#define TOVERO_MATH_AREA_HPP_

#include <tovero/math/geometry/Quantity.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <cmath>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Area : public Quantity
    {
    public:
      // constructors
      Area() : Quantity(0.0) {}
      //   meant for creating an area from a value and a unit
      Area(double value, const Area& unit) : Quantity(value * unit.value()) {}
      static inline Area from_value(double value);
      //  Note: use compiler generated copy constructor and destructor
      // operators
      inline Area& operator=(const Area& q);
      Area operator+(const Area& q) const
      { return Area::from_value(value() + q.value()); }
      Area operator-(const Area& q) const
      { return Area::from_value(value() - q.value()); }
      Area operator-() const { return Area::from_value(-value()); }
      Area operator*(const Unitless& factor) const { return Area::from_value(factor.value() * value()); }
      Area operator*(double factor) const { return Area::from_value(factor * value()); }
      Unitless operator/(const Area& divisor) const { return Unitless::from_value(value() / divisor.value()); }
      Area operator/(const Unitless& divisor) const { return Area::from_value(value() / divisor.value()); }
      Area operator/(double divisor) const { return Area::from_value(value() / divisor); }
      inline Area& operator+=(const Area& addend);
      inline Area& operator-=(const Area& subtrahend);
      inline Area& operator*=(const Unitless& factor);
      inline Area& operator*=(double factor);
      inline Area& operator/=(const Unitless& divisor);
      inline Area& operator/=(double divisor);

      // class constants
      //   units
      //   SI
      static const Area square_meter;
      static const Area square_millimeter;
      //   US
      static const Area square_foot;
      static const Area square_inch;
    };

    // free functions
    inline Area operator*(const Distance& d1, const Distance& d2)
    { return Area::from_value(d1.value() * d2.value()); }

    inline Area operator*(const Unitless& factor, const Area& a)
    { return Area::from_value(factor.value() * a.value()); }

    inline Area operator*(double factor, const Area& a)
    { return Area::from_value(factor * a.value()); }

    inline Distance sqrt(const Area& a)
    { return Distance::from_value(std::sqrt(a.value())); }

    //
    // Inline definitions
    //

    //
    // Constructors
    //

    inline Area Area::from_value(double value)
    {
      Area a;
      a.set_value(value);

      return a;
    }

    //
    //   Operators
    //

    inline Area& Area::operator=(const Area& d)
    {
      if (this != &d)
      {
        set_value(d.value());
      }

      return *this;
    }

    inline Area& Area::operator+=(const Area& addend)
    {
      set_value(value() + addend.value());

      return *this;
    }

    inline Area& Area::operator-=(const Area& subtrahend)
    {
      set_value(value() - subtrahend.value());

      return *this;
    }

    inline Area& Area::operator*=(const Unitless& factor)
    {
      set_value(value() * factor.value());

      return *this;
    }

    inline Area& Area::operator*=(double factor)
    {
      set_value(value() * factor);

      return *this;
    }

    inline Area& Area::operator/=(const Unitless& divisor)
    {
      set_value(value() / divisor.value());

      return *this;
    }

    inline Area& Area::operator/=(double divisor)
    {
      set_value(value() / divisor);

      return *this;
    }
  }
}

#endif // TOVERO_MATH_AREA_HPP_
