// Axis_aligned_box.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Axis_aligned_box.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>

using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor/copy
//

Axis_aligned_box::Axis_aligned_box(const string& name)
  : Polyhedron_base(name),
    m_minimum(Point(Distance(-1.0 * Distance::meter),
                    Distance(-1.0 * Distance::meter),
                    Distance(-1.0 * Distance::meter))),
    m_maximum(Point(Distance(1.0 * Distance::meter),
                    Distance(1.0 * Distance::meter),
                    Distance(1.0 * Distance::meter)))
{
}

Axis_aligned_box::Axis_aligned_box(const Point& minimum,
                                   const Point& maximum,
                                   const string& name)
  : Polyhedron_base(name),
    m_minimum(minimum),
    m_maximum(maximum)
{
}

//
// Predicates
//

bool Axis_aligned_box::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for invalid minimum/maximum
  Vector v = m_maximum - m_minimum;
  if (tolerances.is_distance_near_zero(v[0]),
      tolerances.is_distance_near_zero(v[1]),
      tolerances.is_distance_near_zero(v[2])
      || (v[0] < Distance(0.0 * Distance::meter))
      || (v[1] < Distance(0.0 * Distance::meter))
      || (v[2] < Distance(0.0 * Distance::meter)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": minimum >= maximum for at least one dimension";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

Polyhedron& Axis_aligned_box::generalize() const
{
  vector<Point> vertices;
  vertices.push_back(Point(m_maximum[0], m_minimum[1], m_minimum[2])); // vertex 0
  vertices.push_back(Point(m_maximum[0], m_maximum[1], m_minimum[2])); // vertex 1
  vertices.push_back(Point(m_maximum[0], m_maximum[1], m_maximum[2])); // vertex 2
  vertices.push_back(Point(m_maximum[0], m_minimum[1], m_maximum[2])); // vertex 3
  vertices.push_back(Point(m_minimum[0], m_minimum[1], m_minimum[2])); // vertex 4
  vertices.push_back(Point(m_minimum[0], m_maximum[1], m_minimum[2])); // vertex 5
  vertices.push_back(Point(m_minimum[0], m_maximum[1], m_maximum[2])); // vertex 6
  vertices.push_back(Point(m_minimum[0], m_minimum[1], m_maximum[2])); // vertex 7

  Polyhedron* polyhedron = new Polyhedron(vertices, name());

  postcondition(polyhedron && (8 == polyhedron->vertices().size()));
  return *polyhedron;
}
