// Box.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Box.hpp>
#include <tovero/math/geometry/Axis_aligned_box.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/base/math.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>

using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using Roan_trail::Tovero_math::max;
using Roan_trail::Tovero_math::min;
using namespace Roan_trail::Tovero_math;

//
// Constructor/copy
//

Box::Box(const string& name)
  : Polyhedron_base(name),
    m_vertex(Point(Distance(-1.0 * Distance::meter),
                   Distance(-1.0 * Distance::meter),
                   Distance(-1.0 * Distance::meter))),
    m_width(Unit_vector::x * (2.0 * Distance::meter)),
    m_height(Unit_vector::y * (2.0 * Distance::meter)),
    m_depth(Unit_vector::z * (2.0 * Distance::meter))
{
}

Box::Box(const Point& vertex,
         const Vector& width,
         const Vector& height,
         const Vector& depth,
         const string& name)
  : Polyhedron_base(name),
    m_vertex(vertex),
    m_width(width),
    m_height(height),
    m_depth(depth)
{
}

//
// Predicates
//

bool Box::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for non-zero direction vector lengths
  const Distance height_length = m_height.length();
  const Distance width_length = m_width.length();
  const Distance depth_length = m_depth.length();
  if (tolerances.is_distance_near_zero(height_length)
      || tolerances.is_distance_near_zero(width_length)
      || tolerances.is_distance_near_zero(depth_length))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or more direction vectors is zero length";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for mutually perpendicular direction vectors
  Unit_vector unit_width;
  m_width.normalize(unit_width);
  Unit_vector unit_depth;
  m_depth.normalize(unit_depth);
  Unit_vector unit_height;
  m_height.normalize(unit_height);
  const Unitless height_dot_width = unit_height.dot(unit_width);
  const Unitless height_dot_depth = unit_height.dot(unit_depth);
  const Unitless width_dot_depth = unit_width.dot(unit_depth);
  if (!tolerances.is_cosine_near_zero(height_dot_width)
      || !tolerances.is_cosine_near_zero(height_dot_depth)
      || !tolerances.is_cosine_near_zero(width_dot_depth))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the direction vectors are not mutually perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

Polyhedron& Box::generalize() const
{
  Unit_vector unit_width;
  m_width.normalize(unit_width);

  vector<Point> vertices;
  vertices.push_back(m_vertex);               // vertex 0
  vertices.push_back(m_vertex + m_width);     // vertex 1
  vertices.push_back(vertices[1] + m_height); // vertex 2
  vertices.push_back(m_vertex + m_height);    // vertex 3
  vertices.push_back(m_vertex + m_depth);     // vertex 4
  vertices.push_back(vertices[4] + m_width);  // vertex 5
  vertices.push_back(vertices[5] + m_height); // vertex 6
  vertices.push_back(vertices[4] + m_height); // vertex 7

  Polyhedron* polyhedron = new Polyhedron(vertices, name());

  postcondition(polyhedron && (8 == polyhedron->vertices().size()));
  return *polyhedron;
}

Solid& Box::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>((this)));

  Unit_vector unit_width;
  m_width.normalize(unit_width);

  // check for alignment with the x, y, z axes
  //   (since the width, height, and depth are mutually
  //   perpendicular, only need to check width being
  //   perpendicular to one of the x, y, z axes)
  if (tolerances.is_cosine_near_zero(unit_width.dot(Unit_vector::x))
      || tolerances.is_cosine_near_zero(unit_width.dot(Unit_vector::y))
      || tolerances.is_cosine_near_zero(unit_width.dot(Unit_vector::z)))
  {
    // axis aligned box
    Point P = m_vertex + m_width + m_height + m_depth;
    Point minimum = Point(min(m_vertex[0], P[0]),
                          min(m_vertex[1], P[1]),
                          min(m_vertex[2], P[2]));
    Point maximum = Point(max(m_vertex[0], P[0]),
                          max(m_vertex[1], P[1]),
                          max(m_vertex[2], P[2]));
    return_value = new Axis_aligned_box(minimum,
                                        maximum,
                                        name());
  }

  return *return_value;
}
