// Cylinder.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Cylinder::Cylinder(const string& name)
  : General_cone_base(name),
    m_base(Point::O),
    m_height(Unit_vector::z * Distance::meter),
    m_radius(Distance(1.0 * Distance::meter))
{
}

Cylinder::Cylinder(const Point& base,
                   const Vector& height,
                   const Distance& radius,
                   const string& name)
  : General_cone_base(name),
    m_base(base),
    m_height(height),
    m_radius(radius)
{
}

//
// Predicates
//

bool Cylinder::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for non-zero height vector
  if (tolerances.is_distance_near_zero(m_height.length()))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height vector " << m_height << " must be non-zero";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for positive radius
  if (tolerances.is_distance_near_zero(m_radius)
      || (m_radius < Distance(0.0 * Distance::meter)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " + solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the radius is not positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

General_cone& Cylinder::generalize() const
{
  Unit_vector unit_h;
  m_height.normalize(unit_h);
  const Unit_vector unit_a = unit_h.perpendicular();
  const Vector base_a = unit_a * m_radius;
  const Unit_vector unit_b = unit_a.cross(unit_h);
  const Vector base_b = unit_b * m_radius;

  General_cone* cone = new General_cone(m_base,
                                        m_height,
                                        base_a,
                                        base_b,
                                        base_a,
                                        base_b,
                                        name());
  postcondition(cone);
  return *cone;
}
