// General_cone.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Solid validation based on ./src/librt/primitives/tgc/tgc.c from
// BRL-CAD (version 7.20.4) source:
//
//   Copyright (c) 1985-2011 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Elliptical_cone.hpp>
#include <tovero/math/geometry/Elliptical_cylinder.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Oblique_cone.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

General_cone::General_cone(const string& name)
  : General_cone_base(name),
    m_base(Point::O),
    m_height(Unit_vector::z * Distance::meter),
    m_base_a(Unit_vector::x * Distance::meter),
    m_base_b(Unit_vector::y * Distance::meter),
    m_top_c(Unit_vector::x * Distance::meter),
    m_top_d(Unit_vector::y * Distance::meter)
{
}

General_cone::General_cone(const Point& base,
                           const Vector& height,
                           const Vector& base_a,
                           const Vector& base_b,
                           const Vector& top_c,
                           const Vector& top_d,
                           const string& name)
  : General_cone_base(name),
    m_base(base),
    m_height(height),
    m_base_a(base_a),
    m_base_b(base_b),
    m_top_c(top_c),
    m_top_d(top_d)
{
}

//
// Predicates
//

bool General_cone::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check non-zero height vector
  if (tolerances.is_distance_near_zero(m_height.length()))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height vector must be non-zero";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  Unit_vector unit_a;
  const Distance base_a_length = m_base_a.normalize(unit_a);
  Unit_vector unit_b;
  const Distance base_b_length = m_base_b.normalize(unit_b);
  Unit_vector unit_c;
  const Distance top_c_length = m_top_c.normalize(unit_c);
  Unit_vector unit_d;
  const Distance top_d_length = m_top_d.normalize(unit_d);

  // check for non-zero base and top vectors
  if (tolerances.is_distance_near_zero(base_a_length)
      || tolerances.is_distance_near_zero(base_b_length)
      || tolerances.is_distance_near_zero(top_c_length)
      || tolerances.is_distance_near_zero(top_d_length))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or more of the base and top vectors are zero";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that both ends aren't degenerate (i.e. < 2D)
  const bool base_is_degenerate = tolerances.is_area_near_zero(base_a_length * base_b_length);
  const bool top_is_degenerate = tolerances.is_area_near_zero(top_c_length * top_d_length);
  if (base_is_degenerate && top_is_degenerate)
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": both ends are degenerate (< 2D)";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check to see if the height lies in the base_a, base_b plane
  Unit_vector a_cross_b = unit_a.cross(unit_b);
  Unit_vector unit_h;
  m_height.normalize(unit_h);
  if (tolerances.is_cosine_near_zero(unit_h.dot(a_cross_b)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height vector lies in the base a, base b plane";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that base vectors are perpendicular
  if (!base_is_degenerate && !tolerances.is_cosine_near_zero(unit_a.dot(unit_b)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the base vectors are not perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that top vectors are perpendicular
  if (!top_is_degenerate && !tolerances.is_cosine_near_zero(unit_c.dot(unit_d)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the top vectors are not perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that base a and top c are parallel
  if (!tolerances.is_area_near_zero(base_a_length * top_c_length)
      && !tolerances.is_cosine_near_zero(1.0 - unit_a.dot(unit_c)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": base a and top c are not parallel";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check that base b and top d are parallel
  if (!tolerances.is_area_near_zero(base_b_length * top_d_length)
      && !tolerances.is_cosine_near_zero(1.0 - unit_b.dot(unit_d)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": base a and top c are not parallel";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

// Note: Tovero current only supports a subset of the possible cone types
//       as specialized classes.  Use the General_cone class for unsupported types.

Solid& General_cone::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>(this));

  Unit_vector unit_a;
  const Distance base_a_length = m_base_a.normalize(unit_a);
  const Distance base_b_length = m_base_b.length();
  const Unitless ratio_ac = base_a_length / m_top_c.length();
  const Unitless ratio_bd = base_b_length / m_top_d.length();
  // check to see if ratios of corresponding (a/c and b/d) base vectors are the same
  // (e.g. that we have "similar" base and top shapes)
  if (tolerances.is_unitless_near_zero(ratio_ac - ratio_bd))
  {
    // check to see if the base and top are circular
    if (tolerances.is_distance_near_zero(base_a_length - base_b_length))
    {
      // oblique circular cone
      Oblique_cone* cone = new Oblique_cone(m_base,
                                            m_height,
                                            m_base_a,
                                            ratio_ac,
                                            name());
      return_value = &cone->specialize(tolerances);
      if (return_value != cone)
      {
        cone->reference();
        cone->unreference();
      }
    }
    else
    {
      Unit_vector unit_h;
      const Distance height = m_height.normalize(unit_h);
      // check to see if the height is perpendicular
      //   Note: again, some specialized cone types are missing here
      if (tolerances.is_cosine_near_zero(unit_a.dot(unit_h)))
      {
        // right elliptical cone
        Elliptical_cone* cone = new Elliptical_cone(m_base,
                                                    m_base_a,
                                                    m_base_b,
                                                    height,
                                                    ratio_ac,
                                                    name());
        return_value = &cone->specialize(tolerances);
        if (return_value != cone)
        {
          cone->reference();
          cone->unreference();
        }
      }
    }
  }

  return *return_value;
}
