// Pipe.cpp
//
// Copyright 2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Solid validation based on ./src/librt/primitives/pipe/pipe.c from
// BRL-CAD (version 7.22.0) source:
//
//   Copyright (c) 1990-2012 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/math/geometry/Pipe.hpp>
#include <tovero/math/geometry/Angle.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Pipe_control_point.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/base/math.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>
#include <cmath>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using Roan_trail::Tovero_math::is_quantity_near_zero;
using namespace Roan_trail::Tovero_math;

//
// Internal helpers
//

namespace
{
  //TO DO: move this validity check into the Pipe_control_point class
  bool ih_check_control_point(Pipe_control_point control_point,
                              const Geometric_tolerances& tolerances,
                              string& return_diagnostic)
  {
    bool return_value = false;

    start_error_block();
    static_cast<void>(handler_error); // prevent unused warning

    // check for outer_radius not bigger than inner_radius
    // (Note: this is not in pipe.c, but BRL-CAD's documentation says
    // the check below is a requirement for a valid pipe.)
    if (control_point.outer_radius() <= control_point.inner_radius())
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid pipe control point: outer radius <= inner radius";
      return_diagnostic = diagnostic_stream.str();
      goto exit_point;
    }
    end_error_block();

    return_value = true;
    goto exit_point;

  exit_point:
    return return_value;
  }

  bool ih_check_linear_segment(const Pipe_control_point& c_point1,
                               const Pipe_control_point& c_point2,
                               const Geometric_tolerances& tolerances,
                               string& return_diagnostic)
  {
    bool return_value = false;

    start_error_block();
    static_cast<void>(handler_error); // prevent unused warning

    // there are currently no checks in pipe.c, so do nothing for now

    end_error_block();

    return_value = true;
    goto exit_point;

  exit_point:
    return return_value;
  }

  bool ih_check_bend_segment(const Point& bend_center,
                             const Point& bend_start,
                             const Point& bend_end,
                             const Pipe_control_point& bend_control_point,
                             const Angle& bend_angle,
                             const Distance& prev_outer_radius,
                             const Distance& next_outer_radius,
                             const Geometric_tolerances& tolerances,
                             string& return_diagnostic)
  {
    bool return_value = false;

    start_error_block();
    static_cast<void>(handler_error); // prevent unused warning

    // check for too small bend radius
    // (Note: this is not in pipe.c but BRL-CAD's documentation says
    // the check below is a requirement for a valid pipe.)
    if (bend_control_point.bend_radius() <= bend_control_point.outer_radius())
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid pipe control point at bend: bend radius <= outer radius";
      return_diagnostic = diagnostic_stream.str();
      goto exit_point;
    }

    // The check below comes from BRL-CAD's pipe.c function rt_bend_pipe_prep.
    // check for too large bend angle
    if (bend_angle >= Angle::radian * M_PI)
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid bend pipe segment: bend angle";
      diagnostic_stream << " is >= " << (Angle::radian * M_PI / Angle::radian).value() << " radians.";
      return_diagnostic = diagnostic_stream.str();
      goto exit_point;
    }

    end_error_block();

    return_value = true;
    goto exit_point;

  exit_point:
    return return_value;
  }
}

//
// Constructors
//

Pipe::Pipe(const vector<Pipe_control_point>& control_points,
           const string& name)
  : Solid(name),
    m_control_points(control_points)
{
}

//
// Predicates
//

bool Pipe::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check each control point for validity
  const size_t point_count = m_control_points.size();
  for (size_t i = 0; i < point_count; ++i)
  {
    const Pipe_control_point& control_point = m_control_points[i];
    string control_point_check_diagnostic;
    const int accepted = ih_check_control_point(control_point,
                                                tolerances,
                                                control_point_check_diagnostic);
    if (!accepted)
    {
      stringstream diagnostic_stream;
      diagnostic_stream << control_point_check_diagnostic;
      on_error(true, new Math_error(error_location(),
                                    Math_error::validation,
                                    diagnostic_stream.str()));
    }
  }

  // check validity of pipe segments
  //
  // c_point2 and c_point3 are always control points of the pipe, so are set in
  // the loop based on an index to the list.  c_point1 starts out as the first
  // control point of the pipe, but on future iterations varys: If previous
  // segment contained a bend, then c_point1 is the end of this bend; otherwise
  // (for previous segment linear) c_point1 is the next control point (i.e. what
  // c_point2 was).  So, c_point1 must be declared before the loop and set in
  // the loop to the value needed for the next iteration.
  Pipe_control_point c_point1;
  for (size_t i = 0; i < point_count; ++i)
  {
    if (i == (point_count - 1))
    {
      break;   // checks completed
    }

    if (i == 0)
    {
      c_point1 = m_control_points[i];
    }
    const Pipe_control_point c_point2 = m_control_points[i+1];

    const Vector pt2_to_pt1 = c_point1.position() - c_point2.position();
    if (tolerances.is_distance_near_zero(pt2_to_pt1.length()))
    {
      c_point1 = c_point2;
      continue;   // no checks needed, continue to next segment
    }

    if (i == (point_count - 2)) // last segment
    {
      string linear_segment_check_diagnostic;
      const int accepted = ih_check_linear_segment(c_point1,
                                                   c_point2,
                                                   tolerances,
                                                   linear_segment_check_diagnostic);
      if (!accepted)
      {
        stringstream diagnostic_stream;
        diagnostic_stream << "Invalid " << solid_class();
        const string& solid_name = name();
        diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
        diagnostic_stream << ": ";
        diagnostic_stream << linear_segment_check_diagnostic;
        on_error(true, new Math_error(error_location(),
                                      Math_error::validation,
                                      diagnostic_stream.str()));
      }
    }
    else // last segment NOT reached yet
    {
      const Pipe_control_point c_point3 = m_control_points[i+2];
      const Vector pt2_to_pt3 = c_point3.position() - c_point2.position();
      Unit_vector unit_pt2_to_pt1;
      Unit_vector unit_pt2_to_pt3;
      pt2_to_pt1.normalize(unit_pt2_to_pt1);
      pt2_to_pt3.normalize(unit_pt2_to_pt3);
      const Unit_vector norm = unit_pt2_to_pt1.cross(unit_pt2_to_pt3);
      const Angle angle = Angle::radian * (M_PI - acos(unit_pt2_to_pt1.dot(unit_pt2_to_pt3).value()));
      const Distance dist_to_bend = c_point2.bend_radius() * tan((angle / Angle::radian).value() /2.0);

      // In condition below, BRL-CAD compares norm.length() and dist_to_bend with SQRT_SMALL_FASTF
      if (isnan(dist_to_bend.value()) ||   // uses std lib macros to check if argument is "Not a Number"
          tolerances.is_cosine_near_zero(norm.length()) ||
          tolerances.is_distance_near_zero(dist_to_bend))
      {
        // points are colinear, only have linear segment
        string linear_segment_check_diagnostic;
        const int accepted = ih_check_linear_segment(c_point1,
                                                     c_point2,
                                                     tolerances,
                                                     linear_segment_check_diagnostic);
        if (!accepted)
        {
          stringstream diagnostic_stream;
          diagnostic_stream << "Invalid " << solid_class();
          const string& solid_name = name();
          diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
          diagnostic_stream << ": ";
          diagnostic_stream << linear_segment_check_diagnostic;
          on_error(true, new Math_error(error_location(),
                                        Math_error::validation,
                                        diagnostic_stream.str()));
        }
        // next check starting at c_point2
        c_point1 = c_point2;
      }
      else
      {
        // points are NOT colinear, have linear section and bend section
        Point bend_start = c_point2.position() + dist_to_bend * unit_pt2_to_pt1;
        const Point bend_end = c_point2.position() + dist_to_bend * unit_pt2_to_pt3;

        // linear section
        const Vector initial_linear_piece = c_point1.position() - bend_start;

        // In condition below, BRL-CAD compares with raytrace small length (RT_LEN_TOL)
        if (tolerances.is_distance_near_zero(initial_linear_piece.length()))
        {
          // do not make linear sections that are too small to raytrace
          bend_start = c_point1.position();
        }
        else
        {
          const Pipe_control_point end_linear_point(bend_start,
                                                    c_point2.inner_radius(),
                                                    c_point2.outer_radius(),
                                                    c_point2.bend_radius());
          string linear_segment_check_diagnostic;
          const int accepted = ih_check_linear_segment(c_point1,
                                                       end_linear_point,
                                                       tolerances,
                                                       linear_segment_check_diagnostic);
          if (!accepted)
          {
            stringstream diagnostic_stream;
            diagnostic_stream << "Invalid " << solid_class();
            const string& solid_name = name();
            diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
            diagnostic_stream << ": ";
            diagnostic_stream << linear_segment_check_diagnostic;
            on_error(true, new Math_error(error_location(),
                                          Math_error::validation,
                                          diagnostic_stream.str()));
          }
        }

        // bend section
        const Unit_vector unit_bend_dir = unit_pt2_to_pt1.cross(norm);
        const Point bend_center = bend_start - c_point2.bend_radius() * unit_bend_dir;
        string bend_segment_check_diagnostic;
        const int accepted = ih_check_bend_segment(bend_center,
                                                   bend_start,
                                                   bend_end,
                                                   c_point2,
                                                   angle,
                                                   c_point1.outer_radius(),
                                                   c_point3.outer_radius(),
                                                   tolerances,
                                                   bend_segment_check_diagnostic);
        if (!accepted)
        {
          stringstream diagnostic_stream;
          diagnostic_stream << "Invalid " << solid_class();
          const string& solid_name = name();
          diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
          diagnostic_stream << ": ";
          diagnostic_stream << bend_segment_check_diagnostic;
          on_error(true, new Math_error(error_location(),
                                        Math_error::validation,
                                        diagnostic_stream.str()));
        }

        // next check starting at bend_end
        c_point1 = Pipe_control_point(bend_end,
                                      c_point2.inner_radius(),
                                      c_point2.outer_radius(),
                                      c_point2.bend_radius());
      }
    }
  }

  return_value = true;
  goto exit_point;


  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
