// Plate_triangle_mesh.cpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Plate_triangle_mesh.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using Roan_trail::Tovero_math::is_quantity_near_zero;
using namespace Roan_trail::Tovero_math;

//
// Constructors/copy
//
Plate_triangle_mesh::Plate_triangle_mesh(Triangle_face::Triangle_direction_type triangle_direction,
                                         const vector<Point>& vertices,
                                         const vector<Triangle_face>& faces,
                                         const vector<Unit_vector>& normals,
                                         const vector<Triangle_face>& face_normals,
                                         bool has_surface_normals,
                                         bool use_surface_normals,
                                         const vector<Distance>& thicknesses,
                                         const vector<bool>& append_thicknesses,
                                         const string& name)
  : Triangle_mesh(triangle_direction,
                  vertices,
                  faces,
                  normals,
                  face_normals,
                  has_surface_normals,
                  use_surface_normals,
                  name),
    m_thicknesses(thicknesses),
    m_append_thicknesses(append_thicknesses)
{
}

//
// Predicates
//

bool Plate_triangle_mesh::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Error_param error;

  const bool base_is_valid = Triangle_mesh::is_valid(tolerances, error);
  on_error(!base_is_valid, new Math_error(error_location(),
                                          Math_error::validation,
                                          error()));

  const size_t face_count = faces().size();
  const size_t thickness_count = m_thicknesses.size();
  if (face_count != thickness_count)
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": thickness count " << thickness_count;
    diagnostic_stream << " is not equal to face count " << face_count;
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }
  const size_t append_thickness_count = m_append_thicknesses.size();
  if (thickness_count != append_thickness_count)
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": append thickness count " << append_thickness_count;
    diagnostic_stream << " is not equal to thickness count " << thickness_count;
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
