// Polyhedron.cpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Solid validation based on ./src/librt/primitives/arb8/arb8.c from
// BRL-CAD (version 7.20.4) source:
//
//   Copyright (c) 1985-2011 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Box.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/geometry/Wedge.hpp>
#include <tovero/math/base/math.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <string>
#include <sstream>
#include <vector>
#include <cstddef>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using Roan_trail::Tovero_math::is_quantity_near_zero;
using namespace Roan_trail::Tovero_math;

//
// Internal helpers
//

namespace
{
  struct Ih_face_layout_struct
  {
    const char* description;
    size_t vertex[6];
  };

  // vertices listed in clockwise order
  const struct Ih_face_layout_struct ic_face_layout[6] =
    {
      { "1234", {3, 2, 1, 0} },
      { "8765", {4, 5, 6, 7} },
      { "1485", {4, 7, 3, 0} },
      { "2673", {2, 6, 5, 1} },
      { "1562", {1, 5, 4, 0} },
      { "4378", {7, 6, 2, 3} }
    };

  bool ih_check_face(const vector<Point>& vertices,
                     const string& face_description,
                     const Geometric_tolerances& tolerances,
                     string& return_diagnostic)
  {
    precondition(((vertices.size() == 3) || (vertices.size() == 4))
                 && (face_description.size() == 4));

    bool return_value = false;

    start_error_block();
    static_cast<void>(handler_error); // prevent unused warning

    Vector u = vertices[1] - vertices[0];
    Unit_vector unit_u;
    const Distance u_length = u.normalize(unit_u);
    if (tolerances.is_distance_near_zero(u_length))
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "In face " << face_description << ", ";
      diagnostic_stream << "vertex " << face_description[0];
      diagnostic_stream << " and vertex " << face_description[1];
      diagnostic_stream << " are equivalent.";
      return_diagnostic = diagnostic_stream.str();
      goto exit_point;
    }
    Vector v = vertices[2] - vertices[0];
    Unit_vector unit_v;
    v.normalize(unit_v);
    Unit_vector v_cross_u = unit_v.cross(unit_u);
    const Unitless v_cross_u_length = v_cross_u.length();

    // check for co-linear face vertices
    //
    // (Note: BRL-CAD uses a "sloppy dot tolerance" for validation in
    // arb8.c presumably because of inconsistent use of units
    // (i.e. crosses a vector by a unit vector).  Here we check
    // against the regular cosine tolerance because the cross was
    // between unit vectors and thus have a normalized value.)
    if (tolerances.is_cosine_near_zero(v_cross_u_length))
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "In face " << face_description << ", ";
      diagnostic_stream << "vertex " << face_description[0];
      diagnostic_stream << ", vertex " << face_description[1];
      diagnostic_stream << ", and vertex " << face_description[2];
      diagnostic_stream << " are co-linear.";
      return_diagnostic = diagnostic_stream.str();
      goto exit_point;
    }

    if (vertices.size() == 4)
    {
      Vector w = vertices[3] - vertices[0];
      Unit_vector unit_w;
      w.normalize(unit_w);
      const Unitless dot_product = v_cross_u.dot(unit_w);

      // check for non-planar face
      //
      // (Note: again BRL-CAD is using the "sloppy dot tolerance"
      // constant in arb8.c, but it is changed here to just check the
      // cosine tolerance.)
      if (!tolerances.is_cosine_near_zero(dot_product))
      {
        stringstream diagnostic_stream;
        diagnostic_stream << "In face " << face_description << ", ";
        diagnostic_stream << "non-planar face formed by vertex " << face_description[3];
        return_diagnostic = diagnostic_stream.str();
        goto exit_point;
      }
    }

    end_error_block();

    return_value = true;
    goto exit_point;

  exit_point:
    return return_value;
  }
}

//
// Constructors
//

Polyhedron::Polyhedron(const string& name)
  : Polyhedron_base(name)
{
  m_vertices.push_back(Point(Distance(-1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance( 1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance( 1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance(-1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance(-1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance( 1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter),
                             Distance(-1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance( 1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter)));
  m_vertices.push_back(Point(Distance(-1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter),
                             Distance( 1.0 * Distance::meter)));
}

Polyhedron::Polyhedron(const vector<Point>& vertices,
                       const string& name)
  : Polyhedron_base(name),
    m_vertices(vertices)
{
}

//
// Predicates
//

bool Polyhedron::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  if ((m_vertices.size() < 4) || (m_vertices.size() > 8))
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": wrong number of vertices (must be >=4 and <=8)";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // setup a copy of the vertices for validation
  vector<Point> vertices = m_vertices;
  vertices.reserve(8);
  //   rearrange vertices for validation
  switch (m_vertices.size())
  {
  case 4:
    vertices[7] = vertices[3];
    vertices[6] = vertices[3];
    vertices[5] = vertices[3];
    vertices[4] = vertices[3];
    vertices[3] = vertices[0];
    break;
  case 5:
    vertices[7] = vertices[4];
    vertices[6] = vertices[4];
    vertices[5] = vertices[4];
    break;
  case 6:
    vertices[7] = vertices[5];
    vertices[6] = vertices[5];
    vertices[5] = vertices[4];
    break;
  case 7:
    vertices[7] = vertices[4];
    break;
  case 8:
    break;
  default:
    // should be caught by now
    assert(false && "error, invalid number of vertices");
  }

  // make an array of equivalent vertices, which for each index
  // contains the lowest vertex equivalent to the vertex at that index
  // or its own index if not equivalent to any other vertex
  vector<size_t> equivalent_vertices(8);
  for (size_t i = 1; i < 8; ++i)
  {
    for (int j = i - 1; j >= 0; --j)
    {
      if (tolerances.is_distance_near_zero((vertices[i] - vertices[j]).length()))
      {
        equivalent_vertices[i] = equivalent_vertices[j];
        goto next_point;
      }
    }
    equivalent_vertices[i] = i;
  next_point: ;
  }

  // check each face for validity
  stringstream face_check_diagnostic_stream;
  int face_count = 0;
  for (size_t i = 0; i < 6; ++i)
  {
    // create an array of points representing the vertices of the face
    vector<Point> face_vertices;
    vector<size_t> face_indices;
    for (size_t j = 0; j < 4; ++j)
    {
      size_t vertex_index = ic_face_layout[i].vertex[j];
      vertex_index = equivalent_vertices[vertex_index];
      // make sure this is not a duplicate vertex
      for (vector<size_t>::const_iterator k = face_indices.begin(); k != face_indices.end(); ++k)
      {
        if (*k == vertex_index)
        {
          goto skip_vertex;
        }
      }
      face_vertices.push_back(m_vertices[vertex_index]);
      face_indices.push_back(vertex_index);
    skip_vertex: ;
    }
    if (face_vertices.size() > 2)
    {
      string face_check_diagnostic;
      const int accepted = ih_check_face(face_vertices,
                                         ic_face_layout[i].description,
                                         tolerances,
                                         face_check_diagnostic);
      if (!accepted)
      {
        face_check_diagnostic_stream << "  " << face_check_diagnostic << endl;
      }
      else
      {
        ++face_count;
      }
    }
  }

  if (face_count < 4)
  {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid " << solid_class();
      const string& solid_name = name();
      diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
      diagnostic_stream << ": face count is < 4" << endl;
      const string face_check_diagnostics = face_check_diagnostic_stream.str();
      if ("" != face_check_diagnostics)
      {
        diagnostic_stream << "Details:" << endl << face_check_diagnostics;
      }
      on_error(true, new Math_error(error_location(),
                                    Math_error::validation,
                                    diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

Solid& Polyhedron::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>(this));

  // handle each vertex count
  switch (m_vertices.size())
  {
  case 4:
    break;
  case 5:
    break;
  case 6:
    {
      const Point& vertex = m_vertices[4];
      const Vector width = m_vertices[0] - vertex;
      const Vector height = m_vertices[1] - vertex;
      const Vector depth_3 = m_vertices[3] - m_vertices[0];
      const Vector depth_2 = m_vertices[2] - m_vertices[1];
      const Vector depth_5 = m_vertices[5] - m_vertices[4];
      // check for equal depths
      if (tolerances.is_distance_near_zero((depth_3 - depth_2).length())
          && tolerances.is_distance_near_zero((depth_2 - depth_5).length()))
      {
        Unit_vector unit_width;
        width.normalize(unit_width);
        Unit_vector unit_height;
        height.normalize(unit_height);
        Unit_vector unit_depth;
        depth_5.normalize(unit_depth);
        // check for mutually perpendicular width, height, depth
        if (tolerances.is_cosine_near_zero(unit_width.dot(unit_height))
            && tolerances.is_cosine_near_zero(unit_width.dot(unit_depth))
            && tolerances.is_cosine_near_zero(unit_height.dot(unit_depth)))
        {
          // we have a Wedge
          Wedge* wedge =new Wedge(vertex,
                                   width,
                                   height,
                                   depth_5,
                                   name());
          return_value = wedge;
          break;
        }
      }
    }
    break;
  case 7:
    break;
  case 8:
    {
      const Point& vertex = m_vertices[0];
      const Vector width_1 = m_vertices[1] - vertex;
      const Vector height_2 = m_vertices[2] - m_vertices[1];
      const Vector height_3 = m_vertices[3] - vertex;
      const Vector depth = m_vertices[4] - vertex;
      const Vector width_5 = m_vertices[5] - m_vertices[4];
      const Vector height_6 = m_vertices[6] - m_vertices[5];
      const Vector height_7 = m_vertices[7] - m_vertices[4];
      // check for equal widths and heights
      if (tolerances.is_distance_near_zero((width_1 - width_5).length())
          && tolerances.is_distance_near_zero((height_2 - height_3).length())
          && tolerances.is_distance_near_zero((height_3 - height_6).length())
          && tolerances.is_distance_near_zero((height_6 - height_7).length()))
      {
        Unit_vector unit_width;
        width_1.normalize(unit_width);
        Unit_vector unit_height;
        height_3.normalize(unit_height);
        Unit_vector unit_depth;
        depth.normalize(unit_depth);
        // check for mutually perpendicular width, height, and depth
        if (tolerances.is_cosine_near_zero(unit_width.dot(unit_height))
            && tolerances.is_cosine_near_zero(unit_width.dot(unit_depth))
            && tolerances.is_cosine_near_zero(unit_height.dot(unit_depth)))
        {
          // we have a box
          Box* box = new Box(vertex,
                               width_1,
                               height_3,
                               depth,
                               name());
          // determine the specific type of box
          return_value = &box->specialize(tolerances);
          if (return_value != box)
          {
            box->reference();
            box->unreference();
          }
          break;
        }
      }
    }
    break;
  default: // invalid number of vertices, don't specialize
    break;
  }

  return *return_value;
}
