// Torus.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Torus.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Matrix.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;

using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Torus::Torus(const string& name)
  : Solid(name),
    m_center(Point::O),
    m_normal(Unit_vector::x),
    m_major_radius(Distance(0.8 * Distance::meter)),
    m_minor_radius(Distance(0.2 * Distance::meter))
{
}

Torus::Torus(const Point& center,
             const Unit_vector& normal,
             const Distance& major_radius,
             const Distance& minor_radius,
             const string& name)
  : Solid(name),
    m_center(center),
    m_normal(normal),
    m_major_radius(major_radius),
    m_minor_radius(minor_radius)
{
}

//
// Predicates
//

bool Torus::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for positive radius
  if (tolerances.is_distance_near_zero(m_major_radius)
      || m_major_radius < Distance(0.0 * Distance::meter)
      || tolerances.is_distance_near_zero(m_minor_radius)
      || m_minor_radius < Distance(0.0 * Distance::meter))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " + solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or both radii are non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for non-zero normal vector
  if (tolerances.is_unitless_near_zero(m_normal.length()))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the normal vector is zero";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
