// Triangle_mesh.cpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Solid validation based on ./src/librt/primitives/bot/g_bot_include.c from
// BRL-CAD (version 7.20.4) source:
//
//   Copyright (c) 1999-2011 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>

using std::endl;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using Roan_trail::Tovero_math::is_quantity_near_zero;
using namespace Roan_trail::Tovero_math;

//
// Constructors
//

Triangle_mesh::Triangle_mesh(Triangle_face::Triangle_direction_type triangle_direction,
                             const vector<Point>& vertices,
                             const vector<Triangle_face>& faces,
                             const vector<Unit_vector>& normals,
                             const vector<Triangle_face>& face_normals,
                             bool has_surface_normals,
                             bool use_surface_normals,
                             const string& name)
  : Solid(name),
    m_triangle_direction(triangle_direction),
    m_vertices(vertices),
    m_faces(faces),
    m_normals(normals),
    m_face_normals(face_normals),
    m_has_surface_normals(use_surface_normals),
    m_use_surface_normals(use_surface_normals)
{
}

//
// Predicates
//

bool Triangle_mesh::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for positive number of faces
  const size_t face_count = m_faces.size();
  if (!face_count)
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": no faces";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }
  // check for invalid face indices
  const size_t vertex_count = m_vertices.size();
  for (size_t i = 0; i < face_count; ++i)
  {
    const Triangle_face& face = m_faces[i];
    if ((face.index_1() >= vertex_count)
        || (face.index_2() >= vertex_count)
        || (face.index_3() >= vertex_count))
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid " << solid_class();
      const string& solid_name = name();
      diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
      diagnostic_stream << ": face " << i << "references a nonexistent vertex";
      on_error(true, new Math_error(error_location(),
                                    Math_error::validation,
                                    diagnostic_stream.str()));
    }
  }
  // check normals
  if (m_has_surface_normals)
  {
    const size_t normal_count = m_normals.size();
    for (size_t i = 0; i < normal_count; ++i)
    {
      if (tolerances.is_unitless_near_zero(m_normals[i].length()))
      {
        stringstream diagnostic_stream;
        diagnostic_stream << "Invalid " << solid_class();
        const string& solid_name = name();
        diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
        diagnostic_stream << ": face normal " << i << " is degenerate";
        on_error(true, new Math_error(error_location(),
                                      Math_error::validation,
                                      diagnostic_stream.str()));
      }
    }
    // check that there are enough face normals
    const size_t face_normal_count = m_face_normals.size();
    if (face_normal_count != face_count)
    {
      stringstream diagnostic_stream;
      diagnostic_stream << "Invalid " << solid_class();
      const string& solid_name = name();
      diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
      diagnostic_stream << ": face normal count " << face_normal_count;
      diagnostic_stream << " does not equal face count " << face_count;
      on_error(true, new Math_error(error_location(),
                                    Math_error::validation,
                                    diagnostic_stream.str()));
    }
    // check for invalid face normal indices
    for (size_t i = 0; i < face_normal_count; ++i)
    {
      const Triangle_face& face_normal = m_face_normals[i];
      if ((face_normal.index_1() >= vertex_count)
          || (face_normal.index_2() >= vertex_count)
          || (face_normal.index_3() >= vertex_count))
      {
        stringstream diagnostic_stream;
        diagnostic_stream << "Invalid " << solid_class();
        const string& solid_name = name();
        diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
        diagnostic_stream << ": face normal " << i << " references a nonexistent vertex";
        on_error(true, new Math_error(error_location(),
                                      Math_error::validation,
                                      diagnostic_stream.str()));
      }
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}
