// Vector.hpp
//
// Copyright 2012-2014 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// affine 3D vector

#ifndef TOVERO_MATH_VECTOR_HPP_
#define TOVERO_MATH_VECTOR_HPP_

#include <tovero/math/geometry/Area.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <iostream>
#include <cmath>
#include <cstddef>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Vector
    {
    public:
      // constructors
      inline Vector();
      inline Vector(const Distance& x,
                    const Distance& y,
                    const Distance& z);
      inline Vector(double x,
                    double y,
                    double z,
                    const Distance& units);
      inline Vector(const Vector& v);

      // operators
      inline Vector& operator=(const Vector& v);
      const Distance& operator[](size_t index) const { return m_v[index]; }
      Distance& operator[](size_t index) { return m_v[index]; }
      inline Vector operator+(const Vector& v) const;
      inline Vector operator-(const Vector& v) const;
      inline Vector operator*(const Unitless& factor) const;
      inline Vector operator/(const Unitless& divisor) const;
      inline Vector& operator+=(const Vector& v);
      inline Vector& operator-=(const Vector& v);
      inline Vector& operator*=(const Unitless& factor);
      inline Vector& operator/=(const Unitless& divisor);
      // functions
      inline Distance length() const;
      Area length_squared() const { return (m_v[0] * m_v[0] + m_v[1] * m_v[1] + m_v[2] * m_v[2]); }
      inline Distance normalize(Unit_vector& return_v) const;
      inline const Area dot(const Vector& v) const;
      inline Unit_vector cross(const Vector& v, Area& return_cross_area) const;
    private:
      Distance m_v[3];
    };

    // free functions

    inline std::ostream& operator<<(std::ostream& os, const Vector& v);
    inline Vector operator*(const Unitless& factor, const Vector& v);
    inline Vector operator*(const Unit_vector& u, const Distance& d);
    inline Vector operator*(const Distance& d, const Unit_vector& u);

    //
    // Inline definitions
    //

    //
    //   Constructors
    //

    inline Vector::Vector()
    {
      m_v[0] = Distance::from_value(0.0);
      m_v[1] = Distance::from_value(0.0);
      m_v[2] = Distance::from_value(0.0);
    }

    inline Vector::Vector(const Distance& x,
                          const Distance& y,
                          const Distance& z)
    {
      m_v[0] = x;
      m_v[1] = y;
      m_v[2] = z;
    }

    inline Vector::Vector(double x,
                          double y,
                          double z,
                          const Distance& units)
    {
      m_v[0] = x * units;
      m_v[1] = y * units;
      m_v[2] = z * units;
    }

    inline Vector::Vector(const Vector& v)
    {
      m_v[0] = v.m_v[0];
      m_v[1] = v.m_v[1];
      m_v[2] = v.m_v[2];
    }

    //
    //   Operators
    //

    inline Vector& Vector::operator=(const Vector& v)
    {
      if (this != &v)
      {
        m_v[0] = v.m_v[0];
        m_v[1] = v.m_v[1];
        m_v[2] = v.m_v[2];
      }
      return *this;
    }

    inline Vector Vector::operator+(const Vector& v) const
    {
      return Vector(m_v[0] + v.m_v[0],
                    m_v[1] + v.m_v[1],
                    m_v[2] + v.m_v[2]);
    }

    inline Vector Vector::operator-(const Vector& v) const
    {
      return Vector(m_v[0] - v.m_v[0],
                    m_v[1] - v.m_v[1],
                    m_v[2] - v.m_v[2]);
    }

    inline Vector Vector::operator*(const Unitless& factor) const
    {
      return Vector(m_v[0] * factor,
                    m_v[1] * factor,
                    m_v[2] * factor);
    }

    inline Vector Vector::operator/(const Unitless& divisor) const
    {
      return Vector(m_v[0] / divisor,
                    m_v[1] / divisor,
                    m_v[2] / divisor);
    }

    inline Vector& Vector::operator+=(const Vector& v)
    {
      m_v[0] += v.m_v[0];
      m_v[1] += v.m_v[1];
      m_v[2] += v.m_v[2];

      return *this;
    }

    inline Vector& Vector::operator-=(const Vector& v)
    {
      m_v[0] -= v.m_v[0];
      m_v[1] -= v.m_v[1];
      m_v[2] -= v.m_v[2];

      return *this;
    }

    inline Vector& Vector::operator*=(const Unitless& factor)
    {
      m_v[0] *= Unitless(factor.value());
      m_v[1] *= Unitless(factor.value());
      m_v[2] *= Unitless(factor.value());

      return *this;
    }

    inline Vector& Vector::operator/=(const Unitless& divisor)
    {
      m_v[0] /= Unitless(divisor.value());
      m_v[1] /= Unitless(divisor.value());
      m_v[2] /= Unitless(divisor.value());

      return *this;
    }

    //
    //   Functions
    //

    inline Distance Vector::length() const
    {
      return Distance::from_value(::sqrt(m_v[0].value() * m_v[0].value()
                                         + m_v[1].value() * m_v[1].value()
                                         + m_v[2].value() * m_v[2].value()));
    }

    inline Distance Vector::normalize(Unit_vector& return_u) const
    {
      const Distance old_length = length();

      Unit_vector::normalize(m_v[0] / old_length,
                             m_v[1] / old_length,
                             m_v[2] / old_length,
                             return_u);

      return old_length;
    }

    inline const Area Vector::dot(const Vector& v) const
    {
      const Area return_value = m_v[0] * v.m_v[0] + m_v[1] * v.m_v[1] + m_v[2] * v.m_v[2];

      return return_value;
    }

    inline Unit_vector Vector::cross(const Vector& v, Area& return_cross_area) const
    {
      Unit_vector unit;
      Distance length = normalize(unit);
      Unit_vector unit_v;
      Distance length_v = v.normalize(unit_v);

      return_cross_area = length * length_v;

      return unit.cross(unit_v);
    }

    //
    //   Free functions
    //

    inline std::ostream& operator<<(std::ostream& os, const Vector& v)
    {
      os << "[" << v[0].value() << ", " << v[1].value() << ", " << v[2].value() << "]";

      return os;
    }

    inline Vector operator*(const Unitless& factor, const Vector& v)
    {
      return Vector(factor * v[0],
                    factor * v[1],
                    factor * v[2]);
    }

    inline Vector operator*(const Unit_vector& u, const Distance& d)
    {
      return Vector(u[0] * d,
                    u[1] * d,
                    u[2] * d);
    }

    inline Vector operator*(const Distance& d, const Unit_vector& u)
    {
      return Vector(u[0] * d,
                    u[1] * d,
                    u[2] * d);
    }

  }
}

#endif // TOVERO_MATH_VECTOR_HPP_

