// Wedge.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Wedge.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <vector>

using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor/copy
//

Wedge::Wedge(const string& name)
  : Polyhedron_base(name),
    m_vertex(Point(Distance(-1.0 * Distance::meter),
                   Distance(-1.0 * Distance::meter),
                   Distance(-1.0 * Distance::meter))),
    m_width(Unit_vector::x * (2.0 * Distance::meter)),
    m_height(Unit_vector::y * (2.0* Distance::meter)),
    m_depth(Unit_vector::z * (2.0 * Distance::meter))
{
}

Wedge::Wedge(const Point& vertex,
             const Vector& width,
             const Vector& height,
             const Vector& depth,
             const string& name)
  : Polyhedron_base(name),
    m_vertex(vertex),
    m_width(width),
    m_height(height),
    m_depth(depth)
{
}

//
// Predicates
//

bool Wedge::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for non-zero direction vector lengths
  Unit_vector unit_h;
  const Distance height_length = m_height.normalize(unit_h);
  Unit_vector unit_w;
  const Distance width_length = m_width.normalize(unit_w);
  Unit_vector unit_d;
  const Distance depth_length = m_depth.normalize(unit_d);
  if (tolerances.is_distance_near_zero(height_length)
      || tolerances.is_distance_near_zero(width_length)
      || tolerances.is_distance_near_zero(depth_length))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or more direction vectors is zero length";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for mutually perpendicular direction vectors
  if (!tolerances.is_cosine_near_zero(unit_h.dot(unit_w))
      || !tolerances.is_cosine_near_zero(unit_h.dot(unit_d))
      || !tolerances.is_cosine_near_zero(unit_w.dot(unit_d)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the direction vectors are not mutually perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

Polyhedron& Wedge::generalize() const
{
  vector<Point> vertices;
  vertices.push_back(m_vertex + m_width);     // vertex 0
  vertices.push_back(m_vertex + m_height);    // vertex 1
  vertices.push_back(vertices[1] + m_depth);  // vertex 2
  vertices.push_back(vertices[0] + m_depth);  // vertex 3
  vertices.push_back(m_vertex);               // vertex 4
  vertices.push_back(m_vertex + m_depth);     // vertex 5

  Polyhedron* polyhedron = new Polyhedron(vertices, name());

  postcondition(polyhedron && (6 == polyhedron->vertices().size()));
  return *polyhedron;
}
