# test_elliptical_cylinder.rb
#
# Copyright 2012-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Tovero is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License
# version 2.1 as published by the Free Software Foundation.
#
# Tovero is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.  You should have
# received a copy of the GNU Lesser General Public License along with
# Tovero. If not, see <http://www.gnu.org/licenses/>.

# Ruby unit test (minitest) for the Elliptical_cylinder class

require 'minitest/autorun'
require 'test/unit'
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'

include Libtovero_support_rb_1
include Libtovero_math_rb_1

class TestEllipticalCylinder < Test::Unit::TestCase
  def setup
    @Tolerances = GeometricTolerances.new
  end

  #
  # Solid validity checks
  #

  def test_default
    cylinder = EllipticalCylinder.new
    error = ErrorParam.new
    #
    assert cylinder.is_valid?(@Tolerances, error)
  end

  def test_zero_length_base_vector
    cylinder = EllipticalCylinder.new
    zero_length_vector = Vector.new
    cylinder.base_b = zero_length_vector
    error = ErrorParam.new
    #
    refute cylinder.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_degenerate_base_vectors
    cylinder = EllipticalCylinder.new
    small_vector = UnitVector::x * @Tolerances.distance
    cylinder.base_a = small_vector
    cylinder.base_b = small_vector
    error = ErrorParam.new
    #
    refute cylinder.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_non_perpendicular_base_vectors
    cylinder = EllipticalCylinder.new
    cylinder.base_a = UnitVector::x * Distance.new(1.0, Distance::meter)
    cylinder.base_b = cylinder.base_a
    error = ErrorParam.new
    #
    refute cylinder.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_zero_height
    cylinder = EllipticalCylinder.new
    cylinder.name = "cylinder"
    cylinder.height = Distance.new(0.0, Distance::meter)
    error = ErrorParam.new
    #
    refute cylinder.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  #
  # Miscellaneous checks
  #

  def test_properties
    cylinder = EllipticalCylinder.new
    #
    assert_respond_to cylinder, :base
    assert_respond_to cylinder, :base=
    assert_respond_to cylinder, :base_a
    assert_respond_to cylinder, :base_a=
    assert_respond_to cylinder, :base_b
    assert_respond_to cylinder, :base_b=
    assert_respond_to cylinder, :height
    assert_respond_to cylinder, :height=
  end

end
