# test_general_cone.rb
#
# Copyright 2012-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Tovero is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License
# version 2.1 as published by the Free Software Foundation.
#
# Tovero is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.  You should have
# received a copy of the GNU Lesser General Public License along with
# Tovero. If not, see <http://www.gnu.org/licenses/>.

# Ruby unit test (minitest) for the General_cone class

require 'minitest/autorun'
require 'test/unit'
require 'libtovero_support_rb_1'
require 'libtovero_math_rb_1'

include Libtovero_support_rb_1
include Libtovero_math_rb_1

class TestGeneralCone < Test::Unit::TestCase
  def setup
    @Tolerances = GeometricTolerances.new
    @Zero_length_vector = Vector.new
    @Small_vector = UnitVector::x * @Tolerances.distance
  end

  #
  # Solid validity checks
  #

  def test_default
    cone = GeneralCone.new
    error = ErrorParam.new
    #
    assert cone.is_valid?(@Tolerances, error)
  end

  def test_zero_length_height
    cone = GeneralCone.new
    cone.name = "cone"
    cone.height = @Zero_length_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_zero_length_base_vector
    cone = GeneralCone.new
    cone.base_a = @Zero_length_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_zero_length_top_vector
    cone = GeneralCone.new
    cone.top_d = @Zero_length_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_both_ends_degenerate
    cone = GeneralCone.new
    cone.base_a = @Small_vector
    cone.base_b = @Small_vector
    cone.top_c = @Small_vector
    cone.top_d =@Small_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_height_lies_in_base_a_b_plane
    cone = GeneralCone.new
    cone.height = UnitVector::x * Distance::meter
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_base_vectors_not_perpendicular
    cone = GeneralCone.new
    cone.base_a = Vector.new(1.0, 1.0, 1.0, Distance::meter);
    cone.base_b = Vector.new(1.0, 0.0, 1.0, Distance::meter);
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_top_vectors_not_perpendicular
    cone = GeneralCone.new
    cone = GeneralCone.new
    cone.base_a = Vector.new(1.0, 1.0, 1.0, Distance::meter);
    cone.base_b = Vector.new(1.0, 0.0, 1.0, Distance::meter);
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_base_a_top_c_not_parallel
    cone = GeneralCone.new
    cone = GeneralCone.new
    cone.base_a = Vector.new(1.0, 1.0, 1.0, Distance::meter);
    cone.top_c = Vector.new(1.0, 0.0, 1.0, Distance::meter);
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_base_b_top_d_not_parallel
    cone = GeneralCone.new
    cone = GeneralCone.new
    cone.base_b = Vector.new(1.0, 1.0, 1.0, Distance::meter);
    cone.top_d = Vector.new(1.0, 0.0, 1.0, Distance::meter);
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  #
  # Miscellaneous checks
  #

  def test_properties
    cone = GeneralCone.new
    #
    assert_respond_to cone, :base
    assert_respond_to cone, :base=
    assert_respond_to cone, :height
    assert_respond_to cone, :height=
    assert_respond_to cone, :base_a
    assert_respond_to cone, :base_a=
    assert_respond_to cone, :base_b
    assert_respond_to cone, :base_b=
    assert_respond_to cone, :top_c
    assert_respond_to cone, :top_c=
    assert_respond_to cone, :top_d
    assert_respond_to cone, :top_d=
  end

end
