#!/bin/bash

# check_code.bash
#
# Copyright 2011-2014 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Tovero is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License
# version 2.1 as published by the Free Software Foundation.
#
# Tovero is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.  You should have
# received a copy of the GNU Lesser General Public License along with
# Tovero. If not, see <http://www.gnu.org/licenses/>.

# Note: these checks aren't necessarily complete or exhaustive, just meant
# to help in code review

FAILED=0

RULE_PREFIX="Check code: "

GREP="grep"

TEST_DIR=`pwd`

for code_subdir in ${CHECK_SRC_SUBDIRS}
do
    CODE_SUBDIR="${TEST_DIR}/../src/${code_subdir}"
    cd ${CODE_SUBDIR}
    echo "Checking directory: ${CODE_SUBDIR}"

    # Ref. Standards Rule #9, Standards Rule #43
    CURRENT_RULE="basic C++ char set (no tabs)"
    for i in *.[ch]pp
    do
        EXCLUDE_FILE="no"
        for j in ${CHECK_CODE_EXCLUDE}; do if [ ${j} == ${i} ]; then EXCLUDE_FILE="yes"; break; fi; done
        if [ ${EXCLUDE_FILE} == "yes" ]; then continue; fi

        if "${TEST_DIR}/check_basic_char_set" ${CODE_SUBDIR} ${i}
        then
        # success
            echo SUCCESS > /dev/null
        else
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        fi
    done

    # Ref. Standards Rule #13
    CURRENT_RULE="no multi-byte characters or wide string literals"
    for i in *.[ch]pp
    do
        if ${GREP} -q wchar_t ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #17
    CURRENT_RULE="no errno"
    for i in *.[ch]pp
    do
        if ${GREP} errno ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #18
    CURRENT_RULE="no offsetof"
    for i in *.[ch]pp
    do
        if ${GREP} offsetof ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #19
    CURRENT_RULE="no locale"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'locale\.h|clocale|setlocale' ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #21
    CURRENT_RULE="no signal"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'csignal|signal\.h' ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #20
    CURRENT_RULE="no setjmp or longjmp"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'setjmp|longjmp' ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #22
    CURRENT_RULE="no stdio.h"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'cstdio|stdio\.h' ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #23
    CURRENT_RULE="no atof, atoi, atol"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'atof|atoi|atol' ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #24
    CURRENT_RULE="no abort, exit, getenv and system"
    for i in *.[ch]pp
    do
        if ${GREP} ' exit(' ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
        if ${GREP} getenv\( ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
        if ${GREP} -E 'system\(|abort\(' ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #25
    CURRENT_RULE="no time.h"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'ctime|time\.h' ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    #Ref. Standards Rule #27, Rule #35
    CURRENT_RULE="prevention of multiple inclusions of header files"
    for i in *.hpp
    do
        GOT_IFNDEF=`${GREP} ifndef ${i} | ${GREP} -i ${i%.hpp}_hpp_`
        GOT_DEFINE=`${GREP} define ${i} | ${GREP} -i ${i%.hpp}_hpp_`
        GOT_ENDIF=`${GREP} endif ${i} | ${GREP} -i ${i%.hpp}_hpp_`
        if [ -n "${GOT_IFNDEF}" -a -n "${GOT_DEFINE}" -a -n "${GOT_ENDIF}" ]
        then
        # success
            echo SUCCESS > /dev/null
        else
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        fi
    done

    # Ref. Standards Rule #41
    CURRENT_RULE="120 column source code limit"
    # Note: (121 chars including NL)"
    for i in *.[ch]pp
    do
        EXCLUDE_FILE="no"
        for j in ${CHECK_CODE_EXCLUDE}; do if [ ${j} == ${i} ]; then EXCLUDE_FILE="yes"; break; fi; done
        if [ ${EXCLUDE_FILE} == "yes" ]; then continue; fi

        if diff <(cut -c1-121 ${i}) <(cat ${i}) 2>&1 > /dev/null
        then
        # success
            echo SUCCESS > /dev/null
        else
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        fi
    done

    # Ref. Standards Rule #42
    CURRENT_RULE="separate line for each expression-statement"
    for i in *.cpp # no .hpp files, because "inline" style is allowed
    do
        if ${GREP} -E 'if \(|else|while \(|class|for \(|namespace' ${i} \
            | ${GREP} \{ | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #126
    CURRENT_RULE="no C style comments"
    for i in *.[ch]pp
    do
        EXCLUDE_FILE="no"
        for j in ${CHECK_CODE_EXCLUDE}; do if [ ${j} == ${i} ]; then EXCLUDE_FILE="yes"; break; fi; done
        if [ ${EXCLUDE_FILE} == "yes" ]; then continue; fi

        if ${GREP} -q -E '\/\*|\*\/' ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #140
    CURRENT_RULE="no register class storage"
    for i in *.[ch]pp
    do
        if ${GREP} -q "register " ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #153
    CURRENT_RULE="no unions"
    for i in *.[ch]pp
    do
        if ${GREP} -q union ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #159
    CURRENT_RULE="no overloading operator&"
    for i in *.[ch]pp
    do
        if ${GREP} -q operator\& ${i}
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #169
    CURRENT_RULE="avoid pointers to pointers"
    for i in *.[ch]pp
    do
        if ${GREP} -q '\*\*' ${i} | grep -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #175
    CURRENT_RULE="no comparison or assignment to pointers of NULL"
    for i in *.[ch]pp
    do
        if ${GREP} -q NULL ${i} | grep -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #190
    CURRENT_RULE="no use of continue statement"
    for i in *.[ch]pp
    do
        if ${GREP} -q continue ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #197
    CURRENT_RULE="no floating point variables as loop counters"
    for i in *.[ch]pp
    do
        if ${GREP} -E 'for \(' ${i} | ${GREP} -q -E 'float|double'
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

    # Ref. Standards Rule #205
    CURRENT_RULE="no use of volatile keyword"
    for i in *.[ch]pp
    do
        if ${GREP} -q volatile ${i} | ${GREP} -q -v check_code_ignore
        then
            echo "${CODE_SUBDIR}/${i}: ${RULE_PREFIX}${CURRENT_RULE} FAILED"
            FAILED=1
        else
        # success
            echo SUCCESS > /dev/null
        fi
    done

done

exit ${FAILED}
