/* tpb - program to use the IBM ThinkPad(tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* includes {{{ */
#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include "config.h"

#ifdef HAVE_LIBXOSD
#include <xosd.h>
#endif /* HAVE_LIBXOSD */

#include "cfg.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{*/
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: cfg.c,v 1.5 2002/07/20 21:46:07 mbr Exp $";
#endif
/* }}} */

config cfg;

int parse_cfg_file(char *name) { /* {{{ */
	FILE *fdsc;
	char kb[BSIZE];
	char ab[BSIZE];
	int c;
	int kbptr=0;
	int abptr=0;
	enum mode mode=MODE_INDENT;
	config file_cfg = {
		.nvram = NULL,
		.polltime = -1,
		.tpbcmd = NULL,
		.mixer = -1,
		.mixersteps = -1,
		.mixerdev = NULL,
		.verbose = -1
#ifdef HAVE_LIBXOSD
			,
		.osd = -1,
		.osdfont = NULL,
		.osdcolor = NULL,
		.osdtimeout = -1,
		.osdoffset = -1,
		.osdpos = -1
#endif /* HAVE_LIBXOSD */
	};


	/* open configuration file */
	if((fdsc=fopen(name, "r")) == NULL) {
		return(-1);
	}

	/* loop until we reach the end of the file {{{ */
	while(mode != MODE_END) {
		c=fgetc(fdsc);
		switch(c) {
			//case '=':
			case ' ':
			case '\t': switch(mode) {
									 case MODE_KEYWORD: mode=MODE_SEPARATOR;
																			break;
									 case MODE_ARGUMENT: if(abptr < BSIZE-1) {
																				 ab[abptr++]=c;
																			 }
																			 else {
																				 fputs("Argument in configuration too long!\n",stderr);
																				 exit(1);
																			 }
																			 break;
									 default: break;
								 }
								 break;
			case '\n': mode=MODE_INDENT;
								 kb[kbptr]='\0';
								 ab[abptr]='\0';
								 if(strlen(kb) >0 && strlen(ab)>0) {
									 abptr--;
									 while(ab[abptr] == ' ' || ab[abptr] == '\t') {
										 ab[abptr] = '\0';
										 abptr--;
									 }
									 set_value(kb, ab, &file_cfg);
								 }
								 //if(strlen(kb) >0 && strlen(ab)>0) printf("Keyword: \"%s\", Argument: \"%s\"\n",kb,ab);
								 kbptr=0;
								 abptr=0;
								 break;
			case '#': mode=MODE_COMMENT;
								break;
			case EOF: mode=MODE_END;
								break;
			default: switch(mode) {
								 case MODE_INDENT: mode=MODE_KEYWORD;
								 case MODE_KEYWORD: if(kbptr < BSIZE-1) {
																			kb[kbptr++]=tolower(c);
																		}
																		else {
																			fputs("Keyword in configuration too long!\n",stderr);
																			exit(1);
																		}
																		break;
								 case MODE_SEPARATOR: mode=MODE_ARGUMENT;
								 case MODE_ARGUMENT: if(abptr < BSIZE-1) {
																			 ab[abptr++]=c;
																		 }
																		 else {
																			 fputs("Argument in configuration too long!\n",stderr);
																			 exit(1);
																		 }
																		 break;
								 default: break;
							 }
							 break;
		}
	} /* }}} */

	/* override the config with values from configuration file */
	override_cfg(&file_cfg);

	return(0);
} /* }}} */

void init_cfg(void) { /* {{{ */

	/* set all values to defaults */
	if((cfg.nvram = strdup(NVRAMDEV)) == NULL) {
		fprintf(stderr,"Not enough memory");
		_exit(1);
	}
	cfg.polltime = POLLTIME;
	cfg.tpbcmd = NULL;
	cfg.mixer = 0;
	cfg.mixersteps = MIXERSTEPS;
	cfg.mixerdev = MIXERDEV;
	cfg.verbose = 0;
#ifdef HAVE_LIBXOSD
	cfg.osd = 1;
	if((cfg.osdfont = strdup(OSDFONT)) == NULL) {
		fprintf(stderr,"Not enough memory");
		_exit(1);
	}
	if((cfg.osdcolor = strdup(OSDCOLOR)) == NULL) {
		fprintf(stderr,"Not enough memory");
		_exit(1);
	}
	cfg.osdtimeout = OSDTIMEOUT;
	cfg.osdoffset = OSDOFFSET;
	cfg.osdpos = OSDPOS;
#endif /* HAVE_LIBXOSD */

	return;
} /* }}} */

void override_cfg(config *master) { /* {{{ */
	/* override the defaults if the are set in master struct */
	if(master->nvram != NULL) {
		if(cfg.nvram) free(cfg.nvram);
		cfg.nvram=master->nvram;
	}
	if(master->polltime != -1) {
		cfg.polltime=master->polltime;
	}
	if(master->tpbcmd != NULL) {
		if(cfg.tpbcmd) free(cfg.tpbcmd);
		cfg.tpbcmd=master->tpbcmd;
	}
	if(master->mixer != -1) {
		cfg.mixer=master->mixer;
	}
	if(master->mixersteps != -1) {
		cfg.mixersteps=master->mixersteps;
	}
	if(master->mixerdev != NULL) {
		if(cfg.mixerdev) free(cfg.mixerdev);
		cfg.mixerdev=master->mixerdev;
	}
	if(master->verbose != -1) {
		cfg.verbose=master->verbose;
	}
#ifdef HAVE_LIBXOSD
	if(master->osd != -1) {
		cfg.osd=master->osd;
	}
	if(master->osdfont != NULL) {
		if(cfg.osdfont) free(cfg.osdfont);
		cfg.osdfont=master->osdfont;
	}
	if(master->osdcolor != NULL) {
		if(cfg.osdcolor) free(cfg.osdcolor);
		cfg.osdcolor=master->osdcolor;
	}
	if(master->osdtimeout != -1) {
		cfg.osdtimeout=master->osdtimeout;
	}
	if(master->osdoffset != -1) {
		cfg.osdoffset=master->osdoffset;
	}
	if(master->osdpos != -1) {
		cfg.osdpos=master->osdpos;
	}
#endif /* HAVE_LIBXOSD */
	return;
} /* }}} */

void set_value(char *key, char *arg, config *cfg) { /* {{{ */
	char *endptr;

	if (strcmp(CFG_NVRAM,key) == 0) { /* {{{ */
		if(cfg->nvram != NULL) {
			free(cfg->nvram);
		}
		if((cfg->nvram=strdup(arg)) == NULL) {
			fprintf(stderr,"Not enough memory");
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_POLLTIME,key) == 0) { /* {{{ */
		cfg->polltime=strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->polltime < 0) {
			fprintf(stderr,"Illegal polltime \"%s\"\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_THINKPAD,key) == 0) { /* {{{ */
		if(cfg->tpbcmd != NULL) {
			free(cfg->tpbcmd);
		}
		if((cfg->tpbcmd=strdup(arg)) == NULL) {
			fprintf(stderr,"Not enough memory");
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_MIXER,key) == 0) { /* {{{ */
		int i=0;
		while(arg[i]) {
			arg[i]=tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_MIXER_ON,arg) == 0) cfg->mixer=1;
		else if(strcmp(CFG_MIXER_OFF,arg) == 0) cfg->mixer=0;
		else {
			fprintf(stderr, "Illegal mixer state %s\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_MIXERSTEPS,key) == 0) { /* {{{ */
		cfg->mixersteps=strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->mixersteps < 0) {
			fprintf(stderr,"Illegal mixersteps \"%s\"\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_MIXERDEV,key) == 0) { /* {{{ */
		if(cfg->mixerdev != NULL) {
			free(cfg->mixerdev);
		}
		if((cfg->mixerdev=strdup(arg)) == NULL) {
			fprintf(stderr,"Not enough memory");
			_exit(1);
		}
	} /* }}} */
#ifdef HAVE_LIBXOSD
	else if (strcmp(CFG_OSDFONT,key) == 0) { /* {{{ */
		if(cfg->osdfont != NULL) {
			free(cfg->osdfont);
		}
		if((cfg->osdfont=strdup(arg)) == NULL) {
			fprintf(stderr,"Not enough memory");
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_OSDCOLOR,key) == 0) { /* {{{ */
		if(cfg->osdcolor != NULL) {
			free(cfg->osdcolor);
		}
		if((cfg->osdcolor=strdup(arg)) == NULL) {
			fprintf(stderr,"Not enough memory");
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_OSDTIMEOUT,key) == 0) { /* {{{ */
		cfg->osdtimeout=strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->osdtimeout < 0) {
			fprintf(stderr,"Illegal xosd timeout \"%s\"\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_OSDOFFSET,key) == 0) { /* {{{ */
		cfg->osdoffset=strtol(arg, &endptr, 10);
		if(strlen(endptr)>0) {
			fprintf(stderr,"Illegal xosd offset \"%s\"\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_OSDPOS,key) == 0) { /* {{{ */
		int i=0;
		while(arg[i]) {
			arg[i]=tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_OSDPOS_TOP,arg) == 0) cfg->osdpos=XOSD_top;
		else if(strcmp(CFG_OSDPOS_BOTTOM,arg) == 0) cfg->osdpos=XOSD_bottom;
		else {
			fprintf(stderr, "Illegal xosd position %s\n",arg);
			_exit(1);
		}
	} /* }}} */
	else if (strcmp(CFG_OSD,key) == 0) { /* {{{ */
		int i=0;
		while(arg[i]) {
			arg[i]=tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_OSD_ON,arg) == 0) cfg->osd=1;
		else if(strcmp(CFG_OSD_OFF,arg) == 0) cfg->osd=0;
		else {
			fprintf(stderr, "Illegal xosd state %s\n",arg);
			_exit(1);
		}
	} /* }}} */
#endif /* HAVE_LIBXOSD */
	return;
} /* }}} */

void parse_option(int argc, char **argv) { /* {{{ */
	config cmd_cfg = {
		.nvram = NULL,
		.polltime = -1,
		.tpbcmd = NULL,
		.mixer = -1,
		.mixersteps = -1,
		.mixerdev = NULL,
		.verbose = -1
#ifdef HAVE_LIBXOSD
			,
		.osd = -1,
		.osdfont = NULL,
		.osdcolor = NULL,
		.osdtimeout = -1,
		.osdoffset = -1,
		.osdpos = -1
#endif /* HAVE_LIBXOSD */
	};

	while (1) {
		int option_index = 0;
		char *endptr;
		int c;
		static struct option long_options[] = {
			{"help",     0, NULL, 'h'},
			{"config",   1, NULL, 'c'},
			{"mixer",    1, NULL, 'm'},
#ifdef HAVE_LIBXOSD
			{"osd",      1, NULL, 'o'},
#endif /* HAVE_LIBXOSD */
			{"polltime", 1, NULL, 'p'},
			{"thinkpad", 1, NULL, 't'},
			{"verbose",  0, NULL, 'v'},
			{NULL,       0, NULL,   0}
		};

#ifdef HAVE_LIBXOSD
		c = getopt_long (argc, argv, "hc:m:o:p:t:v", long_options, &option_index);
#else
		c = getopt_long (argc, argv, "hc:m:p:t:v", long_options, &option_index);
#endif /* HAVE_LIBXOSD */
		if (c == -1)
			break;

		switch (c) {
			case 'h':
				print_usage(argv[0]);
				_exit(1);
				break;
			case 'c':
				if(parse_cfg_file(optarg)) {
					fprintf(stderr,"Unable to open config file %s: ",optarg);
					perror(NULL);
				}
				break;
			case 'm':
				set_value(CFG_MIXER, optarg, &cmd_cfg);
				break;
#ifdef HAVE_LIBXOSD
			case 'o':
				set_value(CFG_OSD, optarg, &cmd_cfg);
				break;
#endif /* HAVE_LIBXOSD */
			case 'p':
				set_value(CFG_POLLTIME, optarg, &cmd_cfg);
				break;
			case 't':
				set_value(CFG_THINKPAD, optarg, &cmd_cfg);
				break;
			case 'v':
				cmd_cfg.verbose=1;
				break;
			default:
				print_usage(argv[0]);
				_exit(1);
		}
	}

	/* override the config with values set on command line */
	override_cfg(&cmd_cfg);

	return;
} /* }}} */

void print_usage(const char *prog)/*{{{*/
{
	printf("%s 0.1 (C) 2002 Markus Braun \n\n",prog);
	printf("Usage: %s options\n\n",prog);
	printf("options:\n");
	printf("   -h, --help           display this help\n");
	printf("   -c, --config=FILE    read FILE as additional configuration file\n");
	printf("   -m, --mixer=STATE    use OSS software mixer [off]\n");
#ifdef HAVE_LIBXOSD
	printf("   -o, --osd=STATE      show on-screen display for volume, mute and brightness [on]\n");
#endif /* HAVE_LIBXOSD */
	printf("   -p, --poll=DELAY     set delay between polls in microseconds [%d]\n",POLLTIME);
	printf("   -t, --thinkpad=CMD   string with command and options that should be executed\n");
	printf("                        when ThinkPad button is pressed [none]\n");
	printf("   -v, --verbose        print information about pressed keys\n");
	return;
}/*}}}*/

/* vim:set sw=2:set ts=2: */
/* vim600:set fen:set fdm=marker:set fdl=0: */
