/* tpb - program to use the IBM ThinkPad(tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* includes {{{ */
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "config.h"

#if ENABLE_NLS
#include <libintl.h>
#endif /* ENABLE_NLS */

#ifdef HAVE_LIBXOSD
#include <xosd.h>
#endif /* HAVE_LIBXOSD */

#include "cfg.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{*/
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: cfg.c,v 1.4 2003/04/10 07:57:21 mbr Exp $";
#endif
/* }}} */

/* global variables  {{{ */
config cfg;
/* }}} */

int parse_cfg_file(char *name) { /* {{{ */
	FILE *file_desc;
	char keyword_buffer[BUFFER_SIZE];
	char argument_buffer[BUFFER_SIZE];
	int next_char;
	int keyword_buffer_ptr = 0;
	int argument_buffer_ptr = 0;
	enum mode mode = MODE_INDENT;
	config file_cfg;

	/* clear the config */
	clear_cfg(&file_cfg);

	/* open configuration file */
	if((file_desc = fopen(name, "r")) == NULL) {
		return -1;
	}

	/* loop until we reach the end of the file {{{ */
	while(mode != MODE_END) {
		next_char = fgetc(file_desc);
		switch(next_char) {
			//case '=':
			case ' ':
			case '\t':
				switch(mode) {
					case MODE_KEYWORD:
						mode = MODE_SEPARATOR;
						break;

					case MODE_ARGUMENT:
						if(argument_buffer_ptr < BUFFER_SIZE - 1) {
							argument_buffer[argument_buffer_ptr++] = next_char;
						}
						else {
							fputs(_("Argument in configuration too long!\n"), stderr);
							exit(1);
						}
						break;

					default:
						break;
				}
				break;

			case '\n':
				mode = MODE_INDENT;
				keyword_buffer[keyword_buffer_ptr] = '\0';
				argument_buffer[argument_buffer_ptr] = '\0';
				if(strlen(keyword_buffer) > 0 && strlen(argument_buffer) > 0) {
					argument_buffer_ptr--;
					while(argument_buffer[argument_buffer_ptr] == ' ' || argument_buffer[argument_buffer_ptr] == '\t') {
						argument_buffer[argument_buffer_ptr] = '\0';
						argument_buffer_ptr--;
					}
					set_value(keyword_buffer, argument_buffer, &file_cfg);
				}
				//if(strlen(keyword_buffer) >0 && strlen(argument_buffer)>0) printf("Keyword: \"%s\", Argument: \"%s\"\n",keyword_buffer,argument_buffer);
				keyword_buffer_ptr = 0;
				argument_buffer_ptr = 0;
				break;

			case '#':
				mode = MODE_COMMENT;
				break;

			case EOF:
				mode = MODE_END;
				break;

			default:
				switch(mode) {
					case MODE_INDENT:
						mode = MODE_KEYWORD;
					case MODE_KEYWORD:
						if(keyword_buffer_ptr < BUFFER_SIZE - 1) {
							keyword_buffer[keyword_buffer_ptr++] = tolower(next_char);
						}
						else {
							fputs(_("Keyword in configuration too long!\n"), stderr);
							exit(1);
						}
						break;

					case MODE_SEPARATOR:
						mode = MODE_ARGUMENT;
					case MODE_ARGUMENT:
						if(argument_buffer_ptr < BUFFER_SIZE - 1) {
							argument_buffer[argument_buffer_ptr++] = next_char;
						}
						else {
							fputs(_("Argument in configuration too long!\n"), stderr);
							exit(1);
						}
						break;

					default:
						break;
				}
				break;
		}
	} /* }}} */

	/* override the config with values from configuration file */
	override_cfg(&file_cfg);

	return 0;
} /* }}} */

void init_cfg(void) { /* {{{ */

	/* set all values to defaults */
	cfg.apm = DEFAULT_APM;
	cfg.powermgt = DEFAULT_POWERMGT;
	if((cfg.nvram = strdup(DEFAULT_NVRAMDEV)) == NULL) {
		fprintf(stderr, _("Not enough memory"));
		_exit(1);
	}
	cfg.polltime = DEFAULT_POLLTIME;
	cfg.tpbcmd = NULL;
	cfg.homecmd = NULL;
	cfg.searchcmd = NULL;
	cfg.mailcmd = NULL;
	cfg.callback = NULL;
	cfg.mixer = DEFAULT_MIXER;
	cfg.mixersteps = DEFAULT_MIXERSTEPS;
	if((cfg.mixerdev = strdup(DEFAULT_MIXERDEV)) == NULL) {
		fprintf(stderr, _("Not enough memory"));
		_exit(1);
	}
	cfg.verbose = DEFAULT_VERBOSE;
#ifdef HAVE_LIBXOSD
	cfg.osd = DEFAULT_OSD;
	if((cfg.osdfont = strdup(DEFAULT_OSDFONT)) == NULL) {
		fprintf(stderr, _("Not enough memory"));
		_exit(1);
	}
	if((cfg.osdcolor = strdup(DEFAULT_OSDCOLOR)) == NULL) {
		fprintf(stderr, _("Not enough memory"));
		_exit(1);
	}
	cfg.osdtimeout = DEFAULT_OSDTIMEOUT;
	cfg.osdshadow = DEFAULT_OSDSHADOW;
	cfg.osdvertical = DEFAULT_OSDVERTICAL;
	cfg.osdhorizontal = DEFAULT_OSDHORIZONTAL;
	cfg.osdpos = DEFAULT_OSDPOS;
#ifndef HAVE_LIBXOSD0
	cfg.osdalign = DEFAULT_OSDALIGN;
#endif /* HAVE_LIBXOSD0 */
#endif /* HAVE_LIBXOSD */

	return;
} /* }}} */

void clear_cfg(config *cfg) { /* {{{ */
		cfg->daemon = UNDEFINED;
		cfg->apm = UNDEFINED;
		cfg->powermgt = UNDEFINED;
		cfg->nvram = NULL;
		cfg->polltime = UNDEFINED;
		cfg->tpbcmd = NULL;
		cfg->homecmd = NULL;
		cfg->searchcmd = NULL;
		cfg->mailcmd = NULL;
		cfg->callback = NULL;
		cfg->mixer = UNDEFINED;
		cfg->mixersteps = UNDEFINED;
		cfg->mixerdev = NULL;
		cfg->verbose = UNDEFINED;
#ifdef HAVE_LIBXOSD
		cfg->osd = UNDEFINED;
		cfg->osdfont = NULL;
		cfg->osdcolor = NULL;
		cfg->osdtimeout = UNDEFINED;
		cfg->osdshadow = UNDEFINED;
		cfg->osdvertical = UNDEFINED;
		cfg->osdhorizontal = UNDEFINED;
		cfg->osdpos = UNDEFINED;
#ifndef HAVE_LIBXOSD0
		cfg->osdalign = UNDEFINED;
#endif /* HAVE_LIBXOSD0 */
#endif /* HAVE_LIBXOSD */

	return;
} /* }}} */

void override_cfg(config *master) { /* {{{ */
	/* override the defaults if the are set in master struct */
	if(master->daemon != UNDEFINED) {
		cfg.daemon = master->daemon;
	}

	if(master->apm != UNDEFINED) {
		cfg.apm = master->apm;
	}

	if(master->powermgt != UNDEFINED) {
		cfg.powermgt = master->powermgt;
	}

	if(master->nvram != NULL) {
		if(cfg.nvram != NULL) {
			free(cfg.nvram);
		}
		cfg.nvram = master->nvram;
	}

	if(master->polltime != UNDEFINED) {
		cfg.polltime = master->polltime;
	}

	if(master->tpbcmd != NULL) {
		if(cfg.tpbcmd != NULL) {
			free(cfg.tpbcmd);
		}
		cfg.tpbcmd = master->tpbcmd;
	}

	if(master->homecmd != NULL) {
		if(cfg.homecmd != NULL) {
			free(cfg.homecmd);
		}
		cfg.homecmd = master->homecmd;
	}

	if(master->searchcmd != NULL) {
		if(cfg.searchcmd != NULL) {
			free(cfg.searchcmd);
		}
		cfg.searchcmd = master->searchcmd;
	}

	if(master->mailcmd != NULL) {
		if(cfg.mailcmd != NULL) {
			free(cfg.mailcmd);
		}
		cfg.mailcmd = master->mailcmd;
	}

	if(master->callback != NULL) {
		if(cfg.callback != NULL) {
			free(cfg.callback);
		}
		cfg.callback = master->callback;
	}

	if(master->mixer != UNDEFINED) {
		cfg.mixer = master->mixer;
	}

	if(master->mixersteps != UNDEFINED) {
		cfg.mixersteps = master->mixersteps;
	}

	if(master->mixerdev != NULL) {
		if(cfg.mixerdev != NULL) {
			free(cfg.mixerdev);
		}
		cfg.mixerdev = master->mixerdev;
	}

	if(master->verbose != UNDEFINED) {
		cfg.verbose = master->verbose;
	}

#ifdef HAVE_LIBXOSD
	if(master->osd != UNDEFINED) {
		cfg.osd = master->osd;
	}

	if(master->osdfont != NULL) {
		if(cfg.osdfont != NULL) {
			free(cfg.osdfont);
		}
		cfg.osdfont = master->osdfont;
	}

	if(master->osdcolor != NULL) {
		if(cfg.osdcolor != NULL) {
			free(cfg.osdcolor);
		}
		cfg.osdcolor = master->osdcolor;
	}

	if(master->osdtimeout != UNDEFINED) {
		cfg.osdtimeout = master->osdtimeout;
	}

	if(master->osdshadow != UNDEFINED) {
		cfg.osdshadow = master->osdshadow;
	}

	if(master->osdvertical != UNDEFINED) {
		cfg.osdvertical = master->osdvertical;
	}

	if(master->osdhorizontal != UNDEFINED) {
		cfg.osdhorizontal = master->osdhorizontal;
	}

	if(master->osdpos != UNDEFINED) {
		cfg.osdpos = master->osdpos;
	}

#ifndef HAVE_LIBXOSD0
	if(master->osdalign != UNDEFINED) {
		cfg.osdalign = master->osdalign;
	}
#endif /* HAVE_LIBXOSD0 */
#endif /* HAVE_LIBXOSD */

	return;
} /* }}} */

void set_value(char *key, char *arg, config *cfg) { /* {{{ */
	char *endptr;

	if (strcmp(CFG_APM, key) == 0) { /* {{{ */
		int i = 0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_APM_ON, arg) == 0) {
			cfg->apm = STATE_ON;
		}
		else {
			if(strcmp(CFG_APM_OFF, arg) == 0) {
				cfg->apm = STATE_OFF;
			}
			else {
				fprintf(stderr, _("Illegal apm state: %s\n"), arg);
				_exit(1);
			}
		}
	} /* }}} */

	else if (strcmp(CFG_POWERMGT, key) == 0) { /* {{{ */
		int i=0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_POWERMGT_ON, arg) == 0) {
			cfg->powermgt = STATE_ON;
		}
		else {
			if(strcmp(CFG_POWERMGT_OFF, arg) == 0) {
				cfg->powermgt = STATE_OFF;
			}
			else {
				if(strcmp(CFG_POWERMGT_AUTO, arg) == 0) {
					cfg->powermgt = STATE_AUTO;
				}
				else {
					fprintf(stderr, _("Illegal powermgt state: %s\n"), arg);
					_exit(1);
				}
			}
		}
	} /* }}} */

	else if (strcmp(CFG_NVRAM, key) == 0) { /* {{{ */
		if(cfg->nvram != NULL) {
			free(cfg->nvram);
		}
		if((cfg->nvram=strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_POLLTIME, key) == 0) { /* {{{ */
		cfg->polltime = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->polltime < 0) {
			fprintf(stderr, _("Illegal polltime: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_THINKPAD, key) == 0) { /* {{{ */
		if(cfg->tpbcmd != NULL) {
			free(cfg->tpbcmd);
		}
		if((cfg->tpbcmd=strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_HOME, key) == 0) { /* {{{ */
		if(cfg->homecmd != NULL) {
			free(cfg->homecmd);
		}
		if((cfg->homecmd = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_SEARCH, key) == 0) { /* {{{ */
		if(cfg->searchcmd != NULL) {
			free(cfg->searchcmd);
		}
		if((cfg->searchcmd = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_MAIL, key) == 0) { /* {{{ */
		if(cfg->mailcmd != NULL) {
			free(cfg->mailcmd);
		}
		if((cfg->mailcmd = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_CALLBACK, key) == 0) { /* {{{ */
		if(cfg->callback != NULL) {
			free(cfg->callback);
		}
		if((cfg->callback = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_MIXER, key) == 0) { /* {{{ */
		int i = 0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_MIXER_ON, arg) == 0) {
			cfg->mixer = STATE_ON;
		}
		else {
			if(strcmp(CFG_MIXER_OFF, arg) == 0) {
				cfg->mixer = STATE_OFF;
			}
			else {
				fprintf(stderr, _("Illegal mixer state: %s\n"), arg);
				_exit(1);
			}
		}
	} /* }}} */

	else if (strcmp(CFG_MIXERSTEPS, key) == 0) { /* {{{ */
		cfg->mixersteps = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->mixersteps < 0) {
			fprintf(stderr, _("Illegal mixersteps: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_MIXERDEV, key) == 0) { /* {{{ */
		if(cfg->mixerdev != NULL) {
			free(cfg->mixerdev);
		}
		if((cfg->mixerdev = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

#ifdef HAVE_LIBXOSD
	else if (strcmp(CFG_OSDFONT, key) == 0) { /* {{{ */
		if(cfg->osdfont != NULL) {
			free(cfg->osdfont);
		}
		if((cfg->osdfont = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDCOLOR, key) == 0) { /* {{{ */
		if(cfg->osdcolor != NULL) {
			free(cfg->osdcolor);
		}
		if((cfg->osdcolor = strdup(arg)) == NULL) {
			fprintf(stderr, _("Not enough memory"));
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDTIMEOUT, key) == 0) { /* {{{ */
		cfg->osdtimeout = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0 || cfg->osdtimeout < 0) {
			fprintf(stderr, _("Illegal xosd timeout: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDOFFSET, key) == 0) { /* {{{ */
		cfg->osdvertical = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0) {
			fprintf(stderr, _("Illegal xosd offset: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDSHADOW, key) == 0) { /* {{{ */
		cfg->osdshadow = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0) {
			fprintf(stderr, _("Illegal xosd shadow offset: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDVERTICAL, key) == 0) { /* {{{ */
		cfg->osdvertical = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0) {
			fprintf(stderr, _("Illegal xosd vertical offset: %s\n"), arg);
			_exit(1);
		}
	} /* }}} */

	else if (strcmp(CFG_OSDHORIZONTAL, key) == 0) { /* {{{ */
#ifdef HAVE_LIBXOSD2
		cfg->osdhorizontal = strtol(arg, &endptr, 10);
		if(strlen(endptr) > 0) {
			fprintf(stderr, _("Illegal xosd horizontal offset: %s\n"), arg);
			_exit(1);
		}
#else
		fprintf(stderr, _("Sorry, OSDHORIZONTAL is only supported by xosd 2.0.0 and above.\n"));
#endif
	} /* }}} */

	else if (strcmp(CFG_OSDPOS, key) == 0) { /* {{{ */
		int i = 0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_OSDPOS_TOP, arg) == 0) {
			cfg->osdpos = XOSD_top;
		}
		else {
			if(strcmp(CFG_OSDPOS_MIDDLE, arg) == 0) {
#ifdef HAVE_LIBXOSD2
				cfg->osdpos = XOSD_middle;
#else
				fprintf(stderr, _("Sorry, OSDPOS MIDDLE is only supported by xosd 2.0.0 and above.\n"));
#endif
			}
			else {
				if(strcmp(CFG_OSDPOS_BOTTOM, arg) == 0) {
					cfg->osdpos = XOSD_bottom;
				}
				else {
					fprintf(stderr, _("Illegal xosd position: %s\n"), arg);
					_exit(1);
				}
			}
		}
	} /* }}} */

#ifndef HAVE_LIBXOSD0
	else if (strcmp(CFG_OSDALIGN, key) == 0) { /* {{{ */
		int i = 0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_OSDALIGN_LEFT, arg) == 0) {
			cfg->osdalign = XOSD_left;
		}
		else {
			if(strcmp(CFG_OSDALIGN_CENTER, arg) == 0) {
				cfg->osdalign = XOSD_center;
			}
			else {
				if(strcmp(CFG_OSDALIGN_RIGHT, arg) == 0) {
					cfg->osdalign = XOSD_right;
				}
				else {
					fprintf(stderr, _("Illegal xosd alignment: %s\n"), arg);
					_exit(1);
				}
			}
		}
	} /* }}} */
#endif /* HAVE_LIBXOSD0 */

	else if (strcmp(CFG_OSD, key) == 0) { /* {{{ */
		int i = 0;
		while(arg[i]) {
			arg[i] = tolower(arg[i]);
			i++;
		}
		if(strcmp(CFG_OSD_ON, arg) == 0) {
			cfg->osd = STATE_ON;
		}
		else {
			if(strcmp(CFG_OSD_OFF, arg) == 0) {
				cfg->osd = STATE_OFF;
			}
			else {
				fprintf(stderr, _("Illegal xosd state: %s\n"), arg);
				_exit(1);
			}
		}
	} /* }}} */
#endif /* HAVE_LIBXOSD */

	return;
} /* }}} */

void parse_option(int argc, char **argv) { /* {{{ */
	config cmd_cfg;

	/* clear the config */
	clear_cfg(&cmd_cfg);

	while (1) {
		int option_index = 0;
		int next_option;
		static struct option long_options[] = {
			{"help",     0, NULL, 'h'},
			{"daemon",   0, NULL, 'd'},
			{"apm",      1, NULL, 'A'},
			{"powermgt", 1, NULL, 'P'},
			{"config",   1, NULL, 'c'},
			{"callback", 1, NULL, 'C'},
			{"mixer",    1, NULL, 'm'},
#ifdef HAVE_LIBXOSD
			{"osd",      1, NULL, 'o'},
#endif /* HAVE_LIBXOSD */
			{"polltime", 1, NULL, 'p'},
			{"thinkpad", 1, NULL, 't'},
			{"home",     1, NULL, 'H'},
			{"search",   1, NULL, 'S'},
			{"mail",     1, NULL, 'M'},
			{"verbose",  0, NULL, 'v'},
			{NULL,       0, NULL,   0}
		};

#ifdef HAVE_LIBXOSD
		next_option = getopt_long (argc, argv, "hdA:P:c:C:m:o:p:t:H:S:M:v", long_options, &option_index);
#else
		next_option = getopt_long (argc, argv, "hdA:P:c:C:m:p:t:H:S:M:v", long_options, &option_index);
#endif /* HAVE_LIBXOSD */

		if (next_option == -1) {
			break;
		}

		switch (next_option) {
			case 'h':
				print_usage(argv[0]);
				_exit(1);
				break;

			case 'd':
				cmd_cfg.daemon = STATE_ON;
				break;

			case 'A':
				set_value(CFG_APM, optarg, &cmd_cfg);
				break;

			case 'P':
				set_value(CFG_POWERMGT, optarg, &cmd_cfg);
				break;

			case 'c':
				if(parse_cfg_file(optarg)) {
					fprintf(stderr, _("Unable to open config file %s: "), optarg);
					perror(NULL);
				}
				break;

			case 'C':
				set_value(CFG_CALLBACK, optarg, &cmd_cfg);
				break;

			case 'm':
				set_value(CFG_MIXER, optarg, &cmd_cfg);
				break;

#ifdef HAVE_LIBXOSD
			case 'o':
				set_value(CFG_OSD, optarg, &cmd_cfg);
				break;
#endif /* HAVE_LIBXOSD */

			case 'p':
				set_value(CFG_POLLTIME, optarg, &cmd_cfg);
				break;

			case 't':
				set_value(CFG_THINKPAD, optarg, &cmd_cfg);
				break;

			case 'H':
				set_value(CFG_HOME, optarg, &cmd_cfg);
				break;

			case 'S':
				set_value(CFG_SEARCH, optarg, &cmd_cfg);
				break;

			case 'M':
				set_value(CFG_MAIL, optarg, &cmd_cfg);
				break;

			case 'v':
				cmd_cfg.verbose = STATE_ON;
				break;

			default:
				print_usage(argv[0]);
				_exit(1);
		}
	}

	/* override the config with values set on command line */
	override_cfg(&cmd_cfg);

	return;
} /* }}} */

void print_usage(const char *prog) /* {{{ */
{
	printf("%s " VERSION " (C) 2002-2003 Markus Braun \n\n", prog);
	printf(_("Usage: %s options\n\n"), prog);
	printf(_("options:\n"));
	printf(_("   -h, --help           display this help\n"));
	printf(_("   -d, --daemon         start in daemon mode\n"));
	printf(_("   -c, --config=FILE    read FILE as additional configuration file\n"));
	printf(_("   -m, --mixer=STATE    use OSS software mixer [off]\n"));
#ifdef HAVE_LIBXOSD
	printf(_("   -o, --osd=STATE      show informations as on-screen display [on]\n"));
#endif /* HAVE_LIBXOSD */
	printf(_("   -p, --poll=DELAY     set delay between polls in microseconds [%d]\n"), DEFAULT_POLLTIME);
	printf(_("   -t, --thinkpad=CMD   string with command and options that should be executed\n"));
	printf(_("                        when ThinkPad button is pressed [none]\n"));
	printf(_("   -H, --home=CMD       command and options for Home button [none]\n"));
	printf(_("   -S, --search=CMD     command and options for Search button [none]\n"));
	printf(_("   -M, --mail=CMD       command and options for Mail button [none]\n"));
	printf(_("   -C, --callback=CMD   string with command that should be executed for each\n"));
	printf(_("                        pressed button. It is called with pressed button as\n"));
	printf(_("                        first argument and new state as second [none]\n"));
	printf(_("   -A, --apm=STATE      poll /proc/apm for apm information [off]\n"));
	printf(_("   -P, --powermgt=STATE display of power management messages [auto]\n"));
	printf(_("   -v, --verbose        print information about pressed keys\n"));
	return;
} /* }}} */

/* vim:set sw=2:set ts=2: */
/* vim600:set fen:set fdm=marker:set fdl=0: */
