/* tpb - program to use the IBM ThinkPad(tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* includes {{{ */
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <wait.h>
#include "config.h"

#if ENABLE_NLS
#include <libintl.h>
#endif /* ENABLE_NLS */

#ifdef HAVE_LIBXOSD
#include <xosd.h>
#endif /* HAVE_LIBXOSD */

#include "cfg.h"
#include "tpb.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{ */
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: tpb.c,v 1.5 2003/03/13 07:19:51 mbr Exp $";
#endif
/* }}} */

/* global variables  {{{ */
extern config cfg;
/* }}} */

/* function prototypes {{{ */
#ifdef ENABLE_NLS
void init_i18n(void);
#endif /* ENABLE_NLS */
void daemonize(void);
#ifdef HAVE_LIBXOSD
xosd *init_xosd(void);
#endif /* HAVE_LIBXOSD */
int get_state(t_nvram_state *nvram_state);
int get_apm_state(t_nvram_state *nvram_state);
int fork_app(char * cmd);
void set_nvram_volume_level(t_nvram_state *nvram_state);
int change_volume(int change);
int apmiser_running(void);
/* }}} */

int main(int argc, char **argv) /* {{{ */
{
	t_nvram_state nvram_state, last_nvram_state;
	int vol = 0;
	int display_state = 1;
	char *home;
	char *cfg_file;
	char *msg = "";
	char callbackcmd[CALLBACKCMDLENGTH];
	struct sigaction signal_action;
#ifdef HAVE_LIBXOSD
	xosd *osd_ptr = NULL;
#endif /* HAVE_LIBXOSD */

	/* ignore SIG_CHLD */
	signal_action.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
	signal_action.sa_flags = SA_NOCLDWAIT;
#else
	signal_action.sa_flags = 0;
#endif
	sigemptyset(&signal_action.sa_mask);
	sigaction(SIGCHLD, &signal_action, NULL);

	/* initialize i18n */
#ifdef ENABLE_NLS
	init_i18n();
#endif /* ENABLE_NLS */

	/* initialize config */
	init_cfg();

	/* read global config file */
	parse_cfg_file(GLOBAL_CONFIG_FILE);

	/* read user config file */
	if((home = getenv("HOME")) != NULL) {
		cfg_file = (char *)malloc(strlen(home) + strlen("/") + strlen(PRIVATE_CONFIG_FILE) + 1);
		strcpy(cfg_file, home);
		strcat(cfg_file, "/");
		strcat(cfg_file, PRIVATE_CONFIG_FILE);
		parse_cfg_file(cfg_file);
		free(cfg_file);
	}

	/* parse options */
	parse_option(argc, argv);

	/* become a daemon if requested */
	if(cfg.daemon == STATE_ON) {
		daemonize();
	}

	/* initialize osd */
#ifdef HAVE_LIBXOSD
	if(cfg.osd == STATE_ON) {
		osd_ptr = init_xosd();
	}
#endif /* HAVE_LIBXOSD */

	/* to initialize struct */
	if(get_state(&nvram_state)) {
		_exit(1);
	}

	/* initialize volume */
	if(cfg.mixer == STATE_ON) {
		set_nvram_volume_level(&nvram_state);
		vol = change_volume(0);
	}
	else {
		vol = nvram_state.volume_level * 100 / 14;
	}

	while(1) {
		/* save last state and get new one */
		memcpy(&last_nvram_state, &nvram_state, sizeof(t_nvram_state));
		if(get_state(&nvram_state)) {
			_exit(1);
		}

		/* determine the state of the Thinkpad button  {{{ */
		if(nvram_state.thinkpad_toggle != last_nvram_state.thinkpad_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts(_("ThinkPad button pressed"));
			}
			if(cfg.tpbcmd != NULL) {
				if(fork_app(cfg.tpbcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s thinkpad pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of zoom  {{{ */
		if(nvram_state.zoom_toggle != last_nvram_state.zoom_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf(_("Zoom is %s\n"), nvram_state.zoom_toggle ? _("on") : _("off"));
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s zoom %s", cfg.callback, nvram_state.zoom_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.zoom_toggle ? _("Zoom on") : _("Zoom off"));
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of the home button  {{{ */
		if(nvram_state.home_toggle != last_nvram_state.home_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts(_("Home button pressed"));
			}
			if(cfg.homecmd != NULL) {
				if(fork_app(cfg.homecmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s home pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of the search button  {{{ */
		if(nvram_state.search_toggle != last_nvram_state.search_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts(_("Search button pressed"));
			}
			if(cfg.searchcmd != NULL) {
				if(fork_app(cfg.searchcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s search pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of the mail button  {{{ */
		if(nvram_state.mail_toggle != last_nvram_state.mail_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts(_("Mail button pressed"));
			}
			if(cfg.mailcmd != NULL) {
				if(fork_app(cfg.mailcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s mail pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of ThinkLight {{{ */
		if(nvram_state.thinklight_toggle != last_nvram_state.thinklight_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf(_("ThinkLight is %s\n"), nvram_state.thinklight_toggle ? _("on") : _("off"));
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s thinklight %s", cfg.callback, nvram_state.thinklight_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.thinklight_toggle ? _("ThinkLight on") : _("ThinkLight off"));
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of display  {{{ */
		if(nvram_state.display_toggle != last_nvram_state.display_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {

			/* Some thinkpads have no hardware support to switch lcd/crt. They also
			 * don't reflect the current state in nvram_state.display_state. So, if
			 * nvram_state.display_toggle changes, but nvram_state.display_state does
			 * not change, simulate the display state.
			 */
			if(nvram_state.display_state == last_nvram_state.display_state) {
				display_state = display_state % 3 + 1;
			}
			else {
				display_state = nvram_state.display_state;
			}

			switch (display_state & 0x03) {
				case 0x1:
					msg = _("LCD on, CRT off");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display lcd", cfg.callback);
					break;

				case 0x2:
					msg = _("LCD off, CRT on");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display crt", cfg.callback);
					break;

				case 0x3:
					msg = _("LCD on, CRT on");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display both", cfg.callback);
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf(_("Display changed: %s\n"), msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, _("Display"));
				xosd_display(osd_ptr, 1, XOSD_string, msg);
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of hv expansion  {{{ */
		if(nvram_state.expand_toggle != last_nvram_state.expand_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf(_("HV Expansion is %s\n"), nvram_state.expand_toggle & 0x01 ? _("on") : _("off"));
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s expand %s", cfg.callback, nvram_state.expand_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.expand_toggle ? _("HV Expansion on") : _("HV Expansion off"));
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of the brightness buttons {{{ */
		if(nvram_state.brightness_level != last_nvram_state.brightness_level) {
			if(cfg.verbose == STATE_ON) {
				printf(_("Brightness changed: Level %d\n"), nvram_state.brightness_level * 100 / 7);
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s brightness %d", cfg.callback, nvram_state.brightness_level * 100 / 7);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
		if(nvram_state.brightness_toggle != last_nvram_state.brightness_toggle) {
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, _("Brightness"));
				xosd_display(osd_ptr, 1, XOSD_percentage, nvram_state.brightness_level * 100 / 7);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the volume buttons {{{ */
		if(nvram_state.volume_level != last_nvram_state.volume_level) {
			if(cfg.mixer == STATE_ON) {
				/* if we use the DEFAULT_MIXERSTEPS, we don't need to modify the nvram */
				if(cfg.mixersteps == DEFAULT_MIXERSTEPS) {
					vol = change_volume((MAXVOLUME / cfg.mixersteps * nvram_state.volume_level) - vol);
				}
				else
				{
					vol = change_volume(MAXVOLUME / cfg.mixersteps * (nvram_state.volume_level - last_nvram_state.volume_level)); /* nvram_state.volume_level-last_nvram_state.volume_level gives the direction */
					printf("change %f Level %d\n",MAXVOLUME / cfg.mixersteps * (nvram_state.volume_level - last_nvram_state.volume_level), vol);
					set_nvram_volume_level(&nvram_state);
				}
			}
			else {
				vol = nvram_state.volume_level * 100 / 14;
			}
			if(cfg.verbose == STATE_ON) {
				printf(_("Volume changed: Level %d\n"), vol);
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s volume %d", cfg.callback, vol);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
    /* show volume every time a volume button is pressed and not mute */
		if(nvram_state.volume_toggle != last_nvram_state.volume_toggle &&
			 !nvram_state.mute_toggle) {
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, _("Volume"));
				xosd_display(osd_ptr, 1, XOSD_percentage, vol);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the mute button {{{ */
		if(nvram_state.mute_toggle != last_nvram_state.mute_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf("%s\n", nvram_state.mute_toggle ? _("Mute on") : _("Mute off"));
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s mute %s", cfg.callback, nvram_state.mute_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
		if(nvram_state.mute_toggle != last_nvram_state.mute_toggle ||
			 (nvram_state.volume_toggle != last_nvram_state.volume_toggle && last_nvram_state.mute_toggle)) {
			if(cfg.mixer == STATE_ON) {
				if(nvram_state.mute_toggle) {
					change_volume(-vol); /* mute */
				}
				else {
					change_volume(vol); /* unmute */
				}
			}

			if(osd_ptr != NULL) {
				if(nvram_state.mute_toggle) {
					xosd_display(osd_ptr, 0, XOSD_string, _("Mute on"));
					xosd_display(osd_ptr, 1, XOSD_percentage, 0);
				}
				else {
					xosd_display(osd_ptr, 0, XOSD_string, _("Mute off"));
					xosd_display(osd_ptr, 1, XOSD_percentage, vol);
				}
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of power {{{ */
		if(nvram_state.ac_state != last_nvram_state.ac_state) {
			if(cfg.verbose == STATE_ON) {
				printf(_("Power line changed: %s\n"), nvram_state.ac_state ? _("AC connected") : _("AC disconnected"));
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s ac_power %s", cfg.callback, nvram_state.ac_state ? "connected" : "disconnected");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.ac_state ? _("AC connected") : _("AC disconnected"));
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode AC {{{ */
		if(nvram_state.powermgt_ac != last_nvram_state.powermgt_ac) {
			switch(nvram_state.powermgt_ac) {
				case 0x4:
					msg = _("PM AC high");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac high", cfg.callback);
					break;
					
				case 0x2:
					msg = _("PM AC auto");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac auto", cfg.callback);
					break;
					
				case 0x1:
					msg = _("PM AC manual");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac manual", cfg.callback);
					break;
					
				default:
					msg = _("PM AC unknown");
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf(_("Power management mode AC changed: %s\n"), msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				if (cfg.powermgt == STATE_ON ||  (cfg.powermgt == STATE_AUTO && !apmiser_running())) {
					xosd_display(osd_ptr, 0, XOSD_string, msg);
					xosd_display(osd_ptr, 1, XOSD_string, "");
				}
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode battery {{{ */
		if(nvram_state.powermgt_battery != last_nvram_state.powermgt_battery) {
			switch(nvram_state.powermgt_battery) {
				case 0x4:
					msg = _("PM battery high");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery high", cfg.callback);
					break;

				case 0x2:
					msg = _("PM battery auto");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery auto", cfg.callback);
					break;

				case 0x1:
					msg = _("PM battery manual");
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery manual", cfg.callback);
					break;

				default:
					msg = _("PM battery unknown");
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf(_("Power management mode battery changed: %s\n"), msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				if (cfg.powermgt == STATE_ON ||  (cfg.powermgt == STATE_AUTO && !apmiser_running())) {
					xosd_display(osd_ptr, 0, XOSD_string, msg);
					xosd_display(osd_ptr, 1, XOSD_string, "");
				}
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* sleep for polltime */
		usleep(cfg.polltime);
	}

	_exit(0); /* never reached */
} /* }}} */

#ifdef ENABLE_NLS
void init_i18n(void) /* {{{ */
{
	setlocale (LC_ALL, "");              
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	return;
} /* }}} */
#endif /* ENABLE_NLS */

void daemonize(void) /* {{{ */
{
	int filedes;

	/* code inspired by comp.unix.programmer FAQ */
	switch (fork())
	{
		case 0:
			break;
		case -1:
			fprintf(stderr, _("Unable to fork daemon"));
			perror(NULL);
			_exit(-1);
		default:
			_exit(0);          /* exit the original process */
	}

	/* create a new session */
	if (setsid() < 0) {
		_exit(-1);
	}

	switch (fork())
	{
		case 0:
			break;
		case -1:
			fprintf(stderr, _("Unable to fork daemon"));
			perror(NULL);
			_exit(-1);
		default:
			_exit(0);
	}

	if(cfg.verbose == STATE_ON) {
		puts(_("Daemon started."));
	}

	/* change to root directory */
	chdir("/");

	/* close all filedescriptors */
	for(filedes = 0 ; filedes < sysconf(_SC_OPEN_MAX) ; filedes++) {
		close(filedes);
	}

	/* open /dev/null as file descriptor 0 */
	open("/dev/null",O_RDWR);

	/* duplicate file descriptor 0 to file descriptor 1 (next free) */
	dup(0);

	/* duplicate file descriptor 0 to file descriptor 2 (next free) */
	dup(0);

	return;
} /* }}} */

#ifdef HAVE_LIBXOSD
xosd *init_xosd(void) /* {{{ */
{
	xosd *osd_ptr = NULL;

#ifdef HAVE_LIBXOSD0
	osd_ptr = xosd_init(DEFAULT_OSDFONT, DEFAULT_OSDCOLOR, DEFAULT_OSDTIMEOUT,
											DEFAULT_OSDPOS, DEFAULT_OSDVERTICAL, DEFAULT_OSDSHADOW);
#elif HAVE_LIBXOSD1
	osd_ptr = xosd_init(DEFAULT_OSDFONT, DEFAULT_OSDCOLOR, DEFAULT_OSDTIMEOUT,
											DEFAULT_OSDPOS, DEFAULT_OSDVERTICAL, DEFAULT_OSDSHADOW, DEFAULT_OSDLINES);
#elif HAVE_LIBXOSD2
	osd_ptr = xosd_create(DEFAULT_OSDLINES);
#endif

	if(osd_ptr == NULL) {
		fprintf(stderr, _("Unable to initialize xosd. Running without onsceen display.\n"));
	}
	else {
		/* initialize font */
		if(xosd_set_font(osd_ptr, cfg.osdfont)) {
			if(xosd_set_font(osd_ptr, DEFAULT_OSDFONT)) {
				/* even the default font is not available */
				fprintf(stderr, _("Invalid xosd font \"%s\". Even the default font \"%s\" "),
								cfg.osdfont, DEFAULT_OSDFONT);
				fprintf(stderr, _("is not available. Please add an existing font to your %s file.\n"),
								GLOBAL_CONFIG_FILE);
				fprintf(stderr, _("Running without onsceen display.\n"));

#if defined(HAVE_LIBXOSD0) || defined(HAVE_LIBXOSD1)
				/* Disabled because of a bug in xosd lib
				 * xosd_uninit(osd_ptr);
				 */
#elif HAVE_LIBXOSD2
				/* Disabled because of a bug in xosd lib
				 * xosd_destroy(osd_ptr);
				 */
#endif

				return NULL;
			}
			else {
				/* we run with the default font */
				fprintf(stderr, _("Invalid xosd font \"%s\". Running with the default \"%s\".\n"),
								cfg.osdfont, DEFAULT_OSDFONT);
			}
		}

		/* initialize color */
		if(xosd_set_colour(osd_ptr, cfg.osdcolor)) {
			fprintf(stderr, _("Invalid xosd color \"%s\". Running with the default \"%s\".\n"),
							cfg.osdcolor, DEFAULT_OSDCOLOR);
			xosd_set_colour(osd_ptr, DEFAULT_OSDCOLOR);
		}

		/* initialize timeout */
		if(xosd_set_timeout(osd_ptr, cfg.osdtimeout)) {
			fprintf(stderr, _("Invalid xosd timeout \"%d\". Running with the default \"%d\".\n"),
							cfg.osdtimeout, DEFAULT_OSDTIMEOUT);
			xosd_set_timeout(osd_ptr, DEFAULT_OSDTIMEOUT);
		}

		/* initialize position */
		if(xosd_set_pos(osd_ptr, cfg.osdpos)) {
			fprintf(stderr, _("Invalid xosd position \"%d\". Running with the default \"%d\".\n"),
							cfg.osdpos, DEFAULT_OSDPOS);
			xosd_set_pos(osd_ptr, DEFAULT_OSDPOS);
		}

#ifdef HAVE_LIBXOSD2
		/* initialize horizontal offset */
		if(xosd_set_horizontal_offset(osd_ptr, cfg.osdhorizontal)) {
			fprintf(stderr, _("Invalid horizontal xosd offset \"%d\". Running with the default \"%d\".\n"),
							cfg.osdhorizontal, DEFAULT_OSDHORIZONTAL);
			xosd_set_horizontal_offset(osd_ptr, DEFAULT_OSDHORIZONTAL);
		}

		/* initialize vertical offset */
		if(xosd_set_vertical_offset(osd_ptr, cfg.osdvertical)) {
			fprintf(stderr, _("Invalid vertical xosd offset \"%d\". Running with the default \"%d\".\n"),
							cfg.osdvertical, DEFAULT_OSDVERTICAL);
			xosd_set_vertical_offset(osd_ptr, DEFAULT_OSDVERTICAL);
		}
#else
		/* initialize vertical offset */
		if(xosd_set_offset(osd_ptr, cfg.osdvertical)) {
			fprintf(stderr, _("Invalid vertical xosd offset \"%d\". Running with the default \"%d\".\n"),
							cfg.osdvertical, DEFAULT_OSDVERTICAL);
			xosd_set_offset(osd_ptr, DEFAULT_OSDVERTICAL);
		}
#endif

		/* initialize shadow offset */
		if(xosd_set_shadow_offset(osd_ptr, cfg.osdshadow)) {
			fprintf(stderr, _("Invalid shadow xosd offset \"%d\". Running with the default \"%d\".\n"),
							cfg.osdshadow, DEFAULT_OSDSHADOW);
			xosd_set_shadow_offset(osd_ptr, DEFAULT_OSDSHADOW);
		}

#ifndef HAVE_LIBXOSD0
		/* initialize alignment */
		if(xosd_set_align(osd_ptr, cfg.osdalign)) {
			fprintf(stderr, _("Invalid xosd alignment \"%d\". Running with default \"%d\".\n"),
							cfg.osdalign, DEFAULT_OSDALIGN);
			xosd_set_align(osd_ptr, DEFAULT_OSDALIGN);
		}
#endif /* HAVE_LIBXOSD0 */
	}

	return osd_ptr;
} /* }}} */
#endif /* HAVE_LIBXOSD */

int get_state(t_nvram_state *nvram_state) /* {{{ */
{
	int  fdsc;
	char buffer[114];

	/* open nvram for reading */
	/* must use open/close because seek is not supported by nvram */
	if((fdsc=open(cfg.nvram, O_RDONLY|O_NONBLOCK)) == -1) {
		fprintf(stderr, _("Unable to open device %s: "), cfg.nvram);
		perror(NULL);
		return -1;
	}

	/* read nvram */
	if(read(fdsc, buffer, sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr, _("Unable to read from device %s: "), cfg.nvram);
		perror(NULL);
		return -1;
	}

	/* close nvram device */
	if(close(fdsc) == -1) {
		fprintf(stderr, _("Unable to close device %s: "), cfg.nvram);
		perror(NULL);
		return -1;
	}

	nvram_state->thinkpad_toggle   = ( buffer[0x57] & 0x08) >> 3;
	nvram_state->zoom_toggle       = (~buffer[0x57] & 0x20) >> 5;
	nvram_state->display_toggle    = ( buffer[0x57] & 0x40) >> 6;
	nvram_state->home_toggle       = ( buffer[0x56] & 0x01);
	nvram_state->search_toggle     = ( buffer[0x56] & 0x02) >> 1;
	nvram_state->mail_toggle       = ( buffer[0x56] & 0x04) >> 2;
	nvram_state->thinklight_toggle = ( buffer[0x58] & 0x10) >> 4;
	nvram_state->hibernate_toggle  = ( buffer[0x58] & 0x01);
	nvram_state->display_state     = ( buffer[0x59] & 0x03);
	nvram_state->expand_toggle     = ( buffer[0x59] & 0x10) >> 4;
	nvram_state->brightness_level  = ( buffer[0x5E] & 0x07);
	nvram_state->brightness_toggle = ( buffer[0x5E] & 0x20) >> 5;
	nvram_state->volume_level      = ( buffer[0x60] & 0x0f);
	nvram_state->volume_toggle     = ( buffer[0x60] & 0x80) >> 7;
	nvram_state->mute_toggle       = ( buffer[0x60] & 0x40) >> 6;
	nvram_state->powermgt_ac       = ( buffer[0x39] & 0x07);
	nvram_state->powermgt_battery  = ( buffer[0x39] & 0x38) >> 3;

	if(cfg.apm == STATE_ON) {
		get_apm_state(nvram_state);
	}

	return 0;
} /* }}} */

int get_apm_state(t_nvram_state *nvram_state) {  /* {{{ */
	int i;
	int proc_apm;
	char buffer[38];
	char *tokens[9];

	/* Read the state of the ac line from proc filesystem.
	 * Documentation of /proc/apm from linux kernel (/usr/src/linux/arch/i386/kernel/apm.c)
	 * 
	 * 0) Linux driver version (this will change if format changes)
	 * 1) APM BIOS Version.  Usually 1.0, 1.1 or 1.2.              
	 * 2) APM flags from APM Installation Check (0x00):            
	 *    bit 0: APM_16_BIT_SUPPORT                                
	 *    bit 1: APM_32_BIT_SUPPORT                                
	 *    bit 2: APM_IDLE_SLOWS_CLOCK                              
	 *    bit 3: APM_BIOS_DISABLED                                 
	 *    bit 4: APM_BIOS_DISENGAGED                               
	 * 3) AC line status                                           
	 *    0x00: Off-line                                           
	 *    0x01: On-line                                            
	 *    0x02: On backup power (BIOS >= 1.1 only)                 
	 *    0xff: Unknown                                            
	 * 4) Battery status                                           
	 *    0x00: High                                               
	 *    0x01: Low                                                
	 *    0x02: Critical                                           
	 *    0x03: Charging                                           
	 *    0x04: Selected battery not present (BIOS >= 1.2 only)    
	 *    0xff: Unknown                                            
	 * 5) Battery flag                                             
	 *    bit 0: High                                              
	 *    bit 1: Low                                               
	 *    bit 2: Critical                                          
	 *    bit 3: Charging                                          
	 *    bit 7: No system battery                                 
	 *    0xff: Unknown                                            
	 * 6) Remaining battery life (percentage of charge):           
	 *    0-100: valid                                             
	 *    -1: Unknown                                              
	 * 7) Remaining battery life (time units):                     
	 *    Number of remaining minutes or seconds                   
	 *    -1: Unknown                                              
	 * 8) min = minutes; sec = seconds */                          

	 /* open /proc/apm */
	 if((proc_apm = open("/proc/apm", O_RDONLY)) == -1) {
		 return -1;
	 }

	 /* read apm state */
	 if(read(proc_apm, buffer, sizeof(buffer)) != sizeof(buffer)) {
		 close(proc_apm);
		 return -1;
	 }

	 /* no need to keep file open */
	 close(proc_apm);

	 /* tokenize the apm string */
	 tokens[0] = strtok(buffer, " ");
	 for(i = 1 ; i < sizeof(tokens)/sizeof(char*) ; i++) {
		 tokens[i] = strtok(NULL, " ");
	 }

	 /* determine the AC line status */
	 switch(strtol(tokens[3], NULL, 16)) {
		 case 0x00:
			 nvram_state->ac_state = STATE_OFF;
			 break;

		 case 0x01:
			 nvram_state->ac_state = STATE_ON;
			 break;
	 }

	 return 0;
} /* }}} */
	
int fork_app(char * cmd) { /* {{{ */
	//char *args[4];
	//args[0] = "sh";
	//args[1] = "-c";
	//args[2] = cmd;
	//args[3] = 0;

	switch(fork()) {
		case -1:
			return -1;
			break;

		case 0:
			{
				setsid(); /* children should not be killed if tpb ends */
				//execv("/bin/sh", args);
				system(cmd);
				_exit(0);
				return 127;
			}
			break;

		default:
			return 0;
			break;
	}
	return -1; /* never reached */
} /* }}} */

void set_nvram_volume_level(t_nvram_state *nvram_state) { /* {{{ */
	int  fdsc;
	char buffer;

	/* only use writeback to nvram when cfg.mixersteps is different from DEFAULT_MIXERSTEPS */
	if(cfg.mixersteps != DEFAULT_MIXERSTEPS) {
		/* open nvram */
		if((fdsc = open(cfg.nvram, O_RDWR|O_NONBLOCK)) == -1) {
			fprintf(stderr, _("Unable to open device %s: "), cfg.nvram);
			perror(NULL);
			fprintf(stderr, _("To use mixersteps other than %d you need write access to %s.\n"), cfg.mixersteps, cfg.nvram);
			_exit(1);
		}

		/* jump to volume section */
		if(lseek(fdsc, 0x60, SEEK_SET) == -1 ) {
			fprintf(stderr, _("Unable to seek device %s: "), cfg.nvram);
			perror(NULL);
			_exit(1);
		}

		/* read nvram */
		if(read(fdsc, &buffer,sizeof(buffer)) != sizeof(buffer)) {
			fprintf(stderr, _("Unable to read from device %s: "), cfg.nvram);
			perror(NULL);
			_exit(1);
		}

		nvram_state->volume_level = 0x07; /* set volume_level to the value we write back to nvram */
		buffer &= 0xf0;
		buffer |= nvram_state->volume_level;

		/* jump to volume section */
		if(lseek(fdsc, 0x60, SEEK_SET) == -1 ) {
			fprintf(stderr, _("Unable to seek device %s: "), cfg.nvram);
			perror(NULL);
			_exit(1);
		}

		/* write std value for volume */
		if(write(fdsc, &buffer, sizeof(buffer)) != sizeof(buffer)) {
			fprintf(stderr, _("Unable to write to device %s: "), cfg.nvram);
			perror(NULL);
			_exit(1);
		}

		close(fdsc);
	}

	return;

} /* }}} */

int change_volume(int change) { /* {{{ */
	int mixer;
	int volume;
	int left,right;

	/* open mixer */
	if((mixer = open(cfg.mixerdev, O_RDWR)) == -1) {
		fprintf(stderr, _("Unable to open mixer device %s: "), cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* read mixer volume */
	if(ioctl(mixer, SOUND_MIXER_READ_VOLUME, &volume) == -1) {
		fprintf(stderr, _("Unable to read volume from mixer device %s: "), cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* adjust volume */
	left = (volume & 0xff) + change;
	right = ((volume >> 8) & 0xff) + change;
	if(left < 0) {
		left = 0;
	}
	if(right < 0) {
		right = 0;
	}
	if(left > MAXVOLUME) {
		left = MAXVOLUME;
	}
	if(right > MAXVOLUME) {
		right = MAXVOLUME;
	}
	volume = left | (right << 8);

	/* write volume back to mixer */
	if(ioctl(mixer, SOUND_MIXER_WRITE_VOLUME, &volume) == -1) {
		fprintf(stderr, _("Unable to write volume to mixer device %s: "), cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* close mixer device */
	if(close(mixer) == -1 ) {
		fprintf(stderr, _("Unable to close mixer device %s: "), cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* calc volume percentage and return it */
	return ((left + right) / 2) * 100 / MAXVOLUME;

} /* }}} */

int apmiser_running(void) { /* {{{ */
	/* code inspired by comp.unix.programmer FAQ */
	char line[133];
	char *linep;
	char *token;
	char *cmd = NULL;
	FILE *fp;

	/* open command like a file */
	fp = popen("ps -e 2>/dev/null", "r");
	if (fp == NULL) {
		return 0;
	}

	/* get header */
	if (fgets(line, sizeof(line), fp) == NULL) {
		pclose(fp);
		return 0;
	}

	/* determine column of command name */
	linep = line;
	while(cmd == NULL)
	{
		if ((token = strtok(linep, " \t\n")) == NULL) {
			pclose(fp);
			return 0;
		}
		linep = NULL;

		if (strcmp("COMMAND", token) == 0 || strcmp("CMD", token) == 0) {
			cmd = token;
		}
	}

	/* check if any of the commands is apmiser */
	while(fgets(line, sizeof line, fp) != NULL) {
		if (strstr(strtok(cmd, " \t\n"), "apmiser") != NULL) {
			pclose(fp);
			return 1;
		}
	}

	pclose(fp);

	return 0;
} /* }}} */

/* vim:set sw=2:set ts=2: */
/* vim600:set fen:set fdm=marker:set fdl=0: */
