--------------------------------------------------------------------------------
-- Trombi - Copyright 2007-2008 Louis Paternault
-- 
-- This file is part of Trombi.
-- 
-- Trombi is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
-- 
-- Trombi is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
-- 
-- You should have received a copy of the GNU General Public License
-- along with Trombi.  If not, see <http://www.gnu.org/licenses/>.
--------------------------------------------------------------------------------

-----------------------------------------------
--
-- Quelques operations utiles sur les listes
--
-----------------------------------------------
with ada.unchecked_deallocation;


generic
  type objet is private;

package listes is

  --------------------------- Liste chainee classique --------------
  
  -- Definition d'une cellude de la liste
  type st_liste;
  type liste is access st_liste;
  type st_liste is record
    cell : Objet;
    suiv : liste;
  end record;


  -- Liste vide
  Nil : constant liste := null;

  -- Ajout d'un element en tete
  procedure ajouteTete(o : objet ; l : in out liste);
  -- Ajout d'un element en queue
  procedure ajouteQueue(o : objet ; l : in out liste);
  
  -- Liberation de l'espace memoire
  procedure supprime is new ada.unchecked_deallocation(st_liste, liste);

  -- Renvoie le premier element de la liste
  function tete(l : liste) return objet;
  -- Renvoie la suite de la liste (sans le premier element)
  function suite(l : liste) return liste;
  -- Renvoie la suite de la liste (sans le premier element), et detruit le premier element
  function suiteDestructive(l : liste) return liste;

  -- Renvoie le dernier element de la liste
  function queue(l : liste) return liste;

  -- Renvoie une liste vide
  function listeVide return liste;

  -- Renvoie 'true' ssi la liste est vide
  function estVide(l : liste) return boolean;


  ---------------------------- Liste chainee dont le dernier element est accessible en temps O(1) ---------
  -- Invariant :
  --    le champs 'dernier' du record est un des suivants de 'premier'
  --    et le suivant de 'dernier' est la liste vide
  --
  --    OU 
  --
  --    'dernier' = 'premier' = liste vide

  -- Definition d'une liste accessible par les deux bouts
  type dliste is private;

  -- Ajout d'un element en tete
  procedure ajouteTete(o : objet ; l : in out dliste);
  -- Ajout d'un element en queue
  procedure ajouteQueue(o : objet ; l : in out dliste);
  
  -- Renvoie le premier element de la liste
  function tete(l : dliste) return objet;
  -- Renvoie la suite de la liste (sans le premier element)
  function suite(l : dliste) return dliste;
  -- Renvoie la suite de la liste (sans le premier element), et detruit le premier element
  function suiteDestructive(l : dliste) return dliste;

  -- Renvoie le dernier element de la liste
  function queue(l : dliste) return dliste;

  -- Renvoie une liste vide
  function listeVide return dliste;

  -- Renvoie 'true' ssi la liste est vide
  function estVide(l : dliste) return boolean;

  private

  type dliste is record
    premier : liste;
    dernier : liste;
  end record;

end listes;
