# bb_dumper: dumps a blackboard and SNMP agent code
#
# Copyright (C) 2006,2007,2008
# Frederik Deweerdt <frederik.deweerdt@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

require 'data_tree.rb'
require 'celsoft.com/template'
require 'yaml'
require 'digest/sha1'
require 'optparse'

class Bb_dumper
	attr_reader :efile_string, :hfile_string, :cfile
	@@bit_no = 0

  def Bb_dumper.add_options(optparser, options)
    optparser.on("-w", "--write-all", "Ignore readonly attributes on the MIB") { |options[:writeall]| }
  end#add_options

  def initialize(options)
    @sha1 = ""
    @options = options
    @dest = options[:dest]
    @tmpl = Template::Document.new()
    @templates=YAML::load(File.open("#{ENV['DADI_HOME']}/tmpl/bb.tmpl"))
    @daditype_mapper = {
      "int" => "DADI_INT",
      "uint" => "DADI_UINT",
      "chartab" => "DADI_CHARTAB",
      "macaddr" => "DADI_MACADDR",
      "ipaddr" => "DADI_IPADDR",
    }

    # Support read-only or not, depending on --write-all
    if options[:writeall] then
      @dadiaccess_mapper = {
      "readwrite" => "DADI_RW",
      "readonly" => "DADI_RW"
      }
    else
      @dadiaccess_mapper = {
      "readwrite" => "DADI_RW",
      "readonly" => "DADI_RO"
      }
    end#if
    @bbtype_mapper = {
          "double" => "E_BB_DOUBLE",
          "float" => "E_BB_FLOAT",
          "int8_t" => "E_BB_INT8",
          "int16_t" => "E_BB_INT16",
          "int32_t" => "E_BB_INT32",
          "int" => "E_BB_INT32",
          "uint" => "E_BB_UINT32",
          "int64_t" => "E_BB_INT64",
          "uint8_t" => "E_BB_UINT8",
          "uint16_t" => "E_BB_UINT16",
          "uint32_t" => "E_BB_UINT32",
          "uint64_t" => "E_BB_UINT64",
          "char" => "E_BB_CHAR",
          "unsigned char" => "E_BB_UCHAR"
    }
    @bbtype_mapper.default = "E_BB_USER";
	end#initialize

  def dest_folders
    return ["#{@dest}/src", "#{@dest}/include"]
  end#dest_folders

  def prolog
    @cfile = File.new("#{@dest}/src/mib.c", "w+")
    @hfile = File.new("#{@dest}/include/inlines_mib.h", "w+")
    @efile = File.new("#{@dest}/include/enums.h", "w+")
    @snmp_hack = ""
    @bb_hack = ""
    @hfile.puts "#ifndef __MIB_H__\n#define __MIB_H__"
    @hfile.puts "#include \"enums.h\""
    @efile.puts "#ifndef __ENUMS_H__\n#define __ENUMS_H__"
  end

	def get_template(name, default=nil)
		begin
			@tmpl.load(@templates[name].clone)
			return @tmpl.output
		rescue
			raise "Cant find template for #{name}" if default == nil
			return default
		end
	end#get_template

  def dump_types (tree,bbname)
    # Generate the list of the variables that can trigger
    # events on accessors
    tree.nodes.each do |n|
      @tmpl.data = n
      @tmpl.data["bit_no"] = @@bit_no
      @efile_string += get_template("define_decl")
      @efile_string += get_template("define_oid")
      @@bit_no += 1
    end#node.each

    tree.tables.each do |n|
      @tmpl.data = n
      @tmpl.data["bit_no"] = @@bit_no
      @efile_string += get_template("define_decl")
      @@bit_no += (n["length"].to_i * n["columns"].size)
    end#node.each

    @tmpl.data = {}
    @tmpl.data["name"] = "#{bbname}_max"
    @tmpl.data["bit_no"] = @@bit_no
    @efile_string += get_template("define_decl")

    #Generate the enums defined in the MIB
    tree.enums.each do
      |e|
      enum_pairs = ""
      e["names"].each { |v|
        @tmpl.data = {}
        @tmpl.data["name"] = "#{e['name']}_#{v}"
        @tmpl.data["value"] = e["values"][v]
        enum_pairs += get_template("enum_pair")
      }
      @tmpl.data = e
      @tmpl.data["enum_pairs"] = enum_pairs
      @efile_string += get_template("enum_decl")
    end#done
  end#dump_types

	def dump_datatree(tree,bbname)
    @sha1 += Digest::SHA1.hexdigest(tree.to_s)
    @efile_string = ""
    @cfile_string = ""
    @hfile_string = ""
		dump_types(tree,bbname)
		publishes = ""
		snmp_register_calls = ""
		getters_decl = ""
		setters_decl = ""
		getters = ""
		setters = ""

		#Add additional information on nodes and tables
    tree.nodes.each { |n|
      n["c_oid"] = n["oid"].gsub(".", ",")+",0"
			n["bbname"] = bbname
			n["dadi_type"] = @daditype_mapper[n['type'].to_s]
			n["dadi_access"] = @dadiaccess_mapper[n['access'].to_s]
			@tmpl.data = n
			n["decl"] = get_template("#{n['type']}_decl")
		}
		tree.tables.each { |t|
			t["c_oid"] = t["oid"].gsub(".", ",")
			t["min"] = 0
			t["max"] = t["length"].to_i - 1
			t["columns"].each { |n|
				n["bbname"] = bbname
				n["dadi_type"] = @daditype_mapper[n['type'].to_s]
				n["dadi_access"] = @dadiaccess_mapper[n['access'].to_s]
				@tmpl.data = n
				n["decl"] = get_template("#{n['type']}_decl")
			}
		}

		@tmpl.data = {}
		@tmpl.data["bbname"] = bbname
		@tmpl.data["nodes"] = tree.nodes

		@hfile_string += get_template("extern_mib_pointer")

		@cfile_string += get_template("mib_c_includes")
		@cfile_string += get_template("mib_pointer_decl")

		tree.tables.each do
			|t|
			@tmpl.data = t
			@tmpl.data["bbname"] = bbname
			@tmpl.data["table_name"] = t["name"]

			@hfile_string += get_template("table_structure")
			@hfile_string += get_template("table_length")

			publishes += get_template("table_alias_publish")

      @hfile_string += get_template("table_min_getter_decl")
      @hfile_string += get_template("table_max_getter_decl")

      col_no = 0
			t["columns"].each do
				|c|
				@tmpl.data = c
				@tmpl.data["bbname"] = bbname
				@tmpl.data["col_no"] = col_no
				@tmpl.data["size"] = t["columns"].size
				@tmpl.data["table_name"] = t["name"]
				@tmpl.data["bb_type"] = @bbtype_mapper[c['type'].to_s]

        @hfile_string += get_template("row_id_getter_decl")

				getters_decl += get_template("table_getter_#{c['type']}_decl")
				getters_decl += get_template("table_getter_#{c['type']}_pointer_decl", "")
				setters_decl += get_template("table_setter_#{c['type']}_decl")

				getters += get_template("table_getter_#{c['type']}")
				getters += get_template("table_getter_#{c['type']}_pointer", "")
				setters += get_template("table_setter_#{c['type']}")
				publishes += get_template("table_member_alias_publish")

        col_no += 1
			end#each.columns
		end#each

		@tmpl.data = {}
		@tmpl.data["bbname"] = bbname
		@tmpl.data["nodes"] = tree.nodes if tree.nodes.size > 0
		@tmpl.data["tables"] = tree.tables if tree.tables.size > 0

		@hfile_string += get_template("main_structure")

		tree.nodes.each do
			|n|
			@tmpl.data = n
			@tmpl.data["bbname"] = bbname
			@tmpl.data["bb_type"] = @bbtype_mapper[n['type'].to_s]

			getters_decl += get_template("getter_#{n['type']}_decl")
			getters_decl += get_template("getter_#{n['type']}_pointer_decl", "")
			setters_decl += get_template("setter_#{n['type']}_decl")

			getters += get_template("getter_#{n['type']}")
			getters += get_template("getter_#{n['type']}_pointer", "")
			setters += get_template("setter_#{n['type']}")
			publishes += get_template("alias_publish")

		end#each

		@cfile_string += setters
		@cfile_string += getters

		@hfile_string += setters_decl
		@hfile_string += getters_decl

		@tmpl.data = {}
		@tmpl.data["bbname"] = bbname
		@tmpl.data["nodes"] = tree.nodes if tree.nodes.size > 0
		@tmpl.data["tables"] = tree.tables if tree.tables.size > 0

		nb_elem = tree.nodes.size + 1
		tree.tables.each { |t| nb_elem += t["columns"].size }
		nb_elem += tree.tables.size
		@tmpl.data["nb_elem"] = nb_elem

		@tmpl.data["alias_publish_calls"] = publishes

		@cfile_string += get_template("register_bb")

		@cfile_string += get_template("register_snmp")

		@tmpl.data = {}
		@tmpl.data["bbname"] = bbname
		@tmpl.data["nodes"] = []
		tree.nodes.each { |n|
			if n["type"] == "int" then
				@tmpl.data["nodes"] << n
			end#if
		}
    @hfile.puts @hfile_string
    @cfile.puts @cfile_string
    @efile.puts @efile_string
    @snmp_hack+="#{bbname}_init_snmp();\n"

    @bb_hack+="#{bbname} = #{bbname}_register_bb();\n"
    @bb_hack+="{ extern S_BB_T *bb_#{bbname}; bbtab[bbtab_index++] = bb_#{bbname}; }\n"

	end#dump_datatree

  def epilog
    @cfile.puts "const char dadi_interp[] __attribute__((section(\".interp\"))) = \"/lib/ld-linux.so.2\";"

    @cfile.puts "int standalone_start() { extern int dadi_test(); return dadi_test(); }"
    @cfile.puts "int register_snmp() { #{@snmp_hack} return 1; }"

    @cfile.puts "int dadi_max_id = #{@@bit_no};"

    @hfile.puts "const char *datatree_sha1();"
    @cfile.puts "const char *datatree_sha1() { static const char sha1[] = \"#{@sha1}\"; return sha1; }"

    @cfile.puts <<EOS
      S_BB_T **register_bb() {
        S_BB_T **bbtab = calloc (MAX_BBS, sizeof (*bbtab));
        if (!bbtab) return NULL ;
        int bbtab_index = 0;
        #{@bb_hack}
        return bbtab;
      }
EOS
    @hfile.puts "#endif /* __MIB_H__ */"
    @efile.puts "#endif /* __ENUMS_H__ */"
  end#epilog


end#class Bb_dumper
