/*
    dadi.c - dadi C code, used by the bb_dumper

    Copyright (C) 2006,2007,2008
    Frederik Deweerdt <frederik.deweerdt@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published
    by the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301 USA.
*/
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/queue.h>
#include <sys/socket.h>

#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>

#include "dadi.h"

#define DADI_PID_FILE "/tmp/dadi.pid"
#define DADI_USOCK_FILE "/tmp/dadi.usock"

static dadi_callback_t ActivationCallback;
/* array of all the BB's known to DADI */
static S_BB_T **bbtab;

struct dadi_msg_event {
	int dadi_var;
	enum dadi_event_type event_type;
};

static int running = 1;
int is_master;

static int try_to_be_master();


/* these defines are missing on FC4 */
#ifndef LIST_HEAD_INITIALIZER
#define	LIST_HEAD_INITIALIZER(head)					\
	{ NULL }
#endif

#ifndef LIST_FOREACH
#define	LIST_FOREACH(var, head, field)					\
	for ((var) = ((head)->lh_first);				\
	     (var);							\
	     (var) = ((var)->field.le_next))
#endif

#ifndef LIST_NEXT
#define	LIST_NEXT(elm, field)		((elm)->field.le_next)
#endif

struct client {
	int fd;
	LIST_ENTRY(client) entries;
};

static LIST_HEAD(clients, client) clients =
LIST_HEAD_INITIALIZER(clients);

/* Array indicating whenever a given field has beed accessed or not */
static int *has_been_accessed;

static void dadi_master_add_client(int fd)
{
	struct client *client;
	client = calloc(1, sizeof(struct client));
	if (!client) {
		fprintf(stderr,
			"Dadi: Not enough mem, cannot add new client\n");
		return;
	}

	client->fd = fd;
	LIST_INSERT_HEAD(&clients, client, entries);
}

int dadi_evt(int dadi_var, enum dadi_event_type event_type)
{
	/* We're the master, send msgs to all the connected clients */
	if (is_master) {
		struct dadi_msg_event msg_evt = {
			.dadi_var = dadi_var,
			.event_type = event_type
		};

		struct client *client;
		struct client fake_client; /* used for safe client removal */

		LIST_FOREACH(client, &clients, entries) {
retry_send:
			if (send(client->fd, &msg_evt,
				 sizeof(msg_evt), MSG_NOSIGNAL) <= 0) {
				switch (errno) {
				case EPIPE:
				case ECONNRESET:
					LIST_NEXT(&fake_client, entries) = LIST_NEXT(client, entries);
					LIST_REMOVE(client, entries);
					close(client->fd);
					free(client);
					client = &fake_client;
					break;
				case EINTR:
					goto retry_send;
					break;
				default :
					perror("send");
					exit(1);
				}
			}
		}
	}
	if (ActivationCallback != NULL) {
		return ActivationCallback(dadi_var, event_type);
	} else {
		return -1;
	}
}

static void startup_agent(const char *name)
{
	/* we are a subagent */
	netsnmp_ds_set_boolean(NETSNMP_DS_APPLICATION_ID,
			       NETSNMP_DS_AGENT_ROLE, 1);
	netsnmp_register_loghandler(NETSNMP_LOGHANDLER_STDERR, LOG_WARNING);

	/* initialize tcpip, if necessary */
	SOCK_STARTUP;
	/* initialize the agent library */
	init_agent(name);
	/* register snmp callbacks */
	register_snmp();
	/* startup the agent */
	init_snmp(name);
}


static void *usock_server_thread(void *arg)
{
	int listen_socket;
	struct sockaddr_un local;
	int len;

	listen_socket = socket(AF_UNIX, SOCK_STREAM, 0);
	if (listen_socket == -1) {
		perror("socket");
		exit(1);
	}
	local.sun_family = AF_UNIX;
	strcpy(local.sun_path, DADI_USOCK_FILE);
	unlink(local.sun_path);
	len = strlen(local.sun_path) + sizeof(local.sun_family);
	if (bind(listen_socket, (struct sockaddr *)&local, len) == -1) {
		perror("bind");
		exit(1);
	}
	if (listen(listen_socket, 5)) {
		perror("listen");
		exit(1);
	}

	while (running) {
		int client_socket, ret, resp;
		char client_sha1[2048];

		client_socket = accept(listen_socket, NULL, NULL);
		if (client_socket == -1) {
			perror("accept");
			exit(1);
		}

		/*
		 * Check if client's version matches ours. If not, drop it
		 */
		ret = read(client_socket, client_sha1, sizeof(client_sha1));
		if (ret <= 0)
			continue;

		if (strncmp(datatree_sha1(),
			    client_sha1,
			    strlen(datatree_sha1()))) {
			fprintf(stderr, "Dadi: attempted connection from incompatible client, ignoring\n");
			resp = 0;
			write(client_socket, &resp, sizeof(resp));
			close(client_socket);
			continue;
		}

		resp = 1;
		ret = write(client_socket, &resp, sizeof(resp));
		if (ret != sizeof(resp))
			continue;

		/* Sanity checks OK, add client */
		dadi_master_add_client(client_socket);
	}

	close(listen_socket);
	return NULL;
}

static void dadi_client(void)
{
	int client_socket;
	struct sockaddr_un remote;
	int len, ret;
	static int retries = 0;

	client_socket = socket(AF_UNIX, SOCK_STREAM, 0);
	if (client_socket == -1) {
		perror("socket");
		exit(1);
	}
	remote.sun_family = AF_UNIX;
	strcpy(remote.sun_path, DADI_USOCK_FILE);
	len = strlen(remote.sun_path) + sizeof(remote.sun_family);
retry_connect:
	if (connect(client_socket, (struct sockaddr *)&remote, len) == -1) {
		switch (errno) {
		case ECONNREFUSED:
		case ENOENT:
		case ETIMEDOUT:
			retries++;
			close(client_socket);
			if (retries > 50) {
				perror("Cannot open dadi socket");
				exit(1);
			}
			/* we will retry later */
			return;
		case EINTR:
			goto retry_connect;
			break;
		default:
			perror("connect");
			exit(1);
		}
	}
	/* Send our expected version to the server */
	len = write(client_socket, datatree_sha1(), strlen(datatree_sha1())+1);
	if (len <= 0) {
		perror("client handshake write");
		exit(1);
	}
	/* Read back if server accepted us or not */
	len = read(client_socket, &ret, sizeof(ret));
	if (len <= 0) {
		perror("client handshake read");
		exit(1);
	}
	if (!ret) {
		fprintf(stderr, "Dadi: server version incompatible, exiting\n");
		exit(1);
	}

	retries = 0;

	/* Process dadi events are they are received through the socket */
	while (running) {
		struct dadi_msg_event msg_event;
		int n;
                fd_set rd_set;
                struct timeval tv = { .tv_sec = 1, .tv_usec = 0 };

                FD_ZERO(&rd_set);
                FD_SET(client_socket, &rd_set);

		ret = select(client_socket+1, &rd_set, NULL, NULL, &tv);
		if (ret == 0 || (ret < 0 && errno == EINTR)) {
                        continue;
		}

		n = recv(client_socket, &msg_event, sizeof(msg_event), 0);
		if (n == 0) {
			close(client_socket);
			return;
		}
		if (n < 0) {
			if (errno == EINTR) {
				continue;
			}
			perror("recv");
			exit(1);
		}

		dadi_evt(msg_event.dadi_var, msg_event.event_type);
	}

	close(client_socket);
	return;
}

void sigpipe_handler(int sig, siginfo_t *info, void *arg)
{
	fprintf(stderr, "Dadi: received a SIGPIPE, exiting\n");
	/*
	 * Kill ourselves, to my knowledge there's nothing we can do.
	 * As of net-snmp 5.4 a timeouted agent just gets dumped by the
	 * net-snmpd
	 */
	kill(getpid(), SIGABRT);
}

/** @ingroup dadi_itf
 * @brief first initialisation function for the library.
 *
 * This function makes the pre-initialisation of the library. It allocates
 * necessary strutures and so on.
 * @param theActivationCallback the activation callback that will be called
 * when parameters are modified
 * @return 0 if the initialisation fails, or a pointer on the global
 * configuration and counters
 */
void *dadi_first_init(dadi_callback_t theActivationCallback)
{
	void *ret = NULL;
	struct sigaction sigpipe_action = { .sa_sigaction = sigpipe_handler, };

	sigaction(SIGPIPE, &sigpipe_action, NULL);

	/* Set the callback */
	if (theActivationCallback == NULL)
		return NULL;
	ActivationCallback = theActivationCallback;

	/* build the blackboard and allocate the structure */
	bbtab = register_bb();

	return ret;
}

/**@ingroup dadi_itf
 * @brief initialisation function for the library.
 *
 * This function initialises the library before the main loop is called.
 * Calling this function will fill all the parameters of the global
 * structure and will trigger calls to the activation callback
 * @return 0 if the initialisation succeeded, -1 otherwise
 */
void *dadi_init(dadi_callback_t theActivationCallback)
{
	fprintf(stderr, "Dadi: NOT IMPLEMENTED : %s\n", __FUNCTION__);
	return NULL;
}

/** main dadi loop. Never ends unless ef dadi_stop is called
 * @return 0 if the library closed successfully, -1 otherwise
 */
int dadi_mainloop(void)
{
	fprintf(stderr, "Dadi: started\n");

	is_master = try_to_be_master();
	if (is_master) {
		startup_agent("agent");
	}

	while (running) {
		if (is_master) {
			pthread_t th;
			pthread_create(&th, NULL, &usock_server_thread, NULL);

			while (running) {
				if (agent_check_and_process(1) < 0) {
					if (errno == EINTR)
						sched_yield();
				}
			}
		} else {
			dadi_client();
			if (running) {
				is_master = try_to_be_master();
				if (is_master) {
					startup_agent("agent");
				}
			}
		}
	}

	return 0;
}

int dadi_last_init(void)
{
	return 0;
}

/**
 * @ingroup dadi_itf
 * @brief function that do what's is needed to stop the main_loop
 *
 * @return 0 if the main loop successfully stops, -1 otherwise
 */
int dadi_terminate_request()
{
	running = 0;
}

/**
 * @ingroup dadi_itf
 * @brief function that frees all allocated data within the dadi library
 *
 * @return 0 if the main loop successfully stops, -1 otherwise
 */
int dadi_stop(void)
{
	int i = 0;

	/*
	 * Although this should be called, it actually double frees some stuff
	 * in net-snmp (shutdown_agent, and snmp_shutdown cannot be called
	 * together
	 */
	/* shutdown_agent(); */
	if (is_master) {
		snmp_shutdown(NULL);
		SOCK_CLEANUP;
	}

	/* TSP blackboard cleanup */
	while (bbtab[i] != NULL) {
		bb_detach(&bbtab[i]);
		i++;
	}

	fprintf(stderr, "Dadi: stopped\n");
	return 0;
}

/**
 *  test_callback: prints default message when called
 */
static int test_callback(int var, enum dadi_event_type event_type)
{
	switch (event_type) {
	case DADI_EVENT_TYPE_GET:
	     	has_been_accessed[var] |= 1;
		printf("Get done : %d\n", var);
		break;
	case DADI_EVENT_TYPE_SET:
	     	has_been_accessed[var] |= 2;
		printf("Set done : %d\n", var);
		break;
	default:
		printf("Bad event type : %d\n", event_type);

	}
	return 1;
}

static void print_accesses()
{
	int i;
	for (i=0; i < dadi_max_id; i++) {
		printf("COV %d: %d\n", i, has_been_accessed[i]);
	}
}
/**
 *  dadi_test: test entry point for the library
 */
int dadi_test(int argc, char **argv)
{
	typedef void (*sighandler_t)(int);

	has_been_accessed = calloc(1, sizeof(int)*dadi_max_id);
	dadi_callback_t theActivationCallback = test_callback;
	dadi_first_init(theActivationCallback);
	signal(SIGINT, (sighandler_t)dadi_terminate_request);
	dadi_mainloop();
	dadi_stop();

	print_accesses();

	exit(0);
	return 0;
}



static int try_to_be_master(void)
{
	char buf[256];

	int fd = open(DADI_PID_FILE, O_CREAT | O_EXCL | O_WRONLY,
		      S_IRUSR | S_IWUSR);
	if (fd == -1) {
		if (errno != EEXIST) {
			perror("open (create)");
			exit(1);
		}
		fd = open(DADI_PID_FILE, O_WRONLY);
		if (fd == -1) {
			perror("open (write)");
			exit(1);
		}
	}

	if (lockf(fd, F_TLOCK, 0) == -1) {
		if (errno == EAGAIN) {
			close(fd);
			return 0; /* there is already a client */
		}
		perror("lockf");
		exit(1);
	}
	if (ftruncate(fd, 0) == -1) {
		perror("ftruncate");
		close(fd);
		exit(1);
	}
	sprintf(buf, "%d", getpid());
	if (write(fd, buf, strlen(buf)) < 0) {
		perror("write (pid)");
		close(fd);
		exit(1);
	}
	return 1;
}
