/* $Id: SpecSamplingConfig.java,v 1.4 2006/06/04 08:33:45 erk Exp $
 * -----------------------------------------------------------------------
 * 
 * TSP Library - core components for a generic Transport Sampling Protocol.
 * 
 * Copyright (c) 2002 Yves DUFRENNE, Stephane GALLES, Eric NOULARD and Robert PAGNOT 
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * -----------------------------------------------------------------------
 * 
 * Project    : TSP
 * SubProject : jTSP
 * Maintainer : tsp@astrium-space.com
 * Component  : 
 * 
 * -----------------------------------------------------------------------
 * 
 * Purpose   : Consumer
 * 
 * -----------------------------------------------------------------------
 */


package tsp.consumer.util.configuration;


import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Iterator;
import java.util.Map;

import tsp.util.ToStringHelper;

public class SpecSamplingConfig implements Consolidatable {

    private String version;
    private String comment;


    /**
     * Here we are using explicit LinkedList type, because we need
     * the getLast function not available in List interface
     */

    private LinkedList layoutLists = new LinkedList();
    private LinkedList sampleLists = new LinkedList();
    private LinkedList providerLists = new LinkedList();

    public String toString(){
        return ToStringHelper.toString(this);
    }

    // The layout methods should be public 
    // ib order to ease Layout extensibility.
    
    public SpecLayoutList getLastLayoutList(){
        return (SpecLayoutList)layoutLists.getLast();
    }

    SpecSampleList getLastSampleList(){
        return (SpecSampleList)sampleLists.getLast();
    }

    SpecProviderList getLastProviderList(){
        return (SpecProviderList)providerLists.getLast();
    }

    
    public void addLastLayoutList(SpecLayoutList l){
        layoutLists.addLast(l);
    }

    void addLastSampleList(SpecSampleList l){
        sampleLists.addLast(l);
    }

    void addLastProviderList(SpecProviderList l){
        providerLists.addLast(l);
   }


    // External
    public String getVersion() {
        return version;
    }

    void setVersion(String version) {
        this.version = version;
    }

    public String getComment() {
        return comment;
    }

    void setComment(String comment) {
        this.comment = comment;
    }

    public List getLayoutLists() {
        return layoutLists;
    }

    public List getSampleLists() {
        return sampleLists;
    }
    

    public List getAllLayouts(){
        List allLayouts = new LinkedList();
        Iterator it = layoutLists.listIterator();
        while(it.hasNext()){
            SpecLayoutList layList = (SpecLayoutList)it.next();
            allLayouts.addAll(layList.getLayoutList());
        }
        return allLayouts;
    }
    
    
    public List getAllSamples(){
    	List allSamples = new LinkedList();
    	Iterator it = sampleLists.listIterator();
    	while(it.hasNext()){
    		SpecSampleList sampList = (SpecSampleList)it.next();
    		allSamples.addAll(sampList.getSamples());
    	}
    	return allSamples;
    }

    
    public List getAllProviders(){
    	List allProviders = new LinkedList();
    	Iterator it = providerLists.listIterator();
    	while(it.hasNext()){
    		SpecProviderList provList = (SpecProviderList)it.next();
    		allProviders.addAll(provList.getProviders());
    	}
    	return allProviders;
    }
    
    public Map getAllSamplesByProvider() throws SpecException {
    	Map samplesByProv = new HashMap();
    	Iterator it = getAllSamples().listIterator();
    	while(it.hasNext()){
    		SpecSample sample = (SpecSample)it.next();
    		List sampleList = (List)samplesByProv.get(sample.getProvider());
    		if(sampleList == null){
    			sampleList = new LinkedList();
    			samplesByProv.put(sample.getProvider(), sampleList);
    		}
    		sampleList.add(sample);    		
    	}
    	return samplesByProv;
    }

    public List getProviderLists() {
        return providerLists;
    }

    public void consolidate() throws SpecException {
    	ConsolidatorHelper.Consolidate(layoutLists);
    	ConsolidatorHelper.Consolidate(sampleLists);
    	ConsolidatorHelper.Consolidate(providerLists);
    }
    
    public SpecProvider getProviderByName(String providerName) throws SpecException {
		Iterator itProvList = getProviderLists().iterator();
		SpecProvider foundProvider = null;
		while (itProvList.hasNext() && foundProvider == null) {
			SpecProviderList provList = (SpecProviderList) itProvList.next();
			Iterator itProv = provList.getProviders().iterator();
			while (itProv.hasNext() && foundProvider == null) {
				SpecProvider prov = (SpecProvider) itProv.next();
				if (prov.getName().equals(providerName)) {
					foundProvider = prov;
				}
			}
		}
		if(foundProvider == null){
			String msg = "Unable to find provider named '" + providerName + "'"; 
			throw new SpecException(msg);
		}
		return foundProvider;
	}

}
