#$Id: simple.pm,v 1.1 2005/07/15 23:13:26 erk Exp $
#
#-----------------------------------------------------------------------
#
#TSP Library - core components for a generic Transport Sampling Protocol.
#
#Copyright (c) 2005 Pierre Mallard
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU Lesser General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#Lesser General Public License for more details.
#
#You should have received a copy of the GNU Lesser General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#-----------------------------------------------------------------------
#
#Project    : TSP
#Maintainer : pmallard@silicom.fr
#Component  : TSP::simple simple use of TSP::tsp_consumer
#
#-----------------------------------------------------------------------
#
#Purpose   :
# It is as much an easyer interface to TSP::tsp_consumer module and an example of usage of TSP::tsp_consumer
#
# author : Pierre Mallard <pmallard@silicom.fr>

package TSP::simple;
use TSP::tsp_consumer;
use Time::HiRes qw(  gettimeofday tv_interval );

my $tsp_provider;
my @sym_disp;
my $infos_disp;
my $STS;

sub init {
    (my $url) = @_;
    if (!defined $url) {
      die 'TSP::Simple::init needs an url';
    }
    $tsp_provider = init_connection($url);
}

sub init_connection {
    my ($url) = @_;
    my $tsp_provider;
    TSP::tsp_consumer::TSP_consumer_init int(0),\0;
    ($tsp_provider = TSP::tsp_consumer::TSP_consumer_connect_url $url) or die "Server Unreachable $url";
    (TSP::tsp_consumer::TSP_consumer_request_open($tsp_provider,0,0)) or die "Could not open session";
    return $tsp_provider;
}

sub get_available_symbol{
    TSP::tsp_consumer::TSP_consumer_request_information $tsp_provider;
    $infos_disp = TSP::tsp_consumer::TSP_consumer_get_information $tsp_provider;
    return $infos_disp->{symbols}->{val};
}

sub print_available_symbol{
    my ($sym_val) = @_;
    foreach (@$sym_val){
	print $_->{index}," ",$_->{name},"\n";
    }
}

sub init_new_capture{
    my ($sym_array) = @_;
    my $sym_req_list = TSP::tsp_consumer::TSP_consumer_symbol_requested_list_t->new();
    $sym_req_list->{val} = $sym_array;
    $sym_req_list->{len} = scalar @$sym_array;
    TSP::tsp_consumer::TSP_consumer_request_sample $tsp_provider,$sym_req_list;
    $tmp = TSP::tsp_consumer::TSP_consumer_get_requested_sample($tsp_provider);
    @$sym_array = @{$tmp->{val}};
    #launch capture
    TSP::tsp_consumer::TSP_consumer_request_sample_init $tsp_provider,0,0;
}

sub read_till_timeout{
  my ($timeout,$callback) = @_;
  my $time = [gettimeofday()];
  while (tv_interval($time) < $timeout) {
      my($res,$sample,$is_new) = TSP::tsp_consumer::TSP_consumer_read_sample($tsp_provider);
      if (!$is_new) {
	  return;
	}
      &$callback($$sample);
    }
}

1;

__END__

=head1 NAME

TSP::simple - An example of tsp_consumer use

=head1 SYNOPSIS

init connection to provider :

init(rpc://host/bb_id:num)

retrieve available symbol :

$sym_dispos = get_available_symbol();

print symbol available :

print_available_symbol($sym_dispos);

init new capture :

foreach ('symbol0','symbol1','symbol2','symbol3','symbol4') {
  my $sym_req = tsp_consumer::TSP_consumer_symbol_requested_t->new ();
  $sym_req->{name} = $_;
  $sym_req->{period} = 1;
  push(@request_sym_list,$sym_req);
}

init_new_capture(\@request_sym_list);

Good indices will be set in index field of TSP_consumer_symbol_requested_t inside data structure

read sample :

read_till_timeout($timeout_in_sec,\&my_func);

sub my_func {
  my ($sample) = @_;
  #use @request_sym_list to retrieve symbol with $id
  my $id = $sample->{provider_global_index};
  my $value = $sample->{user_value};
}

=head1 DESCRIPTION

TSP::simple is here to easily use TSP::tsp_consumer
for documentation of the latter, see tsp_consumer.h

=head1 AUTHOR

Pierre Mallard <pmallard@silicom.fr>

=cut
