/*!  \file 

$Id: tsp_provider.c,v 1.35 2006/01/22 09:35:15 erk Exp $

-----------------------------------------------------------------------

TSP Library - core components for a generic Transport Sampling Protocol.

Copyright (c) 2002 Yves DUFRENNE, Stephane GALLES, Eric NOULARD and Robert PAGNOT 

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

-----------------------------------------------------------------------

Project    : TSP
Maintainer : tsp@astrium.eads.net
Component  : Provider

-----------------------------------------------------------------------

Purpose   : Main implementation for the producer module

-----------------------------------------------------------------------
 */

#include <tsp_sys_headers.h>
#include <tsp_provider.h>
#include <tsp_filter_symbol.h>
#include <tsp_session.h>
#include <tsp_glu.h>	
#include <tsp_time.h>
#include <tsp_common.h>
#include "tsp_datapool.h"

/** modified argc and argv, will be returned to user code after init */
static  char** X_argv = 0;

/** Default values for args to the GLU */
static  char** X_glu_argv = 0;
static  int X_glu_argc = 0;

static  GLU_handle_t* firstGLU = NULL;

/** server base number is an offset allowing to distinguish TSP providers **/
/** each Request handler shall look for a channel number from this offset,
    and limited to TSP_MAX_SERVER_NUMBER **/
static int X_server_base_number = 0;


static int X_tsp_provider_init_ok = FALSE;

/**
 * The mutex used to serialize 
 * the asynchronous request that may be issued by different 
 * TSP asynchronous command link (RPC, XML-RPC, CORBA...)
 * Note that normally we may use several mutexes
 * in order to serialize more efficiently the asynchronous request.
 * That is one for tsp_request_open and 1 for each client after that.
 */
static pthread_mutex_t X_tsp_request_mutex = PTHREAD_MUTEX_INITIALIZER;

/** Tells is the GLU is active or pasive */
static int X_glu_is_active;

/* polling time for session garbage collector */
#define TSP_GARBAGE_COLLECTOR_POLL_TIME_US ((int)(5e6))

static int TSP_cmd_line_parser(int* argc, char** argv[])
{
  int i;
  int final_argc = 1;
  int found_stream_start = FALSE;
  int found_stream_stop = FALSE;
  int found_server_number_flag = FALSE;
  char* p;
  int ret = TRUE;

  STRACE_IO(("-->IN"));

  /* FIXME : FUITE */
  X_argv = (char**)calloc(*argc, sizeof(char*));
  X_glu_argv = (char**)calloc(*argc, sizeof(char*));
  X_glu_argc = 0;
  TSP_CHECK_ALLOC(X_argv, FALSE);
  TSP_CHECK_ALLOC(X_glu_argv, FALSE);
  /* Get program name anyway */
  X_argv[0] = (*argv)[0];
  for( i = 1 ; i < *argc && ret ; i++)
    {
      /* Is the arg a TSP arg ? */
      p = strstr( (*argv)[i], TSP_ARG_PREFIX );
      if(p && (p == (*argv)[i] ))
	{
	  /* TSP Arg */
	  STRACE_INFO(("Tsp ARG : '%s'", (*argv)[i]));

	  /* First we must not be looking for a argument */
	  if(found_server_number_flag)
	    {
	      /* Error we are expecting an arguement not a TSP option*/
	      STRACE_WARNING(("Unexpected %s", (*argv)[i]));
	      ret = FALSE;
	    }
	  /* Look for start flag */
	  else if(!strcmp(TSP_ARG_STREAM_INIT_START, (*argv)[i]))
	    {
	      if(!found_stream_stop && !found_stream_start)
		{
		  found_stream_start = TRUE;
		  /* Ok the user wants a default stream control, put
		     the first dummy element */
		  if ( 0 == X_glu_argc )
		    {
		      X_glu_argv[0] = TSP_ARG_DUMMY_PROG_NAME;
		      X_glu_argc = 1;
		    }
		}
	      else
		{
		  STRACE_WARNING(("Unexpected "TSP_ARG_STREAM_INIT_START));
		  ret = FALSE;
		}
	    }
	  else if (!strcmp(TSP_ARG_STREAM_INIT_STOP, (*argv)[i]))
	    {
	      if(found_stream_start && !found_stream_stop)		
		{
		  found_stream_stop = TRUE;
		}
	      else
		{
		  STRACE_WARNING(("Unexpected "TSP_ARG_STREAM_INIT_STOP));
		  ret = FALSE;
		}
	    }
	  else if (!strcmp(TSP_ARG_SERVER_NUMBER, (*argv)[i]))
	    {
	      found_server_number_flag = TRUE;
	      if(found_stream_start && !found_stream_stop)		
		{
		  STRACE_WARNING(("Unexpected "TSP_ARG_SERVER_NUMBER));
		  ret = FALSE;
		}
	    }
	  else
	    {
	      /* Unkown option */
	      STRACE_WARNING(("Unknown TSP option : '%s'",(*argv)[i] ))
	      ret = FALSE;
	    }
	}
      else /* Not a TSP arg */
	{
	  /* Are we in the TSP command line ? */
	  if ( found_stream_start && !found_stream_stop )
	    {
	      X_glu_argv[X_glu_argc++] = (*argv)[i];
	    }
	  else if(found_server_number_flag)
	    {
	      /* Found arg for server number option */
	      found_server_number_flag = FALSE;
	      X_server_base_number = atoi((*argv)[i]);
	      STRACE_INFO(("Server base number = %d", X_server_base_number));
	    }
	  else
	    {	      
	      /* Nop, this arg is for the user */
	      X_argv[final_argc] = (*argv)[i];
	      final_argc++;
	    }
	}
    } /* for */
  
  /* Check is the stop was found */
  
  if( found_stream_start && !found_stream_stop )
    {
      STRACE_WARNING(("A " TSP_ARG_STREAM_INIT_STOP " flag was expected"));
      ret = FALSE;
    }

  if( found_server_number_flag )
    {
      STRACE_WARNING(("An argument was expected after " TSP_ARG_SERVER_NUMBER));
      ret = FALSE;
    }

  /* swap argc and argv values */
  *argc = final_argc;
  *argv = X_argv;
  
  /* Display usage */
  if(!ret)
    {
      STRACE_WARNING((TSP_ARG_PROVIDER_USAGE));
    }
  else
    {
      if (! X_glu_argc )
	{
	  STRACE_INFO(("No GLU stream init provided on command line"));
	}
    }

  /* Memorize GLU type */
  if (  GLU_SERVER_TYPE_ACTIVE == firstGLU->get_type(firstGLU) )
    X_glu_is_active = TRUE;
  else
    X_glu_is_active = FALSE;

  STRACE_IO(("-->OUT"));
  
  return ret;
}

int TSP_provider_is_initialized(void)
{
  return  X_tsp_provider_init_ok;
}

int TSP_provider_get_server_base_number(void)
{
  return  X_server_base_number;
}

const char* TSP_provider_get_name() {
  assert(firstGLU);
  return firstGLU->get_name(firstGLU);
}


void TSP_provider_request_open(const TSP_request_open_t* req_open,
		      TSP_answer_open_t* ans_open)
{
  GLU_handle_t* glu_h;
  char* error_info;
  int i;
	
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);
  STRACE_IO(("-->IN"));


    /* Fortify calls */
  
    /*Fortify_EnterScope();*/

   ans_open->version_id = UNDEFINED_VERSION_ID;
   ans_open->channel_id = UNDEFINED_CHANNEL_ID;
   ans_open->status = TSP_STATUS_ERROR_UNKNOWN;
   ans_open->status_str = "";

   if(req_open->argv.TSP_argv_t_len)
     {
       for( i = 0; i< req_open->argv.TSP_argv_t_len ;i++)
	 {
	   STRACE_DEBUG(("arg %d is '%s'", i,  req_open->argv.TSP_argv_t_val[i]));
	 }
     }
   else
     {
          STRACE_DEBUG(("No custom args from consumer"));
     }


   /*  get GLU instance. If a stream init is provided, use it, else, use default */   
   if( 0 != req_open->argv.TSP_argv_t_len )
     {
       glu_h = firstGLU->get_instance(firstGLU, req_open->argv.TSP_argv_t_len, req_open->argv.TSP_argv_t_val, &error_info);
     }
   else
     {
       /* use fallback if provided */
       glu_h = firstGLU->get_instance(firstGLU, X_glu_argc, X_glu_argv, &error_info);
     }

   
   if(glu_h)
     {
       if(TSP_add_session(&(ans_open->channel_id), glu_h))
	 {
	   if(req_open->version_id <= TSP_VERSION)
	     {
	       ans_open->version_id = TSP_VERSION;
	       ans_open->status = TSP_STATUS_OK;
	     }
	   else
	     {
	       STRACE_ERROR(("TSP version ERROR. Requested=%d Current=%d",req_open->version_id, TSP_VERSION ));
	       ans_open->status = TSP_STATUS_ERROR_VERSION;
	     }
	 }
       else
	 {
	   STRACE_ERROR(("TSP_add_session failed"));
	 }
     }
   else
     {
       STRACE_INFO(("Unable to get GLU instance"));
        ans_open->status = TSP_STATUS_ERROR_SEE_STRING;
	ans_open->status_str = error_info;
     }

  STRACE_IO(("-->OUT"));

  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);	

} /* End of TSP_provider_request_open */


static void TSP_provider_request_close_priv(channel_id_t channel_id)
{
  STRACE_IO(("-->IN"));

  TSP_session_destroy_symbols_table_by_channel(channel_id);
  TSP_session_close_session_by_channel(channel_id);

  STRACE_IO(("-->OUT"));
}

void TSP_provider_request_close(const TSP_request_close_t* req_close)

{
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);

  STRACE_IO(("-->IN"));

  TSP_provider_request_close_priv(req_close->channel_id);

  STRACE_IO(("-->OUT"));

  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
} /* End of TSP_provider_request_close */

void  TSP_provider_request_information(TSP_request_information_t* req_info, 
				       TSP_answer_sample_t* ans_sample)
{
  TSP_provider_request_filtered_information(req_info,TSP_FILTER_NONE,NULL,ans_sample);  
  STRACE_DEBUG(("Nb symbol = %d",ans_sample->symbols.TSP_sample_symbol_info_list_t_len));
  STRACE_DEBUG(("Nb symbol = 0x%08x",(unsigned int)ans_sample->symbols.TSP_sample_symbol_info_list_t_val));
} /* End of TSP_provider_request_information */

void TSP_provider_update_answer_with_minimalinfo(TSP_request_information_t* req_info,
				      TSP_answer_sample_t* ans_sample) {

  ans_sample->version_id            = TSP_VERSION;
  ans_sample->channel_id            = req_info->channel_id;
  ans_sample->status                = TSP_STATUS_ERROR_UNKNOWN;
  ans_sample->provider_group_number = 0;
  ans_sample->base_frequency        = firstGLU->get_base_frequency(firstGLU);
  ans_sample->max_client_number     = firstGLU->get_nb_max_consumer(firstGLU);
  ans_sample->current_client_number = TSP_session_get_nb_session();
  ans_sample->max_period            = TSP_MAX_PERIOD;
  
}

void  TSP_provider_request_filtered_information(TSP_request_information_t* req_info, 
						int filter_kind, char* filter_string,
						TSP_answer_sample_t* ans_sample)
{
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);  

  /* fill-in minimal info in answer_sample */
  TSP_provider_update_answer_with_minimalinfo(req_info,ans_sample);

  ans_sample->symbols.TSP_sample_symbol_info_list_t_len = 0;
  ans_sample->symbols.TSP_sample_symbol_info_list_t_val = NULL;  

  /* check request TSP version */
  if (req_info->version_id > TSP_VERSION) {
    STRACE_ERROR(("TSP version ERROR. Requested=%d Current=%d",req_info->version_id, TSP_VERSION ));
    ans_sample->status = TSP_STATUS_ERROR_VERSION;
    TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
    return;
  }
  
  /* switch case for filter_kind */
  switch (filter_kind) {
  case TSP_FILTER_NONE:
    STRACE_INFO(("Requested filter NONE"));
    TSP_filter_symbol_none(req_info,filter_string,ans_sample);
    break;
  case TSP_FILTER_MINIMAL:
    STRACE_INFO(("Requested filter MINIMAL, filter string = <%s>",filter_string));
    TSP_filter_symbol_minimal(req_info,filter_string,ans_sample);
    break;
  default:
    STRACE_INFO(("Requested filter kind <%d>, filter string = <%s>",filter_kind,filter_string));
    /* 
     * forward other filtered request directly to GLU 
     * such that even non anticipated filtering method could
     * be implemented by specialized consumer and provider pair
     * default GLU will provider reasonnable default implementation.
     */
    firstGLU->get_filtered_ssi_list(firstGLU,filter_kind,filter_string,ans_sample);
    break;
  } /* end switch filter_kind */
    
  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
} /* End of TSP_provider_request_filtered_information */


void TSP_provider_request_sample_free_call(TSP_answer_sample_t* ans_sample)
{
  TSP_session_create_symbols_table_by_channel_free_call(ans_sample);
}

void  TSP_provider_request_sample(TSP_request_sample_t* req_sample, 
			 TSP_answer_sample_t* ans_sample)
{
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);	
  STRACE_IO(("-->IN"));
  int32_t i;
  int32_t invalid_period;
  int32_t invalid_phase;
  ans_sample->version_id            = TSP_VERSION;
  ans_sample->channel_id            = req_sample->channel_id;
  ans_sample->status                = TSP_STATUS_ERROR_UNKNOWN;
  ans_sample->provider_group_number = 0;
  ans_sample->base_frequency        = firstGLU->get_base_frequency(firstGLU);
  ans_sample->max_client_number     = firstGLU->get_nb_max_consumer(firstGLU);
  ans_sample->current_client_number = TSP_session_get_nb_session();
  ans_sample->max_period            = TSP_MAX_PERIOD;
  ans_sample->symbols.TSP_sample_symbol_info_list_t_len = 0;
  ans_sample->symbols.TSP_sample_symbol_info_list_t_val = 0;
  
  STRACE_INFO(("Consumer No %d asked for %d symbols",req_sample->channel_id,req_sample->symbols.TSP_sample_symbol_info_list_t_len  ));
  
  if(req_sample->version_id <= TSP_VERSION)
    {
      
      if(TSP_session_get_symbols_global_index_by_channel(req_sample->channel_id, &(req_sample->symbols) ))
	{  
	  invalid_period = 0;
	  invalid_phase  = 0;
	  /* validate period and phase range */
	  for (i=0;i<req_sample->symbols.TSP_sample_symbol_info_list_t_len;++i) {	    
	    if(req_sample->symbols.TSP_sample_symbol_info_list_t_val[i].period < 1) {
	      req_sample->symbols.TSP_sample_symbol_info_list_t_val[i].provider_global_index = -1;
	      invalid_period++;
	    }
	    if(req_sample->symbols.TSP_sample_symbol_info_list_t_val[i].phase < 0) {
	      req_sample->symbols.TSP_sample_symbol_info_list_t_val[i].provider_global_index = -1;
	      invalid_phase++;
	    }
	  }
	  if ((invalid_phase>0) || (invalid_period>0)) {
	    ans_sample->status = TSP_STATUS_ERROR_UNKNOWN;
	    TSP_common_sample_symbol_info_list_copy(&(ans_sample->symbols), req_sample->symbols);
	    STRACE_DEBUG(("Invalid phase or period"));
	  } else {

	    /* The datapool will be created here (if it does not already exist) */
	    if(TSP_session_create_symbols_table_by_channel(req_sample, ans_sample)) 
	      {
		/* FIXME do we need to start glu here for PASSIVE glu 
		 * instead of TSP_provider_private_run
		 */
		ans_sample->status = TSP_STATUS_OK;
	      }
	    else  
	      {
		STRACE_ERROR(("Function TSP_session_create_symbols_table_by_channel failed"));
	      }    
	  }    
	}
      else
	{
	  STRACE_WARNING(("Function TSP_session_get_symbols_global_index_by_channel failed"));
	  ans_sample->status = TSP_STATUS_ERROR_SYMBOLS;
	  /* Now we shall update answer_sample->symbols in order to indicates
	   * to consumer side which symbols are marked as 'unknown'
	   */
	  TSP_common_sample_symbol_info_list_copy(&(ans_sample->symbols), req_sample->symbols);
	}
    }
  else
    {
      STRACE_WARNING(("TSP version ERROR. Requested=%d Current=%d",req_sample->version_id, TSP_VERSION ));
      ans_sample->status = TSP_STATUS_ERROR_VERSION;
    }


	
  STRACE_IO(("-->OUT"));

  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
} /* End of TSP_provider_request_sample */

void  TSP_provider_request_sample_init(TSP_request_sample_init_t* req_sample_init, 
				       TSP_answer_sample_init_t* ans_sample)
{  
  int start_local_thread;
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);    
  STRACE_IO(("-->IN"));
    
  ans_sample->version_id = UNDEFINED_VERSION_ID;
  ans_sample->channel_id = req_sample_init->channel_id;
  ans_sample->status = TSP_STATUS_ERROR_UNKNOWN;
 
  if(req_sample_init->version_id <= TSP_VERSION)
    {
      ans_sample->version_id = req_sample_init->version_id;
      /* If the sample server is a lazy pasive server, we need a thread per session*/
      start_local_thread = ( X_glu_is_active ? FALSE : TRUE );
      
      if(TSP_session_create_data_sender_by_channel(req_sample_init->channel_id, start_local_thread))
	{
	  ans_sample->status = TSP_STATUS_OK;
	}
      else     
	{
	  STRACE_ERROR(("TSP_data_sender_create failed"));
	}
    
      /* send data address to client */
      ans_sample->data_address =
	(char*)TSP_session_get_data_address_string_by_channel(req_sample_init->channel_id);
  
      STRACE_DEBUG(("DATA_ADDRESS = '%s'", ans_sample->data_address));
    }
  else
    {
      STRACE_ERROR(("TSP version ERROR. Requested=%d Current=%d",req_sample_init->version_id, TSP_VERSION ));
      ans_sample->status = TSP_STATUS_ERROR_VERSION;
    }

  STRACE_IO(("-->OUT"));

  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
} /* End of TSP_provider_request_sample_init */

static void TSP_provider_request_sample_destroy_priv(channel_id_t channel_id)
{
  int stop_local_thread = ( X_glu_is_active ? FALSE : TRUE );  
  if(!TSP_session_destroy_data_sender_by_channel(channel_id, stop_local_thread))
    {
      STRACE_ERROR(("TSP_session_destroy_data_sender_by_channel failed"));
    }
}

void  TSP_provider_request_sample_destroy(TSP_request_sample_destroy_t* req_sample_destroy, 
					  TSP_answer_sample_destroy_t* ans_sample)
{
  TSP_LOCK_MUTEX(&X_tsp_request_mutex,);

  ans_sample->version_id = req_sample_destroy->version_id;
  ans_sample->channel_id = req_sample_destroy->channel_id;
  ans_sample->status = TSP_STATUS_OK;
 
  if(req_sample_destroy->version_id <= TSP_VERSION)
    {      
      TSP_provider_request_sample_destroy_priv(req_sample_destroy->channel_id);
    }
  else
    {
      STRACE_ERROR(("TSP version ERROR. Requested=%d Current=%d for session %d",
		    req_sample_destroy->version_id, TSP_VERSION,req_sample_destroy->channel_id  ));
      ans_sample->status = TSP_STATUS_ERROR_VERSION;
    }

  TSP_UNLOCK_MUTEX(&X_tsp_request_mutex,);
} /* End of  TSP_provider_request_sample_destroy */


void* TSP_provider_garbage_collector_thread(void* dummy)
{
   channel_id_t channel_id;

   /* Save memory ! */
   pthread_detach(pthread_self());

   while(TRUE)
     {
       while(TSP_session_get_garbage_session(&channel_id))
	 {
	   /* Do what some rude consumer should have done itself */
	   TSP_provider_request_sample_destroy_priv(channel_id);
	   TSP_provider_request_close_priv(channel_id);
	   STRACE_INFO(("Session No %d 'garbage-collected'", channel_id));
	 }
       tsp_usleep(TSP_GARBAGE_COLLECTOR_POLL_TIME_US);
     }

   /* never reached */
   return (void*)NULL;
}

int TSP_provider_private_init(GLU_handle_t* theGLU, int* argc, char** argv[])
{
  int ret = TRUE;
  int status;
  pthread_t thread;
  assert(argc);
  assert(argv);
  assert(theGLU);

  firstGLU = theGLU;
  
  ret = TSP_cmd_line_parser(argc, argv);
  if(ret)
     {
      /* init sessions */
      TSP_session_init();

      /* Initialise GLU server */
      ret = theGLU->initialize(theGLU, X_glu_argc, X_glu_argv);

      /* Launch garbage collection for sessions */
      status = pthread_create(&thread, NULL, TSP_provider_garbage_collector_thread,  NULL);
      TSP_CHECK_THREAD(status, FALSE);
      
    }

  if(!ret)
    {
      STRACE_INFO(("TSP init error"));
    }

  X_tsp_provider_init_ok = ret;
  
  return ret;
} /* End of TSP_provider_private_init */

int TSP_provider_private_run() {
  /* instantiate datapool */
  TSP_global_datapool_get_instance(firstGLU);
  /* Launch GLU now 
   */
  /* FIXME do we need to differentiate 
   * PASSIVE and ACTIVE GLU ?
   */
  /*  if (GLU_SERVER_TYPE_ACTIVE == firstGLU->type) { */
    firstGLU->start(firstGLU);
    /* } */
  return 0;
}


int TSP_provider_request_async_sample_write(TSP_async_sample_t* async_sample_write)
{
  int ret = TRUE;
  STRACE_IO(("-->IN"));

  STRACE_DEBUG(("TSP_PROVIDER Before async_write: pgi %d value %s return %d ",async_sample_write->provider_global_index,async_sample_write->data.data_val,ret ));
  
  ret = firstGLU->async_write(firstGLU,
			      async_sample_write->provider_global_index,
  			      async_sample_write->data.data_val,async_sample_write->data.data_len);

  if (0!=ret) {
    ret = FALSE;
  }

  STRACE_DEBUG(("TSP_PROVIDER After async_write: pgi %d value %s return %d ",async_sample_write->provider_global_index,async_sample_write->data.data_val,ret ));
  
  STRACE_IO(("-->OUT"));
  return ret;

} /* End of TSP_async_sample_write */


int TSP_provider_request_async_sample_read(TSP_async_sample_t* async_sample_read)
{
  int ret = TRUE;
  STRACE_IO(("-->IN"));

  STRACE_DEBUG(("TSP_PROVIDER Before async_read: pgi %d value %s return %d ",async_sample_read->provider_global_index,async_sample_read->data.data_val,ret ));
  
  ret = firstGLU->async_read(firstGLU,
			     async_sample_read->provider_global_index,
			     (async_sample_read->data.data_val),
			     &(async_sample_read->data.data_len));

  if (0 != ret) {
    ret = FALSE;
  } else {
    ret = TRUE;
  }

  STRACE_DEBUG(("TSP_PROVIDER After async_read: pgi %d value %f return %d ",async_sample_read->provider_global_index,*((double*)(async_sample_read->data.data_val)),ret ));
  
  STRACE_IO(("-->OUT"));
  return ret;

} /* End of TSP_async_sample_read */
