// -*- Mode: Java; indent-tabs-mode: nil; c-basic-offset: 4; -*-
/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
//----------------------------------------------------------------------
// $Header: /cvsroot/ant-doxygen/ant_task/src/org/doxygen/tools/Attic/DoxygenConfig.java,v 1.1.2.2 2004/01/31 01:38:56 akkumar Exp $
//
package org.doxygen.tools;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.PumpStreamHandler;
import org.apache.tools.ant.taskdefs.Execute;

import java.util.List;
import java.util.TreeMap;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Set;
import java.util.Iterator;
import java.util.Properties;
import java.util.Map;

import java.io.PrintStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.IOException;

/** This class holds the Configuration properties of Doxygen.
 *
 * @version $Revision: 1.1.2.2 $
 * @since Ant-Doxygen 1.3.1
 */
public class DoxygenConfig { 
    

    /** List of task-level properties. */
    private TreeMap taskAttributes = new TreeMap();

    /** List of nested properties. */
    private List nestedAttributes = new Vector();

  
  
    //----------------------------------------------------------------------
    /** 
     * @param dprocess DoxygenProcess to execute 'doxygen' command
     *
     * This constructor sets the initial Doxygen
     *  properties that are different from the default Doxygen
     *  configuration.  Namely (in Doxygen configuration file terms):
     *  <ul>
     *   <li> DETAILS_AT_TOP = YES
     *   <li> EXPAND_TOC = YES
     *   <li> FILE_PATTERNS = *.java
     *   <li> GENERATE_LATEX = NO
     *   <li> HAVE_DOT = YES
     *   <li> HIDE_UNDOC_MEMBERS = YES
     *   <li> PROJECT_NAME = (the Ant project name)
     *   <li> INLINE_SOURCE = YES
     *   <li> INPUT = src
     *   <li> OPTIMIZE_OUTPUT_JAVA = YES
     *   <li> OUTPUT_DIRECTORY = doc
     *   <li> QUIET = YES
     *   <li> RECURSIVE = YES
     *   <li> SOURCE_BROWSER = YES
     *  </ul>
     *
     *  \test Verify that each of these items are set to these values
     *  in the amalgamated configuration file.
     */
    public DoxygenConfig() {
        setProperty("DETAILS_AT_TOP", true);
        setProperty("FILE_PATTERNS", "*.java");
        setProperty("GENERATE_LATEX", false);
        setProperty("HAVE_DOT", false);
        setProperty("HIDE_UNDOC_MEMBERS", true);
        setProperty("INLINE_SOURCES", true);
        setProperty("INPUT", "src");
        setProperty("GENERATE_TREEVIEW", true);
        setProperty("OPTIMIZE_OUTPUT_JAVA", true);
        setProperty("OUTPUT_DIRECTORY", "doc");
        setProperty("QUIET", true);
        setProperty("RECURSIVE", true);
        setProperty("SOURCE_BROWSER", true);
        setProperty("TOC_EXPAND", true);
    }
    //----------------------------------------------------------------------
    /** This method translates an Ant &lt;doxygen&gt; task element
     *  into a Doxygen configuration file property name/value
     *  pair. <br/>
     *
     *  \test String properties containing spaces are double
     *  quoted. <br/>
     *
     *  @param keyName for this property.
     *  @param value for this property.
     *  @param minVersion required to use this property.
     *
     */
    public final void setProperty(final String keyName,
                                  final String value) {
        DoxygenTask.Property attribute = getAttribute(keyName);
        taskAttributes.put(keyName, attribute);

        String val = value;
        if (val.indexOf(' ') > -1) { val =  "\"" + val + "\""; }
        attribute.setValue(val);

    }




    //----------------------------------------------------------------------
    /** This method translates an Ant &lt;doxygen&gt; task element into a
     *  Doxygen configuration file property name/value pair.
     *
     *  @param keyName for this property.
     *  @param value for this property.
     *  @param minVersion required to use this property.
     *
     */
    public final void setProperty(final String keyName,
                                  final int value) {
        DoxygenTask.Property attribute = getAttribute(keyName);
        taskAttributes.put(keyName, attribute);
        attribute.setValue("" + value);
    }




    //----------------------------------------------------------------------
    /** This method translates an Ant &lt;doxygen&gt; task element into a
     *  Doxygen configuration file property name/value pair. <br/>
     *
     *  \test true / false values become "YES" / "NO" values.
     *
     *  @param keyName for this property.
     *  @param value for this property.
     *  @param minVersion required to use this property.
     *
     */
    public final void setProperty(final String keyName,
                                  final boolean value) {
        DoxygenTask.Property attribute = getAttribute(keyName);
        taskAttributes.put(keyName, attribute);

        String val = "YES";
        if  (!value) { val = "NO"; }
        attribute.setValue(val);
    }





    //----------------------------------------------------------------------
    /** This method returns the task attributes for jUnit test
     *  analysis. <br/>
     *
     *  \note this method is for jUnit testing.
     *
     *  @return a <code>TreeMap</code> containing all &lt;doxygen&gt;
     *  tag attributes set by this Ant task.
     */
    public final TreeMap getTaskAttributes() {
        return taskAttributes;
    }





    //----------------------------------------------------------------------
    /** This method returns the list of nested attributes for jUnit
     *  test analysis. <br/>
     *
     *  \note this method is for jUnit testing. <br/>
     *
     *  @return a <code>List</code> containing all nested attributes
     *  set by this Ant task.
     */
    public final List getNestedAttributes() {
        return nestedAttributes;
    }





    //----------------------------------------------------------------------
    /** This method returns the attributes for jUnit test analysis.
     *
     *  @param keyName to be retreived.
     *
     *  @return a <code>Property</code> containing as much of the
     *  specified attribute as is currently set.
     *
     */
    public final DoxygenTask.Property getAttribute(final String keyName) {
        DoxygenTask.Property retval = null;
        if  (taskAttributes.containsKey(keyName)) {
            retval = (DoxygenTask.Property) taskAttributes.get(keyName);
        }
        if  (retval == null) {
            retval = new DoxygenTask.Property();
            retval.setName(keyName);
        }
        return retval;
    }
    
    //--------------------------------------------------
    /**
     * Add Nested Attribute to the already existing list.
     **/
    public void addNestedAttribute(DoxygenTask.Property attr) {
        nestedAttributes.add(attr);        
            
    }

    //----------------------------------------------------------------------
    /** This method writes and synchronizes the properties. <br/>
     *
     *  \test If specified, the base configuration file is not
     *  overwritten.
     *
     *  @param theConfigFilename used by Doxygen.
     *
     */
    public final void writeDoxygenConfig(final String theConfigFilename) {

        PrintStream ps = null;
        TreeMap map = readDoxygenConfig(theConfigFilename);
        cascadeDoxygenConfig(map);
        try {
            ps = new PrintStream(
                new FileOutputStream(DoxygenTask.CONFIG_FILE));
            Set keys = map.entrySet();

            Iterator i = keys.iterator();
            while (i.hasNext()) {
                Map.Entry me = (Map.Entry) i.next();
                String param = (String) me.getKey();
                String value = (String) me.getValue();
                String line  = param + "\t=";
                if  (value != null) { line += " " + value; }
                ps.println(line);
            }
/*            activityLog(false, "Updated Doxygen config file: "
                        + "[" + DoxygenTask.CONFIG_FILE + "]");
                        */
        } catch (IOException ioe) {
            throw new BuildException("Unable to update Doxygen config file: "
                                     + "[" + theConfigFilename + "]", ioe);
        } finally {
            if  (ps != null) {
                ps.close();
                ps = null;
            }
        }
    }





    //----------------------------------------------------------------------
    /** This method reads the Doxygen generated configuration file. <br/>
     *
     *  \note Due to the use of java.util.Properties, all comments in
     *  the base configuration file are dropped in the almagamated
     *  configuration file.  This is acceptable, since the base
     *  configuration file is not overwritten and the almagamated file
     *  is alphabetized. <br/>
     *
     *  \bug ID=xxxxxx: In controlling Doxygen config file generation
     *  verbosity, that output now appears in the config file input
     *  stream. <br/>
     *
     *  \bug ID=xxxxxx: Related to input stream mangling.
     *  PROJECT_NUMBER is loaded at ROJECT_NUMBER, which Doxygen
     *  complains about.  Small fixup to reflect it back to the proper
     *  parameter name. <br/>
     *
     *  @param theConfigFilename generated by Doxygen -g.
     *
     *  @return a <code>TreeMap</code> containing all of the Doxygen
     *  parameters to be used in the amalgamated configuration file.
     *
     */
    public final TreeMap readDoxygenConfig(String theConfigFilename) {
        TreeMap map = new TreeMap();
        Properties p = new Properties();
        try {
            p.load(new FileInputStream(theConfigFilename));
            if  (p.containsKey("Configuration")) {      // bug ID=xxxxxx
                p.remove("Configuration");
                p.remove("to");
                p.remove("doxygen");
                p.remove("Now");
                p.remove("");
            }
            if (p.containsKey("ROJECT_NUMBER")) {       // bug ID=xxxxxx
                p.setProperty("PROJECT_NUMBER", p.getProperty("ROJECT_NUMBER"));
                p.remove("ROJECT_NUMBER");
            }
            Enumeration e = p.keys();
            while (e.hasMoreElements()) {
                String arg = (String) e.nextElement();
                map.put(arg, p.getProperty(arg));
            }
        } catch (IOException ioe) {
            throw new BuildException("Unable to read Doxygen config file: "
                                     + "[" + theConfigFilename + "]", ioe);
        }
        return map;
    }

    //----------------------------------------------------------------------
    /** This method cascades all Ant task attribute and nest attribute
     *  values into the passed TreeMap instance.  This TreeMap
     *  instance is supposed to have been populated from the generated
     *  or user-specified Doxygen configuraton file. <br/>
     *
     *  \test the minimalist case has open-source / Java flavor
     *  Doxygen parameters set. <br/>
     *
     *  \test &lt;doxygen&gt; elements can influence Doxygen. <br/>
     *
     *  \test &lt;doxygen&gt; nested elements can influence Doxygen. <br/>
     *
     *  \test &lt;doxygen&gt; nested elements take precedence over
     *  &lt;doxygen&gt; non-nested elements. <br/>
     *
     *  \todo Add the Doxygen parameter version checking. <br/>
     *
     *  @param map contains all of the Doxygen configuration file basis.
     *
     */
    public final void cascadeDoxygenConfig(final TreeMap map) {
        if  (taskAttributes.size() > 0) {
            Iterator iter = taskAttributes.values().iterator();
            while (iter.hasNext()) {
                DoxygenTask.Property attribute = 
                        (DoxygenTask.Property) iter.next();
                map.put(attribute.getName(), attribute.getValue());
            }
        }
        if  (nestedAttributes.size() > 0) {
            Iterator iter = nestedAttributes.iterator();
            while (iter.hasNext()) {
                DoxygenTask.Property attribute = 
                        (DoxygenTask.Property) iter.next();
                map.put(attribute.getName(), attribute.getValue());
            }
        }
    }
}
