// ball.cc - functions required by different balls, etc.
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <cmath>
#include <string>

#include "ball.h"
#include "misc.h"

// change velocity of ball on bouncing with tunnell wall
void balls::bounce_wall(ball* b)
{
	// calculate cos(2t) and sin(2t), where x = r cos(t), y = r sin(t)
	double sqx = misc::square(b -> l.x());
	double sqy = misc::square(b -> l.y());
	double txy = 2 * b -> l.x() * b -> l.y();
	double cos2t = (sqx - sqy) / (sqx + sqy);
	double sin2t = txy / (sqx + sqy);
	// Then do a rotation of velocity by (2t + pi)
	double vx = b -> v.x();
	double vy = b -> v.y();
	b -> v.setx(-vx * cos2t - vy * sin2t);
	b -> v.sety(vy * cos2t - vx * sin2t);
}

// make sure ball is on inside of wall
void balls::get_inside_wall(ball* b)
{
	double newr = tunnel_r - b -> rad;
	double ang = atan2(b -> l.y(), b -> l.x());
	b -> l.setx(newr * std::cos(ang)).sety(newr * std::sin(ang));
}

// check if this ball hits the tunnel wall (tunnel radius == tunnel_r)
bool balls::hit_wall(const ball* b)
{
	return misc::square(b -> loc().x()) + misc::square(b -> loc().y())
		> misc::square(tunnel_r - b -> radius());
}


// check if b1 and b2 are in contact
bool balls::collide(const ball* b1, const ball* b2)
{
	using misc::square;
	// First check if the cubes enclosing the two spheres are
	// overlapping. Then do a more accurate check if they are.
	// NOTE: If the balls are moving away from each other, they
	//       do NOT collide.
	// In most of the cases, we get out by the first conditions.

	double dx = b1->loc().x() - b2->loc().x();
	double rt = b1->radius() + b2->radius();
	if (dx > rt || dx < -rt)
		return false;
	double dy = b1->loc().y() - b2->loc().y();
	if (dy > rt || dy < -rt)
		return false;
	double dz = b1->loc().z() - b2->loc().z();
	if (dz > rt || dz < -rt)
		return false;
	if (square(dx) + square(dy) + square(dz) > square(rt))
		return false;
	if (dotp(b1->loc() - b2->loc(), b1->vel() - b2->vel()) > 0)
		return false;
	return true;
}

// calculate the change in velocity for b1 caused by a collision with b2
balls::vect3d balls::dvel_collision(const ball* b1, const ball* b2)
{
	// if V1 = change in velocity (vector)
	//    C  = displacent between centres of balls (vector)
	//    U1 = initial velocity (vector)
	//    U2 = initial velocity of other ball (vector)
	//    m1 = mass (scalar)
	//    m2 = mass of other ball (scalar),
	//     V1x   V1y   V1z    2 [ (U2 - U1) . C ]
	// t = --- = --- = --- = ---------------------
	//     Cx    Cy    Cz    (1 + m1 / m2) (C . C)
	if (b2->mass() == 0)
		return vect3d(0, 0, 0);

	vect3d C = b1->loc() - b2->loc();
	double CC = square(C);
	vect3d U21 = b2->vel() - b1->vel();
	double U21C = dotp(U21, C);
	double t = 2 * U21C  / (1 + b1->mass() / b2->mass()) / CC;
	return C * t;
}

void balls::enemy::move(double t)
{
	// Modify vel because of collisions.
	v += dv;
	dv.set();

	// Add velocity * time to distance
	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	if (strength <= 0)
		die();
}

void balls::enemy::do_collide(const ball* other)
{
	// enemy balls are not damaged when bouncing against each other
	if (!is_enemy(other)) {
		double d = other -> damage();
		double shield_loss = d * 2 / 3;
		if (shield_loss > shield)
			shield_loss = shield;
		shield -= shield_loss;
		strength -= d - shield_loss;
		if (strength <= 0) {
			const ship_bullet* sb = dynamic_cast<const ship_bullet*>(other);
			if (sb)
				sb->add_score(score);
		}
	}
	dv += dvel_collision(this, other);
}

double balls::enemy::damage() const
{
	return mas;
}

void balls::bullet::move(double t)
{
	l += v * t;

	if (hit_wall(this) || hit)
		die();
}

void balls::bullet::do_collide(const ball* other)
{
	hit = true;
}

double balls::bullet::damage() const
{
	return dam;
}

void balls::factory_register(factory_map& fm, const std::string& name, factory* f)
{
	fm.add(name);
	fm[name] = f;
}

void balls::factory_register(factory_map& fm, const char* name, factory* f)
{
	factory_register(fm, std::string(name), f);
}

// Local Variables:
// mode: c++
// End:
