// game.cc - functions having to do with the game
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <algorithm>
#include <fstream>
#include <functional>
#include <iterator>
#include <list>

#include "ball.h"
#include "enemy.h"
#include "ship.h"
#include "game.h"
#include "new_ball.h"

namespace Game {

	class panel {
	public:
		panel(graphics::tsurface* t, int x, int y, int w, int h,
		      double vmin, double vmax, double smin, double smax,
		      double hmin, double hmax, double amin, double amax,
		      const graphics::raw_image_map& raw_images);
		~panel();

		void draw_first() const;
		void draw() const;

		int score;
		int depth;
		int time;

		double velocity, aminv, cminv, cmaxv, amaxv;
		double strength, amins, cmins, cmaxs, amaxs;
		double shield, aminh, cminh, cmaxh, amaxh;
		double ammo, amina, cmina, cmaxa, amaxa;
	private:
		enum image_no {
			iscore = 10, idepth, itime,
			ivel, istr, ihard, iammo, itotal
		};
		graphics::image* images[itotal];

		enum dial_no {
			dvel, dstr, dhard, dammo, dtotal
		};
		graphics::dial* dials[dtotal];

		static const int draw_items = 14;
		int ys[draw_items + 1];

		void draw_num(int n, int x, int y) const;

		graphics::tsurface* ts;
		int X, Y;

		int unitx, unity, whole_width;

		panel(const panel&);
		panel& operator=(const panel&);
	};

	class frame {
	public:
		frame(graphics::tsurface* ts,
		      int x1, int x2, int x3, int y1, int y2, int w, int h,
		      const graphics::raw_image_map& raw_images);
		~frame();

		void draw() const;
	private:
		enum frame_no{
			tl, th1, tc, th2, tr, v1, v2, v3, bl, bh1, bc, bh2, br, total
		};

		graphics::image* images[total];
		int x[total], y[total];

		frame(const frame&);
		frame& operator=(const frame&);
	};

} // namespace Game

// Coversion factor of depth to depth score
const double Game::game::depthconv = 4.0 / tunnel_d;
const double Game::game::timeconv = 1.0 / 1000;

Game::panel::panel(graphics::tsurface* t, int x, int y, int w, int h,
		   double vmin, double vmax, double smin, double smax,
		   double hmin, double hmax, double amin, double amax,
		   const graphics::raw_image_map& raw_images)
	: score(0), depth(0), time(0),
	  aminv(vmin), amaxv(vmax), amins(smin), amaxs(smax),
	  aminh(hmin), amaxh(hmax), amina(amin), amaxa(amax),
	  ts(t), X(x), Y(y)
{
	const graphics::raw_image* rimages[itotal];
	rimages[0] = raw_images["number0"];
	rimages[1] = raw_images["number1"];
	rimages[2] = raw_images["number2"];
	rimages[3] = raw_images["number3"];
	rimages[4] = raw_images["number4"];
	rimages[5] = raw_images["number5"];
	rimages[6] = raw_images["number6"];
	rimages[7] = raw_images["number7"];
	rimages[8] = raw_images["number8"];
	rimages[9] = raw_images["number9"];
	rimages[iscore] = raw_images["score"];
	rimages[idepth] = raw_images["depth"];
	rimages[itime] = raw_images["time"];
	rimages[ivel] = raw_images["velocity"];
	rimages[istr] = raw_images["strength"];
	rimages[ihard] = raw_images["shield"];
	rimages[iammo] = raw_images["ammo"];

	unitx = rimages[0] -> w();
	unity = rimages[0] -> h();
	int sep = unity * 4 / 18;
	int isep = unity * 9 / 18;
	int ssep = unity * 11 / 18;
	int dh = unity * 9 / 18;

	ys[0] = 0; // SCORE
	ys[1] = ys[0] + rimages[iscore]->h() + sep; // number
	ys[2] = ys[1] + unity + isep; // DEPTH
	ys[3] = ys[2] + rimages[idepth]->h() + sep; // number
	ys[4] = ys[3] + unity + isep; // TIME
	ys[5] = ys[4] + rimages[itime]->h() + sep; // number
	ys[6] = ys[5] + unity + ssep; // VELOCITY
	ys[7] = ys[6] + rimages[ivel]->h() + sep; // dial
	ys[8] = ys[7] + dh + isep; // STRENGTH
	ys[9] = ys[8] + rimages[istr]->h() + sep; // dial
	ys[10] = ys[9] + dh + isep; // SHIELD
	ys[11] = ys[10] + rimages[ihard]->h() + sep; // dial
	ys[12] = ys[11] + dh + isep; // AMMO
	ys[13] = ys[12] + rimages[iammo]->h() + sep; // dial
	ys[14] = ys[13] + dh; // bottom

	int rw = 7 * unitx;
	for (int i = 10; i < itotal; ++i)
		if (rimages[i]->w() > rw)
			rw = rimages[i]->w();
	int rh = ys[draw_items];

	whole_width = w;
	for (int i = 0; i <= draw_items; ++i) {
		ys[i] = Y + ys[i] * h / rh;
	}

	unitx = unitx * w / rw;
	unity = unity * h / rh;

	for (int i = 0; i < itotal; ++i) {
		const graphics::raw_image* ri = rimages[i];
		images[i] = ts->make_image(ri, ri->w() * w / rw, ri->h() * h / rh);
	}

	const graphics::raw_image* rdial = raw_images["dial"];
	dials[dvel] = ts->make_dial(rdial, vmin, vmax, w, dh * h / rh);
	dials[dstr] = ts->make_dial(rdial, smin, smax, w, dh * h / rh);
	dials[dhard] = ts->make_dial(rdial, hmin, hmax, w, dh * h / rh);
	dials[dammo] = ts->make_dial(rdial, amin, amax, w, dh * h / rh);
}

Game::panel::~panel()
{
	for (int i = 0; i < itotal; ++i)
		delete images[i];
}

void Game::panel::draw_first() const
{
	images[iscore]->draw(X, ys[0]);
	images[idepth]->draw(X, ys[2]);
	images[itime]->draw(X, ys[4]);
	images[ivel]->draw(X, ys[6]);
	images[istr]->draw(X, ys[8]);
	images[ihard]->draw(X, ys[10]);
	images[iammo]->draw(X, ys[12]);
}

void Game::panel::draw() const
{
	ts->clear_rect(X, ys[1], whole_width, unity, 0, 0, 0);
	draw_num(score, X, ys[1]);
	ts->clear_rect(X, ys[3], whole_width, unity, 0, 0, 0);
	draw_num(depth, X, ys[3]);
	ts->clear_rect(X, ys[5], whole_width, unity, 0, 0, 0);
	draw_num(time, X, ys[5]);
	dials[dvel]->draw(X, ys[7], velocity, cminv, cmaxv);
	dials[dstr]->draw(X, ys[9], strength, cmins, cmaxs);
	dials[dhard]->draw(X, ys[11], shield, cminh, cmaxh);
	dials[dammo]->draw(X, ys[13], ammo, cmina, cmaxa);
}

void Game::panel::draw_num(int n, int x, int y) const
{
	int digit[7];
	int len = 0;
	if (n < 0)
		n = 0;
	if (n > 9999999)
		n = 9999999;
	do {
		digit[len] = n % 10;
		n /= 10;
		++len;
	} while(n);

	while (--len >= 0) {
		images[digit[len]]->draw(x, y);
		x += unitx;
	}
}

Game::frame::frame(graphics::tsurface* ts,
                   int x1, int x2, int x3, int y1, int y2, int w, int h,
                   const graphics::raw_image_map& raw_images)
{
	images[tl] = ts->make_image(raw_images["frame_tl"], w, h);
	x[tl] = x1;
	y[tl] = y1;

	images[th1] = ts->make_image(raw_images["frame_h"], x2 - (x1+w), h);
	x[th1] = x1+w;
	y[th1] = y1;

	images[tc] = ts->make_image(raw_images["frame_t"], w, h);
	x[tc] = x2;
	y[tc] = y1;

	images[th2] = ts->make_image(raw_images["frame_h"], x3 - (x2+w), h);
	x[th2] = x2+w;
	y[th2] = y1;

	images[tr] = ts->make_image(raw_images["frame_tr"], w, h);
	x[tr] = x3;
	y[tr] = y1;

	images[v1] = ts->make_image(raw_images["frame_v"], w, y2 - (y1+h));
	x[v1] = x1;
	y[v1] = y1+h;

	images[v2] = images[v1];
	x[v2] = x2;
	y[v2] = y1+h;

	images[v3] = images[v1];
	x[v3] = x3;
	y[v3] = y1+h;

	images[bl] = ts->make_image(raw_images["frame_bl"], w, h);
	x[bl] = x1;
	y[bl] = y2;

	images[bh1] = images[th1];
	x[bh1] = x1+w;
	y[bh1] = y2;

	images[bc] = ts->make_image(raw_images["frame_b"], w, h);
	x[bc] = x2;
	y[bc] = y2;

	images[bh2] = images[th2];
	x[bh2] = x2+w;
	y[bh2] = y2;

	images[br] = ts->make_image(raw_images["frame_br"], w, h);
	x[br] = x3;
	y[br] = y2;
}

Game::frame::~frame()
{
	// DO NOT delete all images, some point to same, namely:
	// th1, bh1
	// th2, bh2
	// v1, v2, v3
	delete images[br];
	delete images[bc];
	delete images[bl];
	delete images[v1];
	delete images[tr];
	delete images[th2];
	delete images[tc];
	delete images[th1];
	delete images[tl];
}

void Game::frame::draw() const
{
	for (int i = 0; i < total; ++i)
		images[i]->draw(x[i], y[i]);
}

Game::game::game(int xr, int yr, const rec::record& options,
                 graphics::tsurface* ts, audio::taudio* ta,
                 const rec::record* b_d, control::controller* c,
                 misc::random& r)
	: xres(xr), yres(yr),
	  Time_passed(0), Depth_moved(0),
	  t_cx(getx(options["center_x"])),
	  t_cy(gety(options["center_y"])),
	  t_w(getx(options["width"])),
	  t_h(gety(options["height"])),
	  wicon_x(getx(options["weapon_x"])),
	  wicon_y(gety(options["weapon_y"])),
	  wicon_w(getx(options["weapon_w"])),
	  wicon_h(gety(options["weapon_h"])),
	  Stat(start),
	  time_wo_ship(options["time_wo_ship"]),
	  time_left(time_wo_ship),
	  rnd(new misc::random),
	  bd(b_d),
	  where_to_draw(ts),
	  where_to_speak(ta),
	  proj(tunnel_d, tunnel_d, tunnel_r, t_w, t_h),
	  mapp(proj, t_cx, t_cy),
	  cont(c)
{
	// fill up factories
	balls::enemy_register(factories);
	balls::ship_register(factories);
	
	// fill up gallery

	graphics::raw_image_map raw_images;
	// read raw images
	{
		std::string image_dir(options["image_dir"]);
		std::ifstream is((image_dir + "/game_index").c_str());
		rec::record r;
		is >> r;
		for (rec::record::iconst_iterator ci = r.begin();
		     ci != r.end(); ++ci) {

			raw_images.add(ci->first);
			raw_images[ci->first]
				= ts->make_raw_image(image_dir + "/"
						     + std::string(ci->second));
		}
	}

	const rec::record& ship_rec = b_d->sub("ship");
	pp = new panel(ts,
		       getx(options["panel_x"]),
		       gety(options["panel_y"]),
		       getx(options["panel_w"]),
		       gety(options["panel_h"]),
		       ship_rec["vmin"],
		       ship_rec["vmax"],
		       ship_rec["smin"],
		       ship_rec["smax"],
		       ship_rec["hmin"],
		       ship_rec["hmax"],
		       ship_rec["amin"],
		       ship_rec["amax"],
		       raw_images);

	fp = new frame(ts,
		       getx(options["frame_x1"]),
		       getx(options["frame_x2"]),
		       getx(options["frame_x3"]),
		       gety(options["frame_y1"]),
		       gety(options["frame_y2"]),
		       getx(options["frame_w"]),
		       gety(options["frame_h"]),
		       raw_images);

	// read tunnel gradient
	tun = ts->make_tunnel(raw_images["tunnel_gradient"], &mapp);

	// read weapon icons and deep_images
	for (rec::record::rconst_iterator ci = bd->sub_begin();
	     ci != bd->sub_end(); ++ci) {
		const rec::record& r = ci->second;
		if (r.has("deep_image")) {
			double dia = 2 * double(r["radius"]);
			if (raw_images.find(r["deep_image"]) == raw_images.end())
				throw std::string("entry ") + std::string(r["deep_image"])
					+ " not found in images index";
			graphics::raw_image* ri = raw_images[r["deep_image"]];
			gallery.add(ci->first);
			gallery[ci->first]
				= ts->make_deep_image(ri, 32, proj, dia, dia);
		}
		if (r.has("wicon")) {
			if (raw_images.find(r["wicon"]) == raw_images.end())
				throw std::string("entry ") + std::string(r["wicon"])
					+ " not found in images index";
			graphics::raw_image* ri = raw_images[r["wicon"]];
			wicons.add(ci->first);
			wicons[ci->first] = ts->make_image(ri, wicon_w, wicon_h);
		}
	}


	for (graphics::raw_image_map::const_iterator ci = raw_images.begin();
	     ci != raw_images.end(); ++ci) {

		delete ci->second;
	}

	// read sounds
	{
		std::string audio_dir(options["audio_dir"]);
		std::ifstream is((audio_dir + "/index").c_str());
		rec::record r;
		is >> r;
		for (rec::record::iconst_iterator ci = r.begin();
		     ci != r.end(); ++ci) {

			audio::raw_sound* rs = ta->make_raw_sound(audio_dir + "/" + std::string(ci->second));
			audio::sound* so = ta->make_sound(rs);
			delete rs;
			sounds.add(ci->first);
			sounds[ci->first] = so;
		}
	}

	// First of all, we create the ship and the master_ball.
	sp = new balls::ship(misc::vect3d(0, 0, 0),
			     misc::vect3d(0, 0, bd->sub("ship", "minvelz")),
			     this, cont);

	{
		std::ifstream is((std::string(options["level_loc"])).c_str());
		is >> levels;
	}

	crea = new balls::level_creator(this, levels);

	flyers.push_back(sp);
}

Game::game::~game()
{
	for (std::list<ball*>::iterator i = flyers.begin();
	     i != flyers.end(); ++i) {

		if (*i != sp)
			delete *i;
	}

	delete crea;
	delete sp;

	for (audio::sound_map::iterator i = sounds.begin();
	     i != sounds.end(); ++i) {

		delete i->second;
	}

	for (graphics::image_map::iterator i = wicons.begin();
	     i != wicons.end(); ++i) {

		delete i->second;
	}

	for (graphics::deep_image_map::iterator i = gallery.begin();
	     i != gallery.end(); ++i) {

		delete i->second;
	}

	for (balls::factory_map::iterator i = factories.begin();
	     i != factories.end(); ++i) {

		delete i->second;
	}

	delete tun;
	delete fp;
	delete pp;
	delete rnd;
}

// play the game for some duration
void Game::game::play(double dur)
{
	// make sure the game is not over yet
	if (Stat == done)
		return;

	// set Stat to run in case the game was paused or just starting
	Stat = run;

	Time_passed += dur;

	// Every item must make a move. In the move, if a ball
	// is broken up, its remove flag is set.
	move_flyers(dur);

	// Then we sort them according to z coordinate. Note that before
	// sorting, broken up balls are removed.
	sort_flyers();

	// Make sure that the ship has depth 0, and move the
	// other balls accordingly so that there is no relative movement.
	// This should also update depth.
	// Also check if there are any collisions between balls and
	// inform concerned balls.
	check_flyers_depth_and_collisions();

	// See if we are to go on. If there aren't any ships, start
	// decrementing time_left until we can go out.
	if (sp->is_dead() && (time_left -= dur) <= 0)
		Stat = done;
}

void Game::game::draw_first()
{
	// clear
	where_to_draw->clear(0, 0, 0);

	// draw frame
	fp->draw();

	// draw panel first time
	pp->draw_first();
}

void Game::game::draw()
{
	tun->draw(misc::vect3d(0, 0, 0));
	balls::ship* s = static_cast<balls::ship*>(sp);
	pp->score = s->score;
	pp->depth = intdepth(Depth_moved);
	pp->time = inttime(Time_passed);
	pp->velocity = s->vel().z();
	pp->cminv = s->min_zvel;
	pp->cmaxv = s->max_zvel;
	pp->strength = s->strength;
	pp->cmins = 0;
	pp->cmaxs = pp->amaxs;
	pp->shield = s->shield;
	pp->cminh = 0;
	pp->cmaxh = pp->amaxh;
	pp->ammo = s->ammo;
	pp->cmina = 0;
	pp->cmaxa = pp->amaxa;
	pp->draw();

	where_to_draw->clear_rect(wicon_x, wicon_y, wicon_w, wicon_h, 0, 0, 0);
	if (s->cur_weap != s->weapons.end())
		(*s->cur_weap)->get_image()->draw(wicon_x, wicon_y);

	// first draw tunnel

	// then draw objects
	// We start from the end as the deepest balls are at the end of
	// the list.
	where_to_draw->clip_rect(t_cx - t_w/2, t_cy - t_h/2, t_w&~1, t_h&~1);
	std::for_each(flyers.rbegin(), flyers.rend(), balls::do_ball_draw());
	where_to_draw->unclip();
}

int Game::game::score() const
{
	return pp->score;
}

double Game::game::depth_moved() const
{
	return Depth_moved;
}

double Game::game::time_passed() const
{
	return Time_passed;
}

void Game::game::move_flyers(double dur)
{
	for (std::list<ball*>::iterator i = flyers.begin();
	     i != flyers.end(); ++i) {
		(*i)->move(dur);
	}
	crea->move(dur);
}

void Game::game::sort_flyers()
{
	// first remove removables in, then sort flyers

	for (std::list<ball*>::iterator i = flyers.begin();
	     i != flyers.end(); ) {
		double d = (*i)->loc().z();
		if (d < mapp.get_depth()
		    || d > mapp.get_depth() + tunnel_dpth + tunnel_r) {

			(*i)->die();
		}
		if ((*i) -> is_dead()) {
			if (*i != sp)
				delete *i;
			i = flyers.erase(i);
		}
		else
			++i;
	}
	flyers.sort(balls::above);
}

void Game::game::check_flyers_depth_and_collisions()
{
	if (flyers.empty())
		return;

	// All balls, (including ship sp), will be moved up by
	// sp -> depth().
	// Then, check which balls below them they collide with

	double depth = sp -> loc().z();
	Depth_moved = depth;

	tun->move(depth);
	mapp.set_depth(depth);

	// We have to check for collisions with the ball at first.
	// If there are no more balls (1st condition of nested for),
	// or the remaining balls are all well below the ball at first
	// (2nd condition), stop.
	for (std::list<ball*>::iterator i = flyers.begin();
	     i != flyers.end(); ++i) {

		std::list<ball*>::iterator other = i;
		for (++other; other != flyers.end(); ++other) {

			// first check if remaining balls are well below
			if ((*other)->loc().z() - (*i)->loc().z() > (*other)->radius() + (*i)->radius())
				break;
			if (balls::collide(*other, *i)) {
				(*other) -> do_collide(*i);
				(*i) -> do_collide(*other);
			}
		}

	}
}

// Local Variables:
// mode: c++
// End:
