// game.h - gives the game class
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef GAME_H
#define GAME_H

#include <list>

#include "audio.h"
#include "ball.h"
#include "graphics.h"
#include "controller.h"
#include "timer.h"
#include "misc.h"
#include "record.h"

namespace Game {
	class panel;
	class frame;

	using balls::ball;

	class game {
	public:
		enum status {
			start, run, pause, done
		};

		game(int xr, int yr,
		     const rec::record& options,
		     graphics::tsurface* ts,
		     audio::taudio* ta,
		     const rec::record* b_d,
		     control::controller* c,
		     misc::random& r);
		~game();

		void play(double dur);
		void draw_first();
		void draw();

		int score() const;
		double depth_moved() const;
		double time_passed() const;

		status stat() const		{ return Stat; }

		const ball* get_ship() const	{ return sp; }
		void skip_level()		{ crea->skip_level(); }
		void swallow(ball* b)		{ flyers.push_front(b); }

		const balls::factory_map& get_factories() const
		{ return factories; }
		const graphics::mapper& get_mapp() const
		{ return mapp; }
		const graphics::deep_image_map& get_gallery() const
		{ return gallery; }
		const graphics::image_map& get_wicons() const
		{ return wicons; }
		const audio::sound_map& get_sounds() const
		{ return sounds; }
		const rec::record& get_bd() const
		{ return *bd; }
		misc::random& rand() const
		{ return *rnd; }
	private:
		int xres, yres;
		int getx(int x) const		{ return x * xres / 640; }
		int gety(int y) const		{ return y * yres / 480; }

		double Time_passed;
		double Depth_moved;
		int t_cx, t_cy, t_w, t_h;
		int wicon_x, wicon_y, wicon_w, wicon_h;
		status Stat;
		double time_wo_ship, time_left;

		misc::random* rnd;

		const rec::record* bd;
		balls::factory_map factories;

		rec::record levels;

		graphics::tsurface* where_to_draw;
		audio::taudio* where_to_speak;
		graphics::projector proj;
		graphics::mapper mapp;

		control::controller* cont;

		panel* pp;
		frame* fp;

		graphics::tunnel_drawable* tun;

		graphics::deep_image_map gallery;
		graphics::image_map wicons;
		audio::sound_map sounds;

		ball* sp;
		std::list<ball*> flyers;
		balls::creator* crea;

		void move_flyers(double dur);
		void sort_flyers();
		void check_flyers_depth_and_collisions();

		int intdepth(double d)	{ return int(d * depthconv); }
		int inttime(double d) { return int(d * timeconv); }
		static const double depthconv;
		static const double timeconv;

		game(const game&);
		game& operator=(const game&);
	};

} // namespace Game

#endif // !GAME_H

// Local Variables:
// mode: c++
// End:
