// graphics.cc
//
// Copyright (C) 2000 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "graphics.h"

// ellipse initialization
//    initializes needed vars:
// rx, ry are x, y radii
// cx, cy are current point
// f is the function (ry^2 * cx + rx^2 * cy - rx^2 * ry^2), which should be
//    0 if (cx, cy) is on ellipse, -ve if (cx, cy) is inside ellipse,
//    +ve if (cx, cy) is outside ellipse.
// dfx is how much f should change when x is incremented
//    = ry^2 * [(x+1)^2 - x^2] = ry^2 * (2*x + 1)
// dfy is how much f should change when y is decremented
//    = rx^2 * [(y-1)^2 - y^2] = rx^2 * (1 - 2*y)
// ddfx is how much dfx should change when x is incremented
//    = 2 * ry^2
// ddfy is how much dfy should change when y is decremented
//    = 2 * rx^2
// second_stage indicates that the gradient has changed from the range
//    [0, -1] to the range [-1, -inf]. Note that if !second_stage, x
//    always increases and if second_stage y always decreases
void graphics::ellipse::reset(int Rx, int Ry)
{
	rx = Rx;
	ry = Ry;
	cx = 0;
	cy = ry;
	f = 0;
	dfx = ry * ry;
	dfy = rx * rx * (1 - 2 * cy);
	ddfx = 2 * dfx;
	ddfy = 2 * rx * rx;
	second_stage = false;
}

// move to next point on ellipse
bool graphics::ellipse::next()
{
	// check if we are ready
	if (cx == rx && cy == 0)
		return false;

	if (!second_stage) {
		// we are still in the first stage where x always increases
		++cx;
		f += dfx;
		dfx += ddfx;

		// if point is out of ellipse, then decrement y
		if (f > 0) {
			--cy;
			f += dfy;
			dfy += ddfy;

			// if point is still outside ellipse, then x
			// should not have been incremented; we are in
			// second stage. Note that cx is reverted, but
			// dfx, ddfx are NOT. Now f is not indicative
			// of (cx, cy), but of (cx + 1, cy).
			if (f > 0) {
				--cx;
				second_stage = true;
			}
		}
		return true;
	}

	// in second stage, always decrement y.
	// f should be kept > 0, i.e., (cx+1, cy) should be kept out of ellipse

	--cy;
	f += dfy;
	dfy += ddfy;

	if (f <= 0) {
		++cx;
		f += dfx;
		dfx += ddfx;
	}

	return true;
}

// get next point
void graphics::line_ellipse::read()
{
	int new_oldx;

	Y = e.y();
	// repeat until we know there are no more points on this line
	do {
		new_oldx = X = e.x();
		if (!e.next())
			end = true;
	} while (!end && Y == e.y());

	// if X was moved straight down, we should increment it
	if (X == oldx)
		++X;
	oldx = new_oldx;
}

// Local Variables:
// mode: c++
// End:
