// main.cc - main()
//
// Copyright (C) 2000 - 2002 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <ctime>
#include <cstring>
#include <exception>
#include <fstream>
#include <iostream>
#include <string>
#include "SDL.h"
#include "SDL_image.h"

#include "../config.h"
#include "destruct.h"
#include "game.h"
#include "misc.h"
#include "record.h"
#include "sdl_audio.h"
#include "sdl_controller.h"
#include "sdl_graphics.h"
#include "sdl_timer.h"

namespace {
	class all {
	public:
		all(const rec::record& options,
		    const rec::record& ball_data,
		    SDL_Surface* s,
		    int frequency, int samples);
		~all();
		void start();
		void play();
	private:
		const rec::record& o;
		const rec::record& bd;

		graphics::tsurface* ts;
		double gamr, gamg, gamb, gam_step_factor, gammin, gammax;
		audio::taudio* ta;
		double excess_audio_time;
		control::controller* c;
		misc::random* r;
		tim::timer* t;

		Game::game* g;

		graphics::image* splash;
		graphics::image* paused;
		graphics::image* wait;

		all(const all&);
		all& operator=(const all&);
	};

	void print_help(std::ostream& os)
	{
		os << ("Usage: tunnel [OPTION]...\n\n"
		       "Run Tunnel, a game where you move a ball going down a tunnel,\n"
		       "avoiding other dangerous balls and picking up some pickups.\n\n"
		       "Options:\n\n"
		       "--width, -w WIDTH       width of screen or window\n"
		       "--height, -h HEIGHT     height of screen or window\n"
		       "--depth, -d DEPTH       depth in bits per pixel\n"
		       "--gamma, -g GAMMA       gamma\n"
		       "--fullscreen COND       use full screen if COND is true\n"
		       "--frequency FREQ        audio frequency in Hz\n"
		       "--file, -f FILE         load extra options from FILE\n\n"
		       "--help, -?              print this help and exit\n"
		       "--version, -v           print the version of Tunnel and exit\n\n"
		       "You can specify long option names with a single -; for\n"
		       "example, -help as well as --help.\n\n"
		       "For more information, look at\n"
		       DATA_DIR "/doc/manual.html\n"
		       "or, for the most up to date information, at\n"
		       "http://tspiteri.org/tunnel/\n\n");
	}

	void print_version(std::ostream& os)
	{
		os << ("Tunnel version "  VERSION "\n"
		       "Copyright (C) 2000-2004 Trevor Spiteri\n"
		       "You may redistribute copies of Tunnel\n"
		       "under the terms of the GNU General Public License.\n"
		       "For more information about these matters, see the file named COPYING.\n");
	}
}

int main(int argc, char* argv[])
{
	try {
		// load options and ball data
		rec::record options;
		{
			std::ifstream is(DATA_DIR "/options");
			is >> options;
		}
		rec::record& groptions = options.sub("graphics");
		rec::record& aoptions = options.sub("audio");

		// now read command line options to override some options
		for (int i = 1; i < argc; ++i) {
			using std::strcmp;

			if (i < argc-1) {
				if (!strcmp(argv[i], "--width")
				    || !strcmp(argv[i], "-width")
				    || !strcmp(argv[i], "-w")) {
					groptions["width"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--height")
				    || !strcmp(argv[i], "-height")
				    || !strcmp(argv[i], "-h")) {
					groptions["height"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--depth")
				    || !strcmp(argv[i], "-depth")
				    || !strcmp(argv[i], "-d")) {
					groptions["depth"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--gamma")
				    || !strcmp(argv[i], "-gamma")
				    || !strcmp(argv[i], "-g")) {
					groptions["gamma"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--anyformat")
				    || !strcmp(argv[i], "-anyformat")) {
					groptions["anyformat"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--fullscreen")
				    || !strcmp(argv[i], "-fullscreen")) {
					groptions["fullscreen"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--frequency")
				    || !strcmp(argv[i], "-frequency")) {
					aoptions["frequency"] = argv[++i];
					continue;
				}
				if (!strcmp(argv[i], "--file")
				    || !strcmp(argv[i], "-file")
				    || !strcmp(argv[i], "-f")) {
					std::ifstream is(argv[++i]);
					options.read_add(is);
					continue;
				}
			}
			if (!strcmp (argv[i], "--help")
			    || !strcmp(argv[i], "-help")
			    || !strcmp(argv[i], "-?")) {
				print_help(std::cout);
				return 0;
			}
			if (!strcmp(argv[i], "--version")
			    || !strcmp(argv[i], "-version")
			    || !strcmp(argv[i], "-v")) {
				print_version(std::cout);
				return 0;
			}
			print_help(std::cerr);
			return 1;
		}

		if (int(options.sub("graphics", "width")) == 0
		    || int(options.sub("graphics", "height")) == 0)
			throw "zero area";

		rec::record bd;
		{
			std::string filename(options.sub("game", "ball_data"));
			std::ifstream is(filename.c_str());
			is >> bd;
		}

		// initialise SDL
		if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_AUDIO) != 0)
			throw "unable to initialize SDL";

		BEGIN_DESTRUCT(sdlquit) {
			SDL_Quit();
		} END_DESTRUCT(sdlquit);

		// load icon, convert it to 32 bit, and set as icon
		std::string icon_filename = options.sub("graphics", "icon");
		SDL_Surface* icon = IMG_Load(icon_filename.c_str());
		if (icon == 0)
			throw std::string("unable to load image: " + icon_filename);
		SDL_Surface* icon32 = SDL_CreateRGBSurface(0, icon->w, icon->h, 32, 0x000000ff, 0x0000ff00, 0x00ff0000, 0xff000000);
		if (icon32 == 0)
			throw "unable to initialize icon";
		SDL_Surface* the_icon = SDL_ConvertSurface(icon, icon32->format, 0);
		if (the_icon == 0)
			throw "unable to initialize icon";
		if (SDL_SetColorKey(the_icon, SDL_SRCCOLORKEY, *static_cast<Uint32*>(the_icon->pixels)) != 0)
			throw "unable to initialize icon";
		SDL_WM_SetIcon(the_icon, 0);
		SDL_FreeSurface(the_icon);
		SDL_FreeSurface(icon32);
		SDL_FreeSurface(icon);

		SDL_WM_SetCaption("Tunnel", "tunnel");

		// initialize main surface
		const rec::record &gro = options.sub("graphics");
		SDL_Surface* s = SDL_SetVideoMode(gro["width"],
						  gro["height"],
						  gro["depth"],
						  (gro["anyformat"] ? SDL_ANYFORMAT : 0) | (gro["fullscreen"] ? SDL_FULLSCREEN : 0) | SDL_SWSURFACE);
		if (s == 0)
			throw "unable to initialize video";

		// initialize audio
		const rec::record &auo = options.sub("audio");

		all a(options, bd, s, auo["frequency"], auo["buffer_samples"]);

		a.start();
	}
	catch (std::exception& e) {
		std::cerr << "An error has occurred: " << e.what() << "\n";
		return 2;
	}
	catch (const std::string& s) {
		std::cerr << "Error: " << s << "\n";
		return 2;
	}
	catch (const char* c) {
		std::cerr << "Error: " << c << "\n";
		return 2;
	}
	catch (...) {
		std::cerr << "An error has occurred.\n";
		return 2;
	}
	return 0;
}

all::all(const rec::record& options,
         const rec::record& ball_data,
         SDL_Surface *s,
         int frequency, int samples)
	: o(options),
	  bd(ball_data),
	  ts(new sdl::tsurface(s)),
	  ta(new sdl::taudio(frequency, samples)),
	  excess_audio_time(0),
	  c(new sdl::controller),
	  r(new misc::random(std::time(0)))
{
	int w = options.sub("graphics", "width");
	int h = options.sub("graphics", "height");
	double gam = options.sub("graphics", "gamma");
	gamr = gam * double(options.sub("graphics", "red_gamma"));
	gamg = gam * double(options.sub("graphics", "green_gamma"));
	gamb = gam * double(options.sub("graphics", "blue_gamma"));
	gam_step_factor = options.sub("graphics", "gamma_step_factor");
	if (gam_step_factor < 1)
		gam_step_factor = 1;
	gammin = options.sub("graphics", "gamma_min");
	gammax = options.sub("graphics", "gamma_max");
	if (gamr > gammax) gamr = gammax;
	if (gamg > gammax) gamg = gammax;
	if (gamb > gammax) gamb = gammax;
	if (gamr < gammin) gamr = gammin;
	if (gamg < gammin) gamg = gammin;
	if (gamb < gammin) gamb = gammin;
	ts->set_gamma(gamr, gamb, gamb);
	graphics::raw_image *ri
		= ts->make_raw_image(options.sub("graphics", "splash"));
	splash = ts->make_image(ri, w, h);
	delete ri;

	ri = ts->make_raw_image(options.sub("graphics", "paused"));
	paused = ts->make_image(ri, w, h);
	delete ri;

	ri = ts->make_raw_image(options.sub("graphics", "wait"));
	wait = ts->make_image(ri, w, h);
	delete ri;

	rec::record key_bindings = options.sub("controller");
	c->add_up(key_bindings["up"]);
	c->add_down(key_bindings["down"]);
	c->add_left(key_bindings["left"]);
	c->add_right(key_bindings["right"]);
	c->add_accel(key_bindings["accel"]);
	c->add_decel(key_bindings["decel"]);
	c->add_fire(key_bindings["fire"]);
	c->add_weapon(key_bindings["weapon"]);
	c->add_start(key_bindings["start"]);
	c->add_quit(key_bindings["quit"]);
	c->add_pause(key_bindings["pause"]);

	SDL_PauseAudio(0);
}

all::~all()
{
	delete splash;
	delete r;
	delete c;
	delete ta;
	delete ts;
}

void all::start()
{
	for (bool main_ready = false; !main_ready; ) {
		splash->draw(0, 0);
		ts->flip();

		c->reset_click();
		for (bool wait_ready = false; !wait_ready; ) {
			c->wait();
			c->poll();
			if (c->start(c->click))
				wait_ready = true;
			if (c->quit(c->click))
				main_ready = wait_ready = true;
			if (c->left(c->click))
				ts->fullscreen(false);
			if (c->right(c->click))
				ts->fullscreen(true);
			if ((c->up(c->click) || c->accel(c->click))
			    &&(gamr * gam_step_factor <= gammax)
			    && (gamg * gam_step_factor <= gammax)
			    && (gamb * gam_step_factor <= gammax)) {

				gamr *= gam_step_factor;
				gamg *= gam_step_factor;
				gamb *= gam_step_factor;
				ts->set_gamma(gamr, gamg, gamb);
			}
			if ((c->down(c->click) || c->decel(c->click))
			    && (gamr / gam_step_factor >= gammin)
			    && (gamg / gam_step_factor >= gammin)
			    && (gamb / gam_step_factor >= gammin)) {

				gamr /= gam_step_factor;
				gamg /= gam_step_factor;
				gamb /= gam_step_factor;
				ts->set_gamma(gamr, gamg, gamb);
			}
		}

		if (!main_ready) {
			splash->draw(0, 0);
			wait->draw(0, 0);
			ts->flip();
			play();
		}
	}
}

void all::play()
{
	double timer_cycle = (double(o.sub("game", "cycle_msec"))
			      * double(o.sub("game", "timer_factor")));
	t = new sdl::timer(timer_cycle,
			   timer_cycle,
			   o.sub("game", "maxtime_nodraw"));
	g = new Game::game(o.sub("graphics", "width"),
			   o.sub("graphics", "height"),
			   o.sub("game"), ts, ta, &bd, c, *r);

	// start timer
	t->start();
	g->draw_first();

	bool first_turn = true;
	for (bool game_ready = false; !game_ready; ) {
		// poll controller
		c->poll();
		if (c->pause(c->press)) {
			first_turn = true;
			g->draw();
			paused->draw(0, 0);
			ts->flip();
			ts->grab_input(false);
			ts->show_cursor();
			//pause
			t->pause();

			// wait for continue or exit
			c->reset_click();
			for (bool pause_ready = false; !pause_ready; ) {
				c->wait();
				c->poll();
				if (c->start(c->click)) {
					pause_ready = true;
					g->draw_first();
				}
				if (c->quit(c->click))
					game_ready = pause_ready = true;
			}
		}
		if (!game_ready) {
			if (first_turn) {
				first_turn = false;
				ts->grab_input();
				ts->show_cursor(false);
				c->poll();
			}
			// if we have been paused, restart
			t->start();

			// get next action
			switch (t->next()) {
			case tim::timer::play:
				{
					double cycle_msec = o.sub("game", "cycle_msec");
					double timer_factor = o.sub("game", "timer_factor");
					g->play(cycle_msec);
					double this_audio_time = excess_audio_time + cycle_msec * timer_factor;
					ta->time_passed(int(this_audio_time));
					excess_audio_time = this_audio_time - int(this_audio_time);

					if (g->stat() == Game::game::done) {
						game_ready = true;
						ts->grab_input(false);
						ts->show_cursor();
					}
					break;
				}
			case tim::timer::draw:
				g->draw();
				ts->flip();
				break;
			default:
				throw "what is this?";
			}
		}
	}

	delete g;
	delete t;
}

// Local Variables:
// mode: c++
// End:
