// misc.h - contains various miscellaneous classes
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef MISC_H
#define MISC_H

#include <cmath>
#include <iterator>
#include <numeric>
#include <string>
#include <vector>

// First set the standards.
// pi is pi
// One time unit is 1 ms.
// The tunnel has diameter of 1000 units and depth of 4000 units.
// Thus, with a velocity of 1, it takes 1 sec to move one
// tunnel diameter.

const double pi = 3.14159265358979323846;

const int unit_msec   = 1;

const double tunnel_d    = 1000.0;
const double tunnel_dpth = 4000.0;
const double tunnel_r    = tunnel_d / 2;

namespace misc {

	// conversions:

	std::string s2lc(const std::string& s);
	bool s2b(const std::string& s);
	int s2i(const std::string& s);
	double s2d(const std::string& s);
	std::string b2s(bool b);
	std::string i2s(int i);
	std::string d2s(double d);

	class string_tokener {
	public:
		string_tokener(const std::string& s, char delim)
			: st(s), i(s.begin()), d(delim) { }
		std::string next();
	private:
		const std::string& st;
		std::string::const_iterator i;
		char d;
	};

	// squares a number
	template <typename T>
	inline T square(const T& t)
	{
		return t * t;
	}

	// a vectx is a single number in the x direction which can be added to
	// 2d and 3d vectors
	class vectx {
	public:
		vectx(double xi = 0)  : X(xi)	{ }

		vectx& set(double xi)		{ X = xi; return *this; }

		const double& x() const		{ return X; }
		double& x()			{ return X; }

	private:
		double X;
	};

	// a vecty is similar to vectx but in the y direction
	class vecty {
	public:
		vecty(double yi = 0)  : Y(yi)	{ }

		vecty& set(double yi)		{ Y = yi; return *this; }

		const double& y() const		{ return Y; }
		double& y()			{ return Y; }

	private:
		double Y;
	};

	// a vectz is similar to vectx but in the z direction
	class vectz {
	public:
		vectz(double zi = 0)  : Z(zi)	{ }

		vectz& set(double zi)		{ Z = zi; return *this; }

		const double& z() const		{ return Z; }
		double& z()			{ return Z; }

	private:
		double Z;
	};

	// class vect2d
	//    two dimensional vector with x and y directions
	// helper functions
	//    mkpolar: makes a vector from polar coordinates
	//    square:  gives the square
	//    abs:     gives the magnitude
	//    arg:     gives the argument
	//    and several operators

	class vect2d : public vectx, public vecty {
	public:
		vect2d(double xi = 0, double yi = 0)	: vectx(xi), vecty(yi) { }
		vect2d(vectx v)				: vectx(v) { }
		vect2d(vecty v)				: vecty(v) { }

		vect2d& set(double xi, double yi)
		{ x() = xi; y() = yi; return *this; }
		vect2d& set(const vect2d& v)
		{ x() = v.x(); y() = v.y(); return *this; }
		vect2d& setx(double xi)		{ x() = xi; return *this; }
		vect2d& sety(double yi)		{ y() = yi; return *this; }

		vect2d& operator=(const vect2d& v)
		{ x() = v.x(); y() = v.y(); return *this; }
		vect2d& operator+=(const vect2d& v)
		{ x() += v.x(); y() += v.y(); return *this; }
		vect2d& operator-=(const vect2d& v)
		{ x() -= v.x(); y() -= v.y(); return *this; }
		vect2d& operator+=(const vectx& v)
		{ x() += v.x(); return *this; }
		vect2d& operator+=(const vecty& v)
		{ y() += v.y(); return *this; }
		vect2d& operator-=(const vectx& v)
		{ x() -= v.x(); return *this; }
		vect2d& operator-=(const vecty& v)
		{ y() -= v.y(); return *this; }
		vect2d& operator*=(double d)
		{ x() *= d; y() *= d; return *this; }
		vect2d& operator/=(double d)
		{ x() /= d; y() /= d; return *this; }

		const vect2d& xy() const	{ return *this; }
		vect2d& xy()			{ return *this; }
	};

	inline vect2d mkpolar(double mag, double arg)
	{
		return vect2d(mag * std::cos(arg), mag * std::sin(arg));
	}

	inline double square(const vect2d& v)
	{
		return square(v.x()) + square(v.y());
	}

	inline double abs(const vect2d& v)
	{
		return std::sqrt(square(v));
	}

	inline double arg(const vect2d& v)
	{
		if (v.y() == 0 && v.x() == 0)
			return 0;
		else
			return std::atan2(v.y(), v.x());
	}

	inline vect2d dir(const vect2d& v)
	{
		double a = arg(v);
		return vect2d(std::cos(a), std::sin(a));
	}

	inline vect2d operator-(const vect2d& v)
	{
		return vect2d(-v.x(), -v.y());
	}

	inline vect2d operator+(const vect2d& v)
	{
		return v;
	}

	inline bool operator==(const vect2d& v1, const vect2d& v2)
	{
		return v1.x() == v2.x() && v1.y() == v2.y();
	}

	inline bool operator!=(const vect2d& v1, const vect2d& v2)
	{
		return v1.x() != v2.x() || v1.y() != v2.y();
	}

	inline vect2d operator+(const vect2d& v1, const vect2d& v2)
	{
		vect2d r = v1;
		r += v2;
		return r;
	}

	inline vect2d operator-(const vect2d& v1, const vect2d& v2)
	{
		vect2d r = v1;
		r -= v2;
		return r;
	}

	inline vect2d operator+(const vect2d& v1, const vectx& v2)
	{
		vect2d r = v1;
		r += v2;
		return r;
	}

	inline vect2d operator+(const vect2d& v1, const vecty& v2)
	{
		vect2d r = v1;
		r += v2;
		return r;
	}

	inline vect2d operator-(const vect2d& v1, const vectx& v2)
	{
		vect2d r = v1;
		r -= v2;
		return r;
	}

	inline vect2d operator-(const vect2d& v1, const vecty& v2)
	{
		vect2d r = v1;
		r -= v2;
		return r;
	}

	inline vect2d operator*(const vect2d& v, double d)
	{
		vect2d r = v;
		r *= d;
		return r;
	}

	inline vect2d operator*(double d, const vect2d& v)
	{
		vect2d r = v;
		r *= d;
		return r;
	}

	inline vect2d operator/(const vect2d& v, double d)
	{
		vect2d r = v;
		r /= d;
		return r;
	}

	//  class vect3d
	//     a three dimensional vector, with x, y and z
	//  helper functions
	//     mkpolar: sort of mkpolar for vect2d
	//     square:  gives the square
	//     abs:     gives the magnitude
	//     dotp:    gives the dot product of two vectors
	//     crossp:  gives the cross product of two vectors
	//     triplep: gives the triple product of three vectors
	//     and several operators

	class vect3d : public vect2d, public vectz {
	public:
		vect3d(double xi = 0, double yi = 0, double zi = 0)
			: vect2d(xi, yi), vectz(zi) { }
		vect3d(const vect2d& v, double zi = 0)
			: vect2d(v), vectz(zi) { }
		vect3d(const vect3d& v)
			: vect2d(v.xy()), vectz(v.z()) { }
		vect3d(vectx v)	: vect2d(v) { }
		vect3d(vecty v)	: vect2d(v) { }
		vect3d(vectz v)	: vectz(v) { }

		vect3d& set(double xi = 0, double yi = 0, double zi = 0)
		{ x() = xi; y() = yi; z() = zi; return *this; }
		vect3d& set(const vect2d& v, double zi = 0)
		{ xy() = v; z() = zi; return *this; }
		vect3d& set(const vect3d& v)
		{ xy() = v.xy(); z() = v.z(); return *this; }
		vect3d& setxy(const vect2d& v)	{ xy() = v; return *this; }
		vect3d& setx(double xi)		{ x() = xi; return *this; }
		vect3d& sety(double yi)		{ y() = yi; return *this; }
		vect3d& setz(double zi)		{ z() = zi; return *this; }

		vect3d& operator=(const vect3d& v)
		{ xy() = v.xy(); z() = v.z(); return *this; }
		vect3d& operator+=(const vect3d& v)
		{ xy() += v.xy(); z() += v.z(); return *this; }
		vect3d& operator-=(const vect3d& v)
		{ xy() -= v.xy(); z() -= v.z(); return *this; }
		vect3d& operator+=(const vect2d& v)
		{ xy() += v.xy(); return *this; }
		vect3d& operator-=(const vect2d& v)
		{ xy() -= v.xy(); return *this; }
		vect3d& operator+=(const vectx& v)
		{ x() += v.x(); return *this; }
		vect3d& operator-=(const vectx& v)
		{ x() -= v.x(); return *this; }
		vect3d& operator+=(const vecty& v)
		{ y() += v.y(); return *this; }
		vect3d& operator-=(const vecty& v)
		{ y() -= v.y(); return *this; }
		vect3d& operator+=(const vectz& v)
		{ z() += v.z(); return *this; }
		vect3d& operator-=(const vectz& v)
		{ z() -= v.z(); return *this; }
		vect3d& operator*=(double d)
		{ xy() *= d; z() *= d; return *this; }
		vect3d& operator/=(double d)
		{ xy() /= d; z() /= d; return *this; }
	};

	// mag is magnitude, argxy give angle on xy plane (from -pi to +pi)
	// and argz give other angle (from -pi/2 to +pi/2)
	inline vect3d mkpolar(double mag, double argxy, double argz)
	{
		return vect3d(mkpolar(mag * std::sin(argz), argxy), mag * std::cos(argz));
	}

	inline double square(const vect3d& v)
	{
		return square(v.x()) + square(v.y()) + square(v.z());
	}

	inline double abs(const vect3d& v)
	{
		return std::sqrt(square(v));
	}

	inline vect3d dir(const vect3d& v)
	{
		double a = abs(v);
		if (a)
			return v / a;
		else
			return 0;
	}

	inline vect3d operator-(const vect3d& v)
	{
		return vect3d(-v.x(), -v.y(), -v.z());
	}

	inline vect3d operator+(const vect3d& v)
	{
		return v;
	}

	inline bool operator==(const vect3d& v1, const vect3d& v2)
	{
		return v1.x() == v2.x() && v1.y() == v2.y() && v1.z() == v2.z();
	}

	inline bool operator!=(const vect3d& v1, const vect3d& v2)
	{
		return v1.x() != v2.x() || v1.y() != v2.y() || v1.z() != v2.z();
	}

	inline vect3d operator+(const vect3d& v1, const vect3d& v2)
	{
		vect3d r = v1;
		r += v2;
		return r;
	}

	inline vect3d operator-(const vect3d& v1, const vect3d& v2)
	{
		vect3d r = v1;
		r -= v2;
		return r;
	}

	inline vect3d operator+(const vect3d& v1, const vect2d& v2)
	{
		vect3d r = v1;
		r += v2;
		return r;
	}

	inline vect3d operator-(const vect3d& v1, const vect2d& v2)
	{
		vect3d r = v1;
		r -= v2;
		return r;
	}

	inline vect3d operator+(const vect3d& v1, const vectx& v2)
	{
		vect3d r = v1;
		r += v2;
		return r;
	}

	inline vect3d operator-(const vect3d& v1, const vectx& v2)
	{
		vect3d r = v1;
		r -= v2;
		return r;
	}

	inline vect3d operator+(const vect3d& v1, const vecty& v2)
	{
		vect3d r = v1;
		r += v2;
		return r;
	}

	inline vect3d operator-(const vect3d& v1, const vecty& v2)
	{
		vect3d r = v1;
		r -= v2;
		return r;
	}

	inline vect3d operator+(const vect3d& v1, const vectz& v2)
	{
		vect3d r = v1;
		r += v2;
		return r;
	}

	inline vect3d operator-(const vect3d& v1, const vectz& v2)
	{
		vect3d r = v1;
		r -= v2;
		return r;
	}

	inline vect3d operator*(const vect3d& v, double d)
	{
		vect3d r = v;
		r *= d;
		return r;
	}

	inline vect3d operator*(double d, const vect3d& v)
	{
		vect3d r = v;
		r *= d;
		return r;
	}

	inline vect3d operator/(const vect3d& v, double d)
	{
		vect3d r = v;
		r /= d;
		return r;
	}

	inline double dotp(const vect3d& v1, const vect3d& v2)
	{
		return v1.x() * v2.x() + v1.y() * v2.y() + v1.z() * v2.z();
	}

	inline vect3d crossp(const vect3d& v1, const vect3d& v2)
	{
		return vect3d(v1.y() * v2.z() - v1.z() * v2.y(),
			      v1.z() * v2.x() - v1.x() * v2.z(),
			      v1.x() * v2.y() - v1.y() * v2.x());
	}

	inline double triplep(const vect3d& v1, const vect3d& v2,
			      const vect3d& v3)
	{
		return dotp(v1, crossp(v2, v3));
	}

	///////////////////////////////////////////////////////
	// Random stuff

	class random {
		// uses rand_r(), which is weak as all the state of the random
		// number gnerator has to be kept in an unsigned long
	public:
		random(unsigned int s = 1) : seed(s) { }

		// gives a random integer in the range [0, RAND_MAX)
		int operator()();
		int max() const;

		// gives a random integer in the range [0, m)
		int operator()(int m)
		{ return int(double(m) * (*this)() / (max() + 1.0)); }

		// gives a random double in the range [0, m)
		double operator()(double m)
		{ return (double(m) * (*this)() / (max() + 1.0)); }
	private:
		unsigned int seed;
	};

	// number in the range [-max, max)
	inline int rndpm(random& r, int max)
	{
		return r(max + max) - max;
	}

	inline double rndpm(random& r, double max)
	{
		return r(max + max) - max;
	}

	// number in the range [min, max)
	inline int rndrng(random& r, int min, int max)
	{
		return r(max - min) + min;
	}

	inline double rndrng(random& r, double min, double max)
	{
		return r(max - min) + min;
	}

	// location that is equally probable on all pts in a circle of
	// radius rad, such that r is in the range [0, rad)
	inline vect2d rndpolar(random& rnd, double rad)
	{
		return mkpolar(rad * std::sqrt(rnd(1.0)), rnd(2.0 * pi));
	}

	inline vect3d rnd3d(random& rnd, double rad)
	{
		return mkpolar(rad * std::pow(rnd(1.0), 1.0/3), rnd(2.0 * pi), rnd(pi));
	}

} // namespace misc

#endif // !MISC_H

// Local Variables:
// mode: c++
// End:
