// sdl_audio.cc - functions for audio using SDL
//
// Copyright (C) 2001, 2002 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <cstring>
#include "SDL.h"

#include "audio.h"
#include "sdl_audio.h"

namespace sdl {

	struct wave {
		SDL_AudioSpec spec;
		Uint8* buf;
		Uint32 len;
	};

	class raw_sound : public audio::raw_sound {
	public:
		raw_sound(const std::string& filename);
		raw_sound(const raw_sound& rs);
		~raw_sound();

		raw_sound& operator=(const raw_sound& other);

		const wave* access() const { return w; }
	private:
		int* ref_count;
		wave* w;
	};

	class sound : public audio::sound {
	public:
		sound(const raw_sound* src, taudio* dst);
		~sound();

		void play(double volume) const;
	private:
		taudio* ta;
		wave* w;

		sound(const sound&);
		sound& operator=(const sound&);
	};

} // namespace sdl

sdl::raw_sound::raw_sound(const std::string& filename)
	: ref_count(new int(1)),
	  w(new wave)
{
	if (SDL_LoadWAV(filename.c_str(), &w->spec, &w->buf, &w->len) == 0) {
		throw std::string("could not load wave file: ") + filename;
	}
}

sdl::raw_sound::raw_sound(const raw_sound& rs)
	: ref_count(rs.ref_count), w(rs.w)
{
	++ *ref_count;
}

// decrease reference count and destroy if count becomes zero
sdl::raw_sound::~raw_sound()
{
	-- *ref_count;
	if (*ref_count == 0) {
		delete ref_count;
		SDL_FreeWAV(w->buf);
		delete w;
	}
}

sdl::raw_sound& sdl::raw_sound::operator=(const raw_sound& rs)
{
	// make sure we are not assigning a sound to itself
	if (&rs == this)
		return *this;

	// first decrement count and destroy if it becomes zero.
	-- *ref_count;
	if (*ref_count == 0) {
		delete ref_count;
		SDL_FreeWAV(w->buf);
		delete w;
	}

	// perform the copying and increment the reference count
	ref_count = rs.ref_count;
	w = rs.w;

	++ *ref_count;

	return *this;
}

sdl::sound::sound(const sdl::raw_sound* src, sdl::taudio* dst)
	: ta(dst)
{
	if (ta->audio_available) {
		w = new wave;
		SDL_AudioCVT cvt;
		const SDL_AudioSpec* src_s = &src->access()->spec;
		const SDL_AudioSpec* dst_s = ta->get_spec();
		if (SDL_BuildAudioCVT(&cvt, src_s->format,
				      src_s->channels,
				      src_s->freq,
				      dst_s->format,
				      dst_s->channels,
				      dst_s->freq)
		    == -1) {
			
			throw "unable to convert audio format";
		}
		cvt.buf = static_cast<Uint8*>(malloc(src->access()->len * cvt.len_mult));
		cvt.len = src->access()->len;
		memcpy(cvt.buf, src->access()->buf, src->access()->len);
		if (SDL_ConvertAudio(&cvt) != 0)
			throw "unable to convert audio format";
		
		w->spec = *dst_s;;
		w->buf = cvt.buf;
		w->len = cvt.len_cvt;
	}
}

sdl::sound::~sound()
{
	if (ta->audio_available) {
		SDL_FreeWAV(w->buf);
		delete w;
	}
}

namespace sdl {
	class active_sound {
	public:
		active_sound(Uint8* buffer, int len, int volume)
			: buf(buffer), rem(len), pos(0), vol(volume) { }

		Uint8* buf;
		int rem, pos;
		int vol;
	};
}

void sdl::sound::play(double volume) const
{
	if (!ta->audio_available)
		return;
	active_sound* s = new active_sound(w->buf, w->len, int(volume * SDL_MIX_MAXVOLUME));
	ta->take_sound(s);
}

extern "C" {
	static void callback(void* userdata, Uint8* stream, int len)
	{
		sdl::taudio* ta = static_cast<sdl::taudio*>(userdata);
		ta->get(stream, len);
	}
}

sdl::taudio::taudio(int frequency, int samples)
	: msamples_left(0)
{
	spec.freq = frequency;
	spec.format = AUDIO_S16SYS;
	spec.channels = 1;
	spec.samples = samples;
	spec.callback = callback;
	spec.userdata = this;

	if (SDL_OpenAudio(&spec, 0) != 0)
		audio_available = false;
	else {
		audio_available = true;

		buf_len = spec.size * 2;
		buf = new Uint8[buf_len];
		for (int i = 0; i < buf_len; ++i)
			buf[i] = 0;
		buf_start = buf_used = 0;
	}
}

sdl::taudio::~taudio()
{
	if (audio_available) {
		delete[] buf;
		SDL_CloseAudio();
	}
}

audio::raw_sound* sdl::taudio::make_raw_sound(const std::string& filename)
{
	return new raw_sound(filename);
}

audio::sound* sdl::taudio::make_sound(const audio::raw_sound* src)
{
	return new sound(static_cast<const raw_sound*>(src), this);
}

void sdl::taudio::get(Uint8* stream, int len)
{
	if (!audio_available)
		return;
	if (buf_used < len)
		return;

	int c1;
	if (buf_start + buf_used > buf_len)
		c1 = buf_len - buf_start;
	else
		c1 = buf_used;

	if (c1 >= len) {
		SDL_MixAudio(stream, buf + buf_start, len, SDL_MIX_MAXVOLUME);
		buf_start += len;
		buf_used -= len;
		return;
	}
	SDL_MixAudio(stream, buf + buf_start, c1, SDL_MIX_MAXVOLUME);
	SDL_MixAudio(stream + c1, buf, len - c1, SDL_MIX_MAXVOLUME);
	buf_start = len - c1;
	buf_used -= len;
}

void sdl::taudio::time_passed(int msec)
{
	if (!audio_available)
		return;
	
	int msamples = spec.freq * msec + msamples_left;
	int samples = msamples / 1000;
	msamples_left = msamples % 1000;
	int len = samples * spec.size / spec.samples;

	SDL_LockAudio();

	int buf_write = (buf_start + buf_used) % buf_len;
	int c1;
	if (buf_write + len > buf_len)
		c1 = buf_len - buf_write;
	else
		c1 = len;

	if (c1 >= len) {
		mix_sounds(buf_write, len);
	}
	else {
		mix_sounds(buf_write, c1);
		mix_sounds(0, len - c1);
	}
	buf_used += len;
	if (buf_used > buf_len) {
		buf_start += buf_used - buf_len;
		buf_start %= buf_len;
		buf_used = buf_len;
	}

	SDL_UnlockAudio();
}

void sdl::taudio::mix_sounds(int offs, int len)
{
	if (!audio_available)
		return;
	
	Uint8* dst = buf + offs;
	memset(dst, spec.silence, len);
	for (std::list<active_sound*>::iterator i = sounds.begin();
	     i != sounds.end(); ) {

		Uint8* srcp = (*i)->buf + (*i)->pos;

		if ((*i)->rem > len) {
			SDL_MixAudio(dst, srcp, len, (*i)->vol);
			(*i)->pos += len;
			(*i)->rem -= len;
			++i;
		} else {
			SDL_MixAudio(dst, srcp, (*i)->rem, (*i)->vol);
			delete *i;
			i = sounds.erase(i);
		}
	}
}

void sdl::taudio::take_sound(active_sound* s)
{
	if (audio_available)
		sounds.push_back(s);
}

// Local Variables:
// mode: c++
// End:
