/* callbacks.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <glade/glade.h>

#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "gui.h"

void on_gui_menuitem_file_new_activate(GtkMenuItem *menuitem,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  editor_song_open(gui->editor, NULL);
}

void on_gui_menuitem_file_open_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_open");

  if(!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);
  else
    gtk_widget_hide(fs);
}

void on_gui_menuitem_file_save_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_save(gui->editor->song, gui->editor->filename);
}

void on_gui_menuitem_file_saveas_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_saveas");

  if(!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);
  else
    gtk_widget_hide(fs);
}

void on_gui_menuitem_file_print_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  /* FIX: Implement */
  fprintf(stderr, "Printing not implemented yet\n"); 
}

void on_gui_menuitem_file_exit_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  gtk_main_quit();
}

void on_gui_menuitem_edit_cut_activate(GtkMenuItem *menuitem,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  if(gui->editor->copyarea)
    block_free(gui->editor->copyarea);
  gui->editor->copyarea=block_copy(gui->editor->song->blocks[gui->editor->block],
			     tracker->sel_start_ch,
			     tracker->sel_start_row,
			     tracker->sel_end_ch,
			     tracker->sel_end_row);
  block_clear(gui->editor->song->blocks[gui->editor->block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_menuitem_edit_copy_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  if(gui->editor->copyarea)
    block_free(gui->editor->copyarea);
  gui->editor->copyarea=block_copy(gui->editor->song->blocks[gui->editor->block],
			     tracker->sel_start_ch, tracker->sel_start_row,
			     tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_menuitem_edit_paste_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copyarea,
	      tracker->cursor_ch, tracker->patpos);
  tracker_redraw(tracker);
}

void on_gui_menuitem_edit_clear_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_gui_menuitem_edit_transpose_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")))==GUI_TRANSPOSE_SELECTION) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")), GUI_TRANSPOSE_SELECTION);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_gui_scale_tempo_value_changed(GtkRange *range,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_set_tempo(gui->editor->song,
		 (unsigned char)gtk_range_get_value(range));
  gui_info_refresh(gui);
}

void on_gui_scale_tpl_value_changed(GtkRange *range,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_set_tpl(gui->editor->song, (unsigned char)gtk_range_get_value(range));
  gui_info_refresh(gui);
}

void on_gui_menuitem_song_trackvolumes_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_trackvolumes");

  if(!GTK_WIDGET_VISIBLE(window)) {
    int oldmax=gui->editor->song->maxtracks;

    if(song_check_maxtracks(gui->editor->song)) {
      player_trackstatus_create(gui->editor->player, oldmax);
      gui_trackvolumes_refresh(gui);
    }
    gtk_widget_show(window);
  } else
    gtk_widget_hide(window);
}

void on_gui_menuitem_song_transpose_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")))==GUI_TRANSPOSE_SONG) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")), GUI_TRANSPOSE_SONG);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_gui_menuitem_song_properties_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml,
					 "gui_dialog_songproperties");

  if(!GTK_WIDGET_VISIBLE(window)) {
    gui_song_refresh(gui);
    gtk_widget_show(window);
  } else
    gtk_widget_hide(window);
}

void on_gui_menuitem_block_cut_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  if(gui->editor->copyblock)
    block_free(gui->editor->copyblock);
  gui->editor->copyblock=block_copy(gui->editor->song->blocks[gui->editor->block],
			      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  block_clear(gui->editor->song->blocks[gui->editor->block],
	      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_copy_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->copyblock)
    block_free(gui->editor->copyblock);
  gui->editor->copyblock=block_copy(gui->editor->song->blocks[gui->editor->block],
			      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
}

void on_gui_menuitem_block_paste_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copyblock,
	      0, 0);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_selectall_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  tracker->sel_start_ch=0;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=gui->editor->song->blocks[gui->editor->block]->tracks-1;
  tracker->sel_end_row=gui->editor->song->blocks[gui->editor->block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_clear_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block],
	      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_block_transpose_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")))==GUI_TRANSPOSE_BLOCK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")), GUI_TRANSPOSE_BLOCK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_gui_menuitem_block_list_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  on_gui_button_block_clicked(NULL, user_data);
}

void on_gui_menuitem_track_cut_activate(GtkMenuItem *menuitem,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  if(gui->editor->copytrack)
    block_free(gui->editor->copytrack);
  gui->editor->copytrack=block_copy(gui->editor->song->blocks[gui->editor->block],
			      tracker->cursor_ch, 0,
			      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  block_clear(gui->editor->song->blocks[gui->editor->block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_copy_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  if(gui->editor->copytrack)
    block_free(gui->editor->copytrack);
  gui->editor->copytrack=block_copy(gui->editor->song->blocks[gui->editor->block],
			      tracker->cursor_ch, 0,
			      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
}

void on_gui_menuitem_track_paste_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copytrack,
	      tracker->cursor_ch, 0);
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_selectall_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  tracker->sel_start_ch=tracker->cursor_ch;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=tracker->cursor_ch;
  tracker->sel_end_row=gui->editor->song->blocks[gui->editor->block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_clear_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_gui_menuitem_track_transpose_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")))==GUI_TRANSPOSE_TRACK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")), GUI_TRANSPOSE_TRACK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_gui_menuitem_song_sysexlist_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_sysex");

  if(!GTK_WIDGET_VISIBLE(window)) {
    gui_sysexlist_refresh(gui);
    gtk_widget_show(window);
  } else
    gtk_widget_hide(window);
}

void on_gui_menuitem_midi_killallnotes_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  player_stop_all_notes(gui->editor->player);
}

void on_gui_menuitem_midi_resetpitch_activate(GtkMenuItem *menuitem,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  player_reset_pitch(gui->editor->player);
}

void on_gui_menuitem_settings_preferences_activate(GtkMenuItem *menuitem,
						   gpointer user_data) {
  /* FIX: implement */
  fprintf(stderr, "Preferences not implemented yet\n");
}

void on_gui_menuitem_about_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gchar *authors[] = { "Vesa Halttunen", NULL };

  if(gui->window_about==NULL || !(GTK_IS_WINDOW(gui->window_about))) {
    gui->window_about=gnome_about_new("Tutka", VERSION,
				     "(C) 2002-2003 Vesa Halttunen",
				     "http://www.freesoftware.fsf.org/tutka/",
				     (const gchar **)authors,
				     NULL,
				     NULL,
				     NULL);
    gtk_widget_show(gui->window_about);
  } else {
    gtk_widget_destroy(gui->window_about);
    gui->window_about=NULL;
  }
}


GtkWidget* create_gui_tracker(gchar *widget_name, gchar *string1,
			      gchar *string2, gint int1, gint int2) {
  return tracker_new();
}

void on_gui_button_playsong_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  player_start(gui->editor->player, MODE_PLAY_SONG,
	       gui->editor->section, gui->editor->position,
	       gui->editor->block, 0);

  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_patpos(tracker, 0);

  gui_info_refresh(gui);
  gui_timer_refresh(gui, 0);
  gui_blocklist_refresh(gui, FALSE);
  gui_playseq_refresh(gui, FALSE);
  gui_sectionlist_refresh(gui, FALSE);
}

void on_gui_button_playblock_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  player_start(gui->editor->player, MODE_PLAY_BLOCK,
	       gui->editor->section, gui->editor->position,
	       gui->editor->block, 0);
  tracker_set_patpos(tracker, 0);

  gui_info_refresh(gui);
  gui_timer_refresh(gui, 0);
}

void on_gui_button_contsong_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->player->mode!=MODE_PLAY_SONG) {
    player_start(gui->editor->player, MODE_PLAY_SONG,
		 gui->editor->section, gui->editor->position,
		 gui->editor->block, 1);
    gui_info_refresh(gui);
  }
}

void on_gui_button_contblock_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->player->mode!=MODE_PLAY_BLOCK) {
    player_start(gui->editor->player, MODE_PLAY_BLOCK,
		 gui->editor->section, gui->editor->position,
		 gui->editor->block, 1);
    gui_info_refresh(gui);
  }
}

void on_gui_button_stop_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  player_stop(gui->editor->player);
  gui_info_refresh(gui);
}

void on_gui_scale_volume_value_changed(GtkRange *range,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->defaultvelocity=(unsigned char)gtk_range_get_value(range);
}

void on_gui_scale_transpose_value_changed(GtkRange *range,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->transpose=(unsigned char)gtk_range_get_value(range);
}

void on_gui_scale_hold_value_changed(GtkRange *range,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->hold=(unsigned char)gtk_range_get_value(range);
}

void on_gui_scale_midich_value_changed(GtkRange *range,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->midichannel=(unsigned char)gtk_range_get_value(range)-1;
}

void on_gui_button_properties_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_instrumentproperties");
  
  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_gui_checkbutton_edit_toggled(GtkToggleButton *togglebutton,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->edit=1;
  else
    gui->editor->edit=0;
}

void on_gui_checkbutton_chord_toggled(GtkToggleButton *togglebutton,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->chord=1;
  else
    gui->editor->chord=0;
}

void on_gui_spinbutton_space_changed(GtkSpinButton *spinbutton,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->space=gtk_spin_button_get_value(spinbutton);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "gui_button_properties"));
}

void on_gui_button_section_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_sectionlist");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_gui_button_position_clicked(GtkButton *button,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_playseq");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_gui_button_block_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_blocklist");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_gui_button_trackvolumes_close_clicked(GtkButton *button,
					      gpointer user_data) {
  on_gui_menuitem_song_trackvolumes_activate(NULL, user_data);
}

gboolean on_gui_label_timer_button_press_event(GtkWidget *widget,
					       GdkEventButton *event,
					       gpointer user_data) {
  /* FIX: implement this */

  return FALSE;
}

void on_gui_selection_sysexlist_changed(GtkTreeSelection *selection,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char sysex[6+8];
  GtkTreeIter iter;
  GtkTreeModel *model;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter))
    gtk_tree_model_get(model, &iter, 0, &gui->sysexlist_sysex, -1);

  /* Show the number of the selected SysEx message */
  snprintf(sysex, 6+8, "SysEx %d/%d", gui->sysexlist_sysex+1,
	   gui->editor->song->numsysexes);
  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "gui_frame_sysex_controls")), sysex);

  /* Show name of the selected SysEx message */
  if(gui->editor->song->sysexes[gui->sysexlist_sysex]->name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml,
						      "gui_entry_sysex_name")),
		       gui->editor->song->sysexes[gui->sysexlist_sysex]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml,
						      "gui_entry_sysex_name")),
		       "");

  /* Show length of the selected SysEx message */
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "gui_spinbutton_sysex_length")), gui->editor->song->sysexes[gui->sysexlist_sysex]->length);
  if(gui->editor->song->sysexes[gui->sysexlist_sysex]->autosend)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "gui_checkbutton_sysex_autosend")), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "gui_checkbutton_sysex_autosend")), FALSE);

  /* Make widgets sensitive */
  gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_table_sysex1")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_sysex_send")), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_sysex_receive")), TRUE);
  /* gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_checkbutton_sysex_autostop")), TRUE); */
  gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_sysex_delete")), TRUE);
}

void on_gui_button_sysex_appendnew_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_sysex(gui->editor->song, gui->editor->song->numsysexes);
  gui_sysexlist_refresh(gui);
}

void on_gui_button_sysex_delete_clicked(GtkButton *button,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    song_delete_sysex(gui->editor->song, gui->sysexlist_sysex);
    gui_sysexlist_refresh(gui);
  }
}

void on_gui_button_sysex_insertnew_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_sysex(gui->editor->song, gui->sysexlist_sysex);
  if(gui->sysexlist_sysex==-1)
    gui->sysexlist_sysex=0;
  gui_sysexlist_refresh(gui);
}

void on_gui_button_sysex_send_clicked(GtkButton *button,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    midi_system_exclusive(gui->editor->player->midi,
			  gui->editor->song->sysexes[gui->sysexlist_sysex]->data,
			  gui->editor->song->sysexes[gui->sysexlist_sysex]->length);
  }
}

void on_gui_button_sysex_receive_clicked(GtkButton *button,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    midi_read_system_exclusive(gui->editor->player->midi,
			       gui->editor->song->sysexes[gui->sysexlist_sysex],
			       gui->sysexlist_autostop);
  }
}

void on_gui_checkbutton_sysex_autostop_toggled(GtkToggleButton *togglebutton,
					       gpointer user_data) {
  
}

void on_gui_selection_sectionlist_changed(GtkTreeSelection *selection,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  char section[8+8], playseq[17+8];

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &gui->editor->section, -1);
    gui->editor->section--;
  }

  editor_refresh_playseq_and_block(gui->editor);

  snprintf(section, 8+8, "Section %d/%d", gui->editor->section+1,
	   gui->editor->song->numsections);
  snprintf(playseq, 17+8, "Playing Sequence %d/%d", gui->editor->playseq+1,
	   gui->editor->song->numplayseqs);

  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "gui_frame_sectionlist")), section);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "gui_label_sectionlist_playseq")), playseq);

  gui_sectionlist_buttons_refresh(gui);
  gui_playseq_refresh(gui, FALSE);
  gui_info_refresh(gui);
}

void on_gui_button_sectionlist_next_clicked(GtkButton *button,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  if(gui->editor->song->sections[gui->editor->section]<gui->editor->song->numplayseqs-1)
    gui->editor->song->sections[gui->editor->section]++;

  editor_refresh_playseq_and_block(gui->editor);

  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_sectionlist_delete_clicked(GtkButton *button,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_delete_section(gui->editor->song, gui->editor->section);
  if(gui->editor->section>=gui->editor->song->numsections)
    gui->editor->section=gui->editor->song->numsections-1;

  editor_refresh_playseq_and_block(gui->editor);

  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_sectionlist_append_clicked(GtkButton *button,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_insert_section(gui->editor->song, gui->editor->song->numsections);

  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_sectionlist_prev_clicked(GtkButton *button,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->song->sections[gui->editor->section]>0)
    gui->editor->song->sections[gui->editor->section]--;

  editor_refresh_playseq_and_block(gui->editor);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_sectionlist_insert_clicked(GtkButton *button,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_insert_section(gui->editor->song, gui->editor->section);

  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_selection_playseq_changed(GtkTreeSelection *selection,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &gui->editor->position, -1);
    gui->editor->position--;
  }

  /* Refresh the playing sequence window widgets and the rest of the GUI */
  gui_playseq_info_refresh(gui);
  gui_info_refresh(gui);
}

void on_gui_button_playseq_next_clicked(GtkButton *button,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  unsigned int playseq=gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position];

  if(playseq<gui->editor->song->numblocks-1)
    playseq_set(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position,
		playseq+1);

  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_playseq_delete_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_delete(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position);
  editor_refresh_playseq_and_block(gui->editor);
  
  if(gui->editor->position>=gui->editor->song->playseqs[gui->editor->playseq]->length)
    gui->editor->position=gui->editor->song->playseqs[gui->editor->playseq]->length-1;
  
  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_playseq_append_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_insert(gui->editor->song->playseqs[gui->editor->playseq],
		 gui->editor->song->playseqs[gui->editor->playseq]->length);
  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_playseq_prev_clicked(GtkButton *button,
                                        gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  unsigned int playseq=gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position];

  if(playseq>0)
    playseq_set(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position,
		playseq-1);

  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_playseq_insert_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_insert(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position);
  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_playseq_insertnew_clicked(GtkButton *button,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_playseq(gui->editor->song, gui->editor->playseq);
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_playseq_appendnew_clicked(GtkButton *button,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_playseq(gui->editor->song, gui->editor->song->numplayseqs);
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_button_playseq_deletethis_clicked(GtkButton *button,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_delete_playseq(gui->editor->song, gui->editor->playseq);
  
  editor_refresh_playseq_and_block(gui->editor);
  
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_gui_selection_blocklist_changed(GtkTreeSelection *selection,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");
  GtkTreeIter iter;
  GtkTreeModel *model;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &gui->editor->block, -1);
    gui->editor->block--;
  }
  
  /* Refresh the block list window widgets and the rest of the GUI */
  gui_blocklist_info_refresh(gui);
  gui_info_refresh(gui);
  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
}

void on_gui_spinbutton_blocklist_tracks_changed(GtkEditable *editable,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  int oldmax=gui->editor->song->maxtracks;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");
  
  block_set_tracks(gui->editor->song->blocks[gui->editor->block],
		   (unsigned int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(editable)));
  tracker_set_num_channels(tracker, (int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(editable)));
  tracker_redraw(tracker);
  
  if(song_check_maxtracks(gui->editor->song)) {
    player_trackstatus_create(gui->editor->player, oldmax);
    gui_trackvolumes_refresh(gui);
  }
}

void on_gui_spinbutton_blocklist_length_changed(GtkEditable *editable,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");
  
  block_set_length(gui->editor->song->blocks[gui->editor->block],
		   (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
  tracker->curpattern=gui->editor->song->blocks[gui->editor->block];
  tracker_redraw(tracker);
}

void on_gui_spinbutton_blocklist_commandpages_changed(GtkEditable *editable,
						      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  block_set_commandpages(gui->editor->song->blocks[gui->editor->block],
			 (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
  if(gui->editor->cmdpage>=gui->editor->song->blocks[gui->editor->block]->commandpages) {
    while(gui->editor->cmdpage>=gui->editor->song->blocks[gui->editor->block]->commandpages)
      gui->editor->cmdpage--;
    gui_tracker_refresh(gui);
  }
  gui_info_refresh(gui);
}

void on_gui_button_blocklist_insertnew_clicked(GtkButton *button,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_block(gui->editor->song,
		    gui->editor->block, gui->editor->block);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_blocklist_appendnew_clicked(GtkButton *button,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_block(gui->editor->song, gui->editor->song->numblocks,
		    gui->editor->block);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
}

void on_gui_button_blocklist_delete_clicked(GtkButton *button,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_delete_block(gui->editor->song, gui->editor->block);
  
  if(gui->editor->block>=gui->editor->song->numblocks)
    gui->editor->block=gui->editor->song->numblocks-1;
  
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
}

gboolean on_gui_entry_blocklist_focus_out_event(GtkWidget *widget,
						GdkEventFocus *event,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));
  
  if(gui->editor->song->blocks[gui->editor->block]->name)
    free(gui->editor->song->blocks[gui->editor->block]->name);
  
  gui->editor->song->blocks[gui->editor->block]->name=(char *)strdup(name);
  
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);

  return FALSE;
}

gboolean on_gui_entry_sysex_name_focus_out_event(GtkWidget *widget,
						 GdkEventFocus *event,
						 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));
    
    if(gui->editor->song->sysexes[gui->sysexlist_sysex]->name)
      free(gui->editor->song->sysexes[gui->sysexlist_sysex]->name);
    
    gui->editor->song->sysexes[gui->sysexlist_sysex]->name=(char *)strdup(name);
    gui_sysexlist_refresh(gui);
  }

  return FALSE;
}

gboolean on_gui_entry_playseq_name_focus_out_event(GtkWidget *widget,
						   GdkEventFocus *event,
						   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->playseqs[gui->editor->playseq]->name)
    free(gui->editor->song->playseqs[gui->editor->playseq]->name);
  
  gui->editor->song->playseqs[gui->editor->playseq]->name=(char *)strdup(name);
  
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);

  return FALSE;
}

gboolean on_gui_entry_instrumentproperties_name_focus_out_event(GtkWidget *widget, GdkEventFocus *event, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->instruments[gui->editor->instrument]->name!=NULL)
    free(gui->editor->song->instruments[gui->editor->instrument]->name);

  gui->editor->song->instruments[gui->editor->instrument]->name=strdup(name);

  gui_instrument_refresh(gui, FALSE);

  return FALSE;
}

void on_gui_button_sysex_close_clicked(GtkButton *button, gpointer user_data) {
  on_gui_menuitem_song_sysexlist_activate(NULL, user_data);
}

void on_gui_button_saveas_ok_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_saveas");

  /* Store the filename */
  if(gui->editor->filename!=NULL)
    free(gui->editor->filename);
  gui->editor->filename=strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
  
  /* Save... */
  song_save(gui->editor->song, gui->editor->filename);
  
  /* ...and close the file selector */
  gtk_widget_hide(fs);
}

void on_gui_button_open_ok_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_open");

  editor_song_open(gui->editor, (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

  gtk_widget_hide(fs);
}

void on_gui_button_open_cancel_clicked(GtkButton *button,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_open");

  gtk_widget_hide(fs);
}

void on_gui_button_saveas_cancel_clicked(GtkButton *button,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "gui_fileselection_saveas");

  gtk_widget_hide(fs);
}

void on_gui_spinbutton_sysex_length_changed(GtkEditable *editable,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1)
    sysex_set_length(gui->editor->song->sysexes[gui->sysexlist_sysex],
		 (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
}

gboolean on_gui_entry_songname_focus_out_event(GtkWidget *widget,
					       GdkEventFocus *event,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->name!=NULL)
    free(gui->editor->song->name);

  gui->editor->song->name=strdup(name);

  gui_info_refresh(gui);

  return FALSE;
}

void on_gui_button_songproperties_close_clicked(GtkButton *button,
						gpointer user_data) {
  on_gui_menuitem_song_properties_activate(NULL, user_data);
}

void on_gui_spinbutton_instrument_changed(GtkEditable *editable,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  /* Select an instrument if an instrument selection key was pressed */
  gui->editor->instrument=(unsigned char)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value-1;

  /* Make sure the instrument exists */
  song_check_instrument(gui->editor->song, gui->editor->instrument);
  gui_instrument_refresh(gui, FALSE);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "gui_button_properties"));
}

void on_gui_adjustment_trackvolumes_changed(GtkAdjustment *adjustment,
					    gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  *data=((*data)&128)|((unsigned char)adjustment->value);
}

void on_gui_togglebutton_trackvolumes_toggled(GtkToggleButton *togglebutton,
					      gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    *data=((*data)&127)|128;
  else
    *data=(*data)&127;
}

void on_gui_optionmenu_keyboard_changed(GtkOptionMenu *optionmenu,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  gui->editor->octave=gtk_option_menu_get_history(optionmenu);
}

void on_gui_button_transpose_close_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "gui_dialog_transpose");

  gtk_widget_hide(window);
}

void on_gui_button_transpose_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_area")));
  int instruments=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_instruments")));
  int mode=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_transpose_mode")));

  int instrument=-1, halfnotes=0;

  switch(mode) {
  case 0:
    halfnotes=12;
    break;
  case 1:
    halfnotes=1;
    break;
  case 2:
    halfnotes=-1;
    break;
  case 3:
    halfnotes=-12;
    break;
  default:
    break;
  }

  if(instruments==1)
    instrument=gui->editor->instrument;

  switch(area) {
  case GUI_TRANSPOSE_SONG:
    song_transpose(gui->editor->song, instrument, halfnotes);
    break;
  case GUI_TRANSPOSE_BLOCK:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes, 0, 0,
		    gui->editor->song->blocks[gui->editor->block]->tracks-1,
		    gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_TRANSPOSE_TRACK:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes,
		    tracker->cursor_ch, 0, tracker->cursor_ch,
		    gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_TRANSPOSE_SELECTION:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes,
		    tracker->sel_start_ch, tracker->sel_start_row,
		    tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_gui_checkbutton_sendsync_toggled(GtkToggleButton *togglebutton,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->song->sendsync=1;
  else
    gui->editor->song->sendsync=0;
}

void on_gui_checkbutton_sysex_autosend_toggled(GtkToggleButton *togglebutton,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    if(gtk_toggle_button_get_active(togglebutton))
      sysex_set_autosend(gui->editor->song->sysexes[gui->sysexlist_sysex], 1);
    else
      sysex_set_autosend(gui->editor->song->sysexes[gui->sysexlist_sysex], 0);
  }
}
