/* midi.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include "midi.h"

/* Opens a MIDI device */
FILE *midi_open(char *filename) {
  FILE *midi;

  /* Open MIDI device */
  midi=fopen(filename, "r+");
  if(midi==NULL)
    fprintf(stderr, "Could not open MIDI device %s.\n", filename);

  return midi;
}

/* Closes a MIDI device */
void midi_close(FILE *midi) {
  if(midi==NULL) {
    fprintf(stderr, "midi_close() called with null file\n");
    return;
  }

  fclose(midi);
}

/* Stops a note playing on a MIDI channel using requested velocity */
void midi_note_off(FILE *midi, unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_note_off() called with null file\n");
    return;
  }

  data[0]=0x80|channel;
  data[1]=note&0x7f;
  data[2]=velocity&0x7f;
  
  fwrite(data, 3, 1, midi);
  fflush(midi);
}

/* Plays a note on a MIDI channel using requested velocity */
void midi_note_on(FILE *midi, unsigned char channel, unsigned char note,
		   unsigned char velocity) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_note_on() called with null file\n");
    return;
  }

  data[0]=0x90|channel;
  data[1]=note&0x7f;
  data[2]=velocity&0x7f;
  
  fwrite(data, 3, 1, midi);
  fflush(midi);
}

/* Sets the aftertouch pressure of a note playing on a MIDI channel */
void midi_aftertouch(FILE *midi, unsigned char channel, unsigned char note,
		     unsigned char pressure) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_aftertouch() called with null file\n");
    return;
  }

  data[0]=0xa0|channel;
  data[1]=note&0x7f;
  data[2]=pressure&0x7f;
  
  fwrite(data, 3, 1, midi);
  fflush(midi);
}

/* Sets the MIDI controller value of a MIDI channel */
void midi_controller(FILE *midi, unsigned char channel, 
		     unsigned char controller, unsigned char value) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_controller() called with null file\n");
    return;
  }

  data[0]=0xb0|channel;
  data[1]=controller&0x7f;
  data[2]=value&0x7f;
  
  fwrite(data, 3, 1, midi);
  fflush(midi);
}

/* Send a program change on a MIDI channel */
void midi_program_change(FILE *midi, unsigned char channel,
			 unsigned char program) {
  unsigned char data[2];

  if(midi==NULL) {
    fprintf(stderr, "midi_program_change() called with null file\n");
    return;
  }

  data[0]=0xc0|channel;
  data[1]=program&0x7f;
  
  fwrite(data, 2, 1, midi);
  fflush(midi);
}

/* Sets the channel pressure of a MIDI channel */
void midi_channel_pressure(FILE *midi, unsigned char channel,
			   unsigned char pressure) {
  unsigned char data[2];

  if(midi==NULL) {
    fprintf(stderr, "midi_channel_pressure() called with null file\n");
    return;
  }

  data[0]=0xd0|channel;
  data[1]=pressure&0x7f;
  
  fwrite(data, 2, 1, midi);
  fflush(midi);
}

/* Sets the pitch wheel value of a MIDI channel */
void midi_pitch_wheel(FILE *midi, unsigned char channel,
		      unsigned short value) {
  unsigned char data[3];

  if(midi==NULL) {
    fprintf(stderr, "midi_pitch_wheel() called with null file\n");
    return;
  }

  data[0]=0xe0|channel;
  data[1]=(value>>7)&0x7f;
  data[2]=value&0x7f;
  
  fwrite(data, 3, 1, midi);
  fflush(midi);
}

/* Sends a MIDI SysEX message */
void midi_system_exclusive(FILE *midi, unsigned char *message,
			   unsigned short length) {
  unsigned char *data=(unsigned char *)calloc(length+2, sizeof(unsigned char));

  if(midi==NULL) {
    fprintf(stderr, "midi_system_exclusive() called with null file\n");
    return;
  }

  data[0]=0xf0;
  memcpy(data+1, message, length);
  data[length+1]=0xf7;
  
  fwrite(data, length+2, 1, midi);
  fflush(midi);

  free(data);
}

/* Send a clock message */
void midi_clock(FILE *midi) {
  if(midi==NULL) {
    fprintf(stderr, "midi_clock() called with null file\n");
    return;
  }

  fputc(0xf8, midi);
  fflush(midi);
}


/* Receives a MIDI SysEx message */
void midi_read_system_exclusive(FILE *midi, struct sysex *sysex,
				int autostop) {
  int status=0, i=0;

  if(midi==NULL) {
    fprintf(stderr, "midi_read_system_exclusive() called with null file\n");
    return;
  }

  if(sysex==NULL) {
    fprintf(stderr, "midi_read_system_exclusive() called with null sysex\n");
    return;
  }
    
  /* Read until a SysEx dump begins */
  while(status!=0xf0 && status!=EOF)
    status=fgetc(midi);
  
  /* Return on error */
  if(status==EOF)
    return;
  
  /* Read until the buffer is full or to the end of the SysEx (autostop) */
  while(i<sysex->length) {
    status=fgetc(midi);
    
    if(status==EOF)
      break;
    
    sysex->data[i++]=status;
    if(autostop==1 && sysex->data[i-1]==0xf7)
      break;
  }
  
  /* Remove unnecessary bytes if necessary */
  if(i<sysex->length) {
    sysex->length=i;
    sysex->data=(unsigned char *)realloc(sysex->data, sysex->length);
  }
}

/* Sets the length of a MIDI SysEx message */
void sysex_set_length(struct sysex *sysex, unsigned int length) {
  if(sysex==NULL) {
    fprintf(stderr, "sysex_set_length() called with null sysex\n");
    return;
  }

  sysex->length=length;
  sysex->data=(unsigned char *)realloc(sysex->data, sysex->length);
}

/* Sets the autosend flag of a MIDI SysEx message */
void sysex_set_autosend(struct sysex *sysex, unsigned int autosend) {
  if(sysex==NULL) {
    fprintf(stderr, "sysex_set_autosend() called with null sysex\n");
    return;
  }

  sysex->autosend=autosend;
}

/* Parses a SysEx element in an XML file */
struct sysex *sysex_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct sysex *sysex=NULL;
  
  if((!xmlStrcmp(cur->name, "sysex")) && (cur->ns==ns)) {
    xmlNodePtr temp=cur->xmlChildrenNode;
    char c[3], *prop;
    int i;
    unsigned int d;

    /* Temporary string for hexadecimal parsing */
    c[2]=0;

    /* Allocate sysex */
    sysex=(struct sysex *)calloc(1, sizeof(struct sysex));
    sysex->name=xmlGetProp(cur, "name");
    sysex->length=strlen(temp->content)/2;
    prop=xmlGetProp(cur, "autosend");
    if(prop!=NULL)
      sysex->autosend=atoi(prop);
    sysex->data=(unsigned char *)calloc(sysex->length, sizeof(unsigned char));
    for(i=0; i<sysex->length; i++) {
      c[0]=temp->content[i*2];
      c[1]=temp->content[i*2+1];
      sscanf(c, "%X", &d);
      sysex->data[i]=d;
    }
  } else
    fprintf(stderr, "XML error: expected sysex, got %s\n", cur->name);

  return sysex;
}

/* Saves a SysEx message to an XML file */
void sysex_save(struct sysex *sysex, int number, xmlNodePtr parent) {
  char *c, t[10];
  xmlNodePtr node, lb;
  int i;

  if(sysex==NULL) {
    fprintf(stderr, "sysex_save() called with null sysex\n");
    return;
  }

  c=(char *)calloc(2*sysex->length+1, sizeof(char));
  for(i=0; i<sysex->length; i++)
    sprintf(c+(i*2), "%.2X", sysex->data[i]);

  node=xmlNewChild(parent, NULL, "sysex", c);
  snprintf(t, 10, "%d", number);
  xmlSetProp(node, "number", t);
  xmlSetProp(node, "name", sysex->name);
  snprintf(t, 10, "%d", sysex->autosend);
  xmlSetProp(node, "autosend", t);

  lb=xmlNewText("\n");
  xmlAddChild(parent, lb);

  free(c);
}
