/* mmd.c
 *
 * Copyright 2002 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mmd.h"

#define getlong(a) (((*(a+0))<<24)|((*(a+1))<<16)|((*(a+2))<<8)|((*(a+3))))
#define getword(a) (((*(a+0))<<8)|((*(a+1))))
#define getbyte(a) (*(a))

/* Loads an MMD2 file and returns a pointer to its MMD0 structure */
struct MMD2 *MMD2_load(char *filename) {
  FILE *file;
  struct MMD2 *mmd;

  if((file=fopen(filename, "r"))!=NULL) {
    int flength;
    unsigned char *data;
    
    /* Get file length */
    fseek(file, 0, SEEK_END);
    flength=ftell(file);
    rewind(file);
    
    /* Allocate memory and read file */
    data=(unsigned char *)malloc(flength);
    fread(data, sizeof(char), flength, file);

    mmd=MMD2_parse(data, 0);

    free(data);
    fclose(file);
    return mmd;
  }
  return NULL;
}

/* Allocates a new MMD0 structure and initializes it to default values */
struct MMD2 *MMD2_alloc(void) {
  struct MMD2 *mmd;

  mmd=(struct MMD2 *)calloc(1, sizeof(struct MMD2));
  mmd->id=ID_MMD2;
  mmd->song=MMD2song_alloc(1, 1, 4, 1);
  mmd->blockarr=(struct MMD1Block **)calloc(1, sizeof(struct MMD1Block *));
  mmd->blockarr[0]=MMD1Block_alloc(4, 64, 1);
  mmd->expdata=(struct MMD0exp *)calloc(1, sizeof(struct MMD0exp));
  mmd->expdata->songname="<unnamed>";

  return(mmd);
}

/* Frees an MMD0 structure and all associated data */
void MMD2_free(struct MMD2 *mmd) {
  int i;

  if(mmd) {
    for(i=0; i<mmd->song->numblocks-1; i++) {
      if(mmd->blockarr[i])
	MMD1Block_free(mmd->blockarr[i]);
    }

    if(mmd->blockarr)
      free(mmd->blockarr);

    if(mmd->song)
      MMD2song_free(mmd->song);

    if(mmd->expdata)
      free(mmd->expdata);

    free(mmd);
  }
}

/* Allocates a new MMD2song structure and initializes it to given values */
struct MMD2song *MMD2song_alloc(int numblocks, int songlen, int numtracks, int numpseqs)
{
  struct MMD2song *song;
  int i;

  song=(struct MMD2song *)(calloc(1, sizeof(struct MMD2song)));
  song->numblocks=numblocks;
  song->songlen=songlen;
  song->numtracks=numtracks;
  song->numpseqs=numpseqs;
  song->deftempo=130;
  song->tempo2=6;
  song->mastervol=64;
  song->trackvols=(unsigned char *)calloc(64, sizeof(unsigned char));
  song->sectiontable=(unsigned short *)calloc(1, sizeof(unsigned short));
  song->playseqtable=(struct PlaySeq **)calloc(1, sizeof(struct PlaySeq *));
  song->playseqtable[0]=PlaySeq_alloc("test", 1);
  
  for(i=0; i<64; i++)
    song->trackvols[i]=64;

  return(song);
}

/* Frees an MMD2song structure and all associated data */
void MMD2song_free(struct MMD2song *song) {
  unsigned short i;

  if(song) {
    for(i=0; i<song->numpseqs; i++) {
      if(song->playseqtable[i]) {
	free(song->playseqtable[i]);
	song->playseqtable[i]=0;
      }
    }

    free(song);
  }
}

/* Allocates a new MMD1Block structure and initializes it to given values */
struct MMD1Block *MMD1Block_alloc(int numtracks, int lines, int pages) {
  int i;
  struct MMD1Block *block;

  block=(struct MMD1Block *)calloc(1, sizeof(struct MMD1Block)+4*numtracks*lines);
  block->info=(struct BlockInfo *)calloc(1, sizeof(struct BlockInfo));
  block->info->blockname=(char *)calloc(1, sizeof(char));
  block->info->pagetable=(struct BlockCmdPageTable *)calloc(1, sizeof(struct BlockCmdPageTable));
  block->info->pagetable->num_pages=pages;
  for(i=0; i<pages; i++)
    block->info->pagetable->page[i]=(unsigned short *)calloc(1, 2*numtracks*lines);
  block->numtracks=numtracks;
  block->lines=lines;

  return(block);
}

/* Frees an MMD1Block structure and all associated data */
void MMD1Block_free(struct MMD1Block *block) {
  int i;

  if(block) {
    if(block->info->pagetable) {
      for(i=0; i<block->info->pagetable->num_pages-1; i++) {
	if(block->info->pagetable->page[i])
	  free(block->info->pagetable->page[i]);
      }
      free(block->info->pagetable);
    }
    free(block);
  }
}

struct PlaySeq *PlaySeq_alloc(char *name, int len) {
  struct PlaySeq *playseq;

  playseq=(struct PlaySeq *)calloc(1, sizeof(struct PlaySeq));
  strncpy(playseq->name, name, 32);
  playseq->length=len;

  return(playseq);
}

void PlaySeq_free(struct PlaySeq *playseq) {
  if(playseq)
    free(playseq);
}

struct MMD2 *MMD2_parse(unsigned char *base, int offset) {
  unsigned char *data=base+offset, *data2;
  struct MMD2 *mmd=NULL;
  int i;

  /* Check whether the file begins with MMD2 or not */
  if(data[0]==0x4d && data[1]==0x4d && data[2]==0x44 && data[3]==0x32) {
    mmd=(struct MMD2 *)calloc(1, sizeof(struct MMD2));
    /* Read in MMD2 structure */
    mmd->id=getlong(data);
    mmd->modlen=getlong(data+4);
    mmd->song=MMD2song_parse(base, getlong(data+8));
    mmd->psecnum=getword(data+12);
    mmd->pseq=getword(data+14);
    mmd->blockarr=(struct MMD1Block **)calloc(mmd->song->numblocks,
					      sizeof(struct MMD1Block *));
    for(i=0; i<mmd->song->numblocks; i++)
      mmd->blockarr[i]=MMD1Block_parse(base,
				       getlong(base+getlong(data+16)+i*4));
    mmd->mmdflags=getbyte(data+20);
    mmd->smplarr=(struct InstrHdr **)calloc(mmd->song->numsamples,
					    sizeof(struct InstrHdr *));
    for(i=0; i<mmd->song->numsamples; i++) {
      data2=base+getlong(data+24);
      mmd->smplarr[i]=(struct InstrHdr *)calloc(1, sizeof(struct InstrHdr));
      mmd->smplarr[i]->length=getlong(data2);
      mmd->smplarr[i]->type=getword(data2+4);
    }
    mmd->expdata=MMD0exp_parse(base, getlong(data+32));
    mmd->pstate=getword(data+40);
    mmd->pblock=getword(data+42);
    mmd->pline=getword(data+44);
    mmd->pseqnum=getword(data+46);
    mmd->actplayline=getword(data+48);
    mmd->counter=getbyte(data+50);
    mmd->extra_songs=getbyte(data+51);
  }
  return mmd;
}

/* Allocates a new MMD2song structure and initializes it with values in data */
struct MMD2song *MMD2song_parse(unsigned char *base, int offset) {
  unsigned char *data=base+offset, *data2;
  struct MMD2song *song;
  int i, j;

  song=(struct MMD2song *)(calloc(1, sizeof(struct MMD2song)));
  /* Read in MMD2song structure */
  for(i=0; i<63; i++) {
    song->sample[i].rep=getword(data+i*8);
    song->sample[i].replen=getword(data+i*8+2);
    song->sample[i].midich=getbyte(data+i*8+4);
    song->sample[i].midipreset=getbyte(data+i*8+5);
    song->sample[i].svol=getbyte(data+i*8+6);
    song->sample[i].strans=getbyte(data+i*8+7);
  }
  song->numblocks=getword(data+504);
  song->songlen=getword(data+506);
  song->sectiontable=(unsigned short *)calloc(song->songlen, sizeof(unsigned short));
  data2=base+getlong(data+512);
  for(i=0; i<song->songlen; i++)
    song->sectiontable[i]=getword(data2+i*2);
  song->numtracks=getword(data+520);
  song->trackvols=(unsigned char *)calloc(song->numtracks, sizeof(unsigned char));
  data2=base+getlong(data+516);
  for(i=0; i<song->numtracks; i++)
    song->trackvols[i]=getbyte(data2+i);
  song->numpseqs=getword(data+522);
  song->playseqtable=(struct PlaySeq **)calloc(song->numpseqs,
					       sizeof(struct PlaySeq *));
  for(i=0; i<song->numpseqs; i++) {
    data2=base+getlong(base+getlong(data+508)+i*4);
    song->playseqtable[i]=(struct PlaySeq *)calloc(1, sizeof(struct PlaySeq)+2*getword(data2+40));

    memcpy(song->playseqtable[i]->name, data2, 32);
    song->playseqtable[i]->length=getword(data2+40);
    for(j=0; j<song->playseqtable[i]->length; j++)
      song->playseqtable[i]->seq[j]=getword(data2+42+j*2);
  }
  song->trackpans=(char *)calloc(song->numtracks, sizeof(char));
  data2=base+getlong(data+524);
  for(i=0; i<song->numtracks; i++)
    song->trackpans[i]=getbyte(data2+i);
  song->flags3=getlong(data+528);
  song->voladj=getword(data+532);
  song->channels=getword(data+534);
  song->mix_echotype=getbyte(data+536);
  song->mix_echodepth=getbyte(data+537);
  song->mix_echolen=getword(data+538);
  song->mix_stereosep=getbyte(data+540);
  song->deftempo=getword(data+764);
  song->playtransp=getbyte(data+766);
  song->flags=getbyte(data+767);
  song->flags2=getbyte(data+768);
  song->tempo2=getbyte(data+769);
  song->mastervol=getbyte(data+786);
  song->numsamples=getbyte(data+787);

  return song;
}

struct MMD1Block *MMD1Block_parse(unsigned char *base, int offset) {
  unsigned char *data=base+offset, *data2, *data3;
  struct MMD1Block *block=(struct MMD1Block *)calloc(1, sizeof(struct MMD1Block)+4*getword(data)*(getword(data+2)+1));
  int i, j, k;

  block->numtracks=getword(data);
  block->lines=getword(data+2);
  block->info=(struct BlockInfo *)calloc(1, sizeof(struct BlockInfo));
  /* BlockInfo */
  if(getlong(data+4)!=0) {
    data2=base+getlong(data+4);
    block->info->hlmask=(unsigned int *)calloc(1+block->lines/32,
					       sizeof(unsigned long));
    block->info->blocknamelen=getlong(data2+8);
    block->info->blockname=(char *)calloc(block->info->blocknamelen,
					  sizeof(char));
    memcpy(block->info->blockname, base+getlong(data2+4),
	   block->info->blocknamelen);
    /* BlockCmdPageTables */
    if(getlong(data2+12)!=0) {
      data2=base+getlong(data2+12);
      block->info->pagetable=(struct BlockCmdPageTable *)calloc(1, sizeof(struct BlockCmdPageTable)+getword(data2)*sizeof(unsigned short *));
      block->info->pagetable->num_pages=getword(data2);
      for(i=0; i<block->info->pagetable->num_pages; i++) {
	block->info->pagetable->page[i]=(unsigned short *)calloc(block->numtracks*(block->lines+1)*2, sizeof(unsigned short));
	for(j=0; j<(block->lines+1); j++)
	  for(k=0; k<block->numtracks; k++) {
	    data3=base+getlong(data2+4+i*4);
	    block->info->pagetable->page[i][(j*block->numtracks+k)*2]=getbyte(data3+(j*block->numtracks+k)*2);
	    block->info->pagetable->page[i][(j*block->numtracks+k)*2+1]=getbyte(data3+(j*block->numtracks+k)*2+1);
	  }
      }
    }
  }

  data2=(unsigned char *)block+sizeof(struct MMD1Block);
  for(j=0; j<(block->lines+1); j++)
    for(k=0; k<block->numtracks; k++) {
      data2[(j*block->numtracks+k)*4]=getbyte(data+8+(j*block->numtracks+k)*4);
      data2[(j*block->numtracks+k)*4+1]=getbyte(data+8+(j*block->numtracks+k)*4+1);
      data2[(j*block->numtracks+k)*4+2]=getbyte(data+8+(j*block->numtracks+k)*4+2);
      data2[(j*block->numtracks+k)*4+3]=getbyte(data+8+(j*block->numtracks+k)*4+3);
    }

  return block;
}

struct MMD0exp *MMD0exp_parse(unsigned char *base, int offset) {
  unsigned char *data=base+offset, *data2, *data3;
  struct MMD0exp *mmd0exp=(struct MMD0exp *)calloc(1, sizeof(struct MMD0exp));
  int i;

  if(getlong(data)!=0)
    mmd0exp->nextmod=(struct MMD0 *)MMD2_parse(base, getlong(data));
  mmd0exp->s_ext_entries=getword(data+8);
  mmd0exp->s_ext_entrsz=sizeof(struct InstrExt); /* getword(data+10); */
  mmd0exp->exp_smp=(struct InstrExt *)calloc(mmd0exp->s_ext_entries,
					     sizeof(struct InstrExt));
  /* mmd0exp->s_ext_entrsz); */
  data2=base+getlong(data+4);
  for(i=0; i<mmd0exp->s_ext_entries; i++) {
    mmd0exp->exp_smp[i].hold=getbyte(data2+i*mmd0exp->s_ext_entrsz);
    mmd0exp->exp_smp[i].decay=getbyte(data2+i*mmd0exp->s_ext_entrsz+1);
    mmd0exp->exp_smp[i].suppress_midi_off=getbyte(data2+i*mmd0exp->s_ext_entrsz+2);
    mmd0exp->exp_smp[i].finetune=getbyte(data2+i*mmd0exp->s_ext_entrsz+3);
    if(mmd0exp->s_ext_entrsz>4) {
      mmd0exp->exp_smp[i].default_pitch=getbyte(data2+i*mmd0exp->s_ext_entrsz+4);
      mmd0exp->exp_smp[i].instr_flags=getbyte(data2+i*mmd0exp->s_ext_entrsz+5);
      mmd0exp->exp_smp[i].long_midi_preset=getbyte(data2+i*mmd0exp->s_ext_entrsz+6);
      if(mmd0exp->s_ext_entrsz>8)
	mmd0exp->exp_smp[i].output_device=getbyte(data2+i*mmd0exp->s_ext_entrsz+8);
    }
  }
  mmd0exp->annolen=getlong(data+16);
  mmd0exp->annotxt=(unsigned char *)calloc(mmd0exp->annolen,
					   sizeof(unsigned char));
  memcpy(mmd0exp->annotxt, base+getlong(data+12), mmd0exp->annolen);
  mmd0exp->i_ext_entries=getword(data+24);
  mmd0exp->i_ext_entrsz=getword(data+26);
  mmd0exp->iinfo=(struct MMDInstrInfo *)calloc(mmd0exp->i_ext_entries,
					       mmd0exp->i_ext_entrsz);
  for(i=0; i<mmd0exp->i_ext_entries; i++)
    memcpy(&mmd0exp->iinfo[i], base+getlong(data+20)+i*mmd0exp->i_ext_entrsz,
	   40);
  mmd0exp->jumpmask=getlong(data+28);
  mmd0exp->rgbtable=(unsigned short *)calloc(8, sizeof(unsigned short));
  memcpy(mmd0exp->rgbtable, data+32, 16);
  mmd0exp->channelsplit[0]=getbyte(data+36);
  mmd0exp->channelsplit[1]=getbyte(data+37);
  mmd0exp->channelsplit[2]=getbyte(data+38);
  mmd0exp->channelsplit[3]=getbyte(data+39);

  if(getlong(data+40)!=0) {
    mmd0exp->n_info=(struct NotationInfo *)calloc(1, sizeof(struct NotationInfo));
    data2=base+getlong(data+40);
    mmd0exp->n_info->n_of_sharps=getbyte(data2);
    mmd0exp->n_info->flags=getbyte(data2+1);
    mmd0exp->n_info->trksel[0]=getword(data2+2);
    mmd0exp->n_info->trksel[1]=getword(data2+4);
    mmd0exp->n_info->trksel[2]=getword(data2+6);
    mmd0exp->n_info->trksel[3]=getword(data2+8);
    mmd0exp->n_info->trksel[4]=getword(data2+10);
    memcpy(mmd0exp->n_info->trkshow, data2+12, 96);
  }
  mmd0exp->songnamelen=getlong(data+48);
  mmd0exp->songname=(char *)calloc(mmd0exp->songnamelen, sizeof(char));
  memcpy(mmd0exp->songname, base+getlong(data+44), mmd0exp->songnamelen);

  if(getlong(data+52)!=0) {
    struct MMDDump **dumps;

    data2=base+getlong(data+52);
    mmd0exp->dumps=(struct MMDDumpData *)calloc(1, sizeof(struct MMDDumpData)+getword(data2)*sizeof(struct MMDDump *));
    mmd0exp->dumps->numdumps=getword(data2);
    /* Is this really the only way to do this? I know, this is horrible but
     * this is what the lame MMD spec says */
    dumps=(struct MMDDump **)((char *)mmd0exp->dumps+sizeof(struct MMDDumpData));
    for(i=0; i<mmd0exp->dumps->numdumps; i++) {
      data3=base+getlong(data2+8+4*i);
      dumps[i]=(struct MMDDump *)calloc(1, sizeof(struct MMDDump));
      dumps[i]->length=getlong(data3);
      dumps[i]->ext_len=getword(data3+8);
      dumps[i]->data=(unsigned char *)calloc(dumps[i]->length,
					     sizeof(unsigned char));
      memcpy(dumps[i]->data, base+getlong(data3+4), dumps[i]->length);

      dumps[i]->ext_len=getword(data3+8);
      if(dumps[i]->ext_len>=20)
	memcpy(dumps[i]->name, data3+10, 20);
    }    
  }

  if(getlong(data+56)!=0) {
    mmd0exp->mmdinfo=(struct MMDInfo *)calloc(1, sizeof(struct MMDInfo));
    data2=base+getlong(data+56);
    /* FIX: IMPLEMENT THIS */
  }

  if(getlong(data+64)!=0) {
    mmd0exp->mmdcmd3x=(struct MMDMIDICmd3x *)calloc(1, sizeof(struct MMDMIDICmd3x));
    data2=base+getlong(data+64);
    mmd0exp->mmdcmd3x->struct_vers=getbyte(data2);
    mmd0exp->mmdcmd3x->num_of_settings=getword(data2+2);
    mmd0exp->mmdcmd3x->ctrlr_types=(unsigned char *)calloc(mmd0exp->mmdcmd3x->num_of_settings, sizeof(unsigned char));
    mmd0exp->mmdcmd3x->ctrlr_numbers=(unsigned short *)calloc(mmd0exp->mmdcmd3x->num_of_settings, sizeof(unsigned short));
    data3=base+getlong(data2+4);
    for(i=0; i<mmd0exp->mmdcmd3x->num_of_settings; i++)
      mmd0exp->mmdcmd3x->ctrlr_types[i]=getbyte(data3+i);
    data3=base+getlong(data2+8);
    for(i=0; i<mmd0exp->mmdcmd3x->num_of_settings; i++)
      mmd0exp->mmdcmd3x->ctrlr_numbers[i]=getword(data3+i*2);
  }

  return mmd0exp;
}

void MMD2_print(struct MMD2 *mmd, char *path) {
  char *notestrings[] = {
    "---",
    "C-1", "C#1", "D-1", "D#1", "E-1", "F-1", "F#1", "G-1", "G#1", "A-1", "A#1", "B-1",
    "C-2", "C#2", "D-2", "D#2", "E-2", "F-2", "F#2", "G-2", "G#2", "A-2", "A#2", "B-2",
    "C-3", "C#3", "D-3", "D#3", "E-3", "F-3", "F#3", "G-3", "G#3", "A-3", "A#3", "B-3",
    "C-4", "C#4", "D-4", "D#4", "E-4", "F-4", "F#4", "G-4", "G#4", "A-4", "A#4", "B-4",
    "C-5", "C#5", "D-5", "D#5", "E-5", "F-5", "F#5", "G-5", "G#5", "A-5", "A#5", "B-5",
    "C-6", "C#6", "D-6", "D#6", "E-6", "F-6", "F#6", "G-6", "G#6", "A-6", "A#6", "B-6",
    "C-7", "C#7", "D-7", "D#7", "E-7", "F-7", "F#7", "G-7", "G#7", "A-7", "A#7", "B-7",
    "C-8", "C#8", "D-8", "D#8", "E-8", "F-8", "F#8", "G-8", "G#8", "A-8", "A#8", "B-8",
    "C-9", "C#9", "D-9", "D#9", "E-9", "F-9", "F#9", "G-9", "G#9", "A-9", "A#9", "B-9",
    "C-A", "C#A", "D-A", "D#A", "E-A", "F-A", "F#A", "G-A", "G#A", "A-A", "A#A", "B-A"
  };

  FILE *file;
  struct MMD2song *song;
  int i, j, k;

  song=mmd->song;

  if(path)
    file=fopen(path, "r+");
  else
    file=stdout;

  if(file) {
    fprintf(file, "======= Song: %s\n\n", mmd->expdata->songname);
    fprintf(file, "#  Instrument name                    Repeat RepLen Mc Mpst Vol Trns Hold Dec\n");
    
    fprintf(file, "\nDefault Tempo = %d/%d, Play Transpose = %d\n\n", song->deftempo, song->tempo2, song->playtransp);
    fprintf(file, "Master Volume = %d, Track Volumes:", song->mastervol);
    for(i=0; i<64; i++) {
      if(!(i%8))
	fprintf(file, "\n");
      fprintf(file, "%2d: %3d  ", i, song->trackvols[i]);
    }
    
    fprintf(file, "\n\n======================================================================\n\n");

    for(i=0; i<song->numpseqs; i++) {
      struct PlaySeq *playseq;

      playseq=song->playseqtable[i];

      fprintf(file, "Playing Sequence List #%d (length = %d): %s", i+1, playseq->length, playseq->name);
      for(j=0; j<playseq->length; j++) {
	if(!(j%8))
	  fprintf(file, "\n");
	fprintf(file, "%3d ", playseq->seq[j]);
      }
      fprintf(file, "\n\n");
    }

    fprintf(file, "Section List (length = %d):\n", song->songlen);
    for(i=0; i<song->songlen; i++) {
      fprintf(file, "%.3d: %.3d %s\n", i+1, song->sectiontable[i]+1, song->playseqtable[song->sectiontable[i]]->name);
    }

    fprintf(file, "\n======================================================================\n\n");

    for(i=0; i<song->numblocks; i++) {
      struct MMD1Block *block;
      char *blockdata;
      char datastring[6];
      datastring[5]=0;

      block=mmd->blockarr[i];
      blockdata=(char *)block+sizeof(struct MMD1Block);

      fprintf(file, "Block #%3d: %3d tracks, %3d lines\n", i, block->numtracks, block->lines);
      fprintf(file, "=================================\n");

      for(j=0; j<block->lines; j++) {
	fprintf(file, "%.3d ", j);
	for(k=0; k<block->numtracks; k++) {
	  datastring[0]='0'+blockdata[4*block->numtracks*j+4*k+1]%0x3f;
	  datastring[1]='0'+((blockdata[4*block->numtracks*j+4*k+2]%0xf0)>>4);
	  datastring[2]='0'+blockdata[4*block->numtracks*j+4*k+2]%0x0f;
	  datastring[3]='0'+((blockdata[4*block->numtracks*j+4*k+3]%0xf0)>>4);
	  datastring[4]='0'+blockdata[4*block->numtracks*j+4*k+3]%0x0f;
	  fprintf(file, "%s %s ",
		  notestrings[blockdata[4*block->numtracks*j+4*k]%0x7f],
		  datastring);
	}
	fprintf(file, "\n");
      }
	
      fprintf(file, "\n");
    }
  }
}
