/* player.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define USE_NANOSLEEP 1

#include <gtk/gtk.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>
#ifndef USE_NANOSLEEP
#include <sched.h>
#endif
#include "song.h"
#include "midi.h"
#include "player.h"
#include "editor.h"

/* Refreshes playseq from section and block from position */
static void player_refresh_playseq_and_block(struct player *player) {
  struct song *song;

  if(player==NULL) {
    fprintf(stderr,
	    "player_refresh_playseq_and_block() called with null player\n");
    return;
  }

  song=player->song;

  if(player->section>=song->numsections)
    player->section=0;
  player->playseq=song->sections[player->section];

  if(player->position>=song->playseqs[player->playseq]->length) {
    player->position=0;
    player_refresh_playseq_and_block(player);
  } else
    player->block=song->playseqs[player->playseq]->blocknumbers[player->position];
}

/* Advances in section and jumps to the beginning if necessary */
static void player_next_section(struct player *player) {
  if(player==NULL) {
    fprintf(stderr, "player_next_section() called with null player\n");
    return;
  }

  player->section++;
  if(player->section>=player->song->numsections)
    player->section=0;
}

/* Advances in playing sequence and jumps to next section if necessary */
static void player_next_playseq(struct player *player) {
  if(player==NULL) {
    fprintf(stderr, "player_next_playseq() called with null player\n");
    return;
  }

  player->position++;
  if(player->position>=player->song->playseqs[player->playseq]->length) {
    player->position=0;
    player_next_section(player);
  }
}

/* Handles one tick by playing all notes scheduled to be played */
static void player_next_tick(struct player *player) {
  int i, j, changetempo=0;
  struct timeval tod;
  struct song *song;
  struct block *b;
  unsigned char postcommand=0, postvalue=0;

  if(player==NULL) {
    fprintf(stderr, "player_next_tick() called with null player\n");
    return;
  }

  song=player->song;
  b=song->blocks[player->block];

  /* Send MIDI sync if requested */
  if(song->sendsync)
    midi_clock(player->midi);

  /* Play notes scheduled to be played */
  for(i=0; i<b->tracks; i++) {
    unsigned int volume=127;
    int delay=0, hold=-1;
    unsigned char note=b->notes[(b->tracks*player->line+i)*2];
    unsigned char instrument=b->notes[(b->tracks*player->line+i)*2+1];
    /* Handle commands on all command pages */
    for(j=0; j<b->commandpages; j++) {
      unsigned char command=b->commands[j*b->tracks*b->length*2+(b->tracks*player->line+i)*2];
      unsigned char value=b->commands[j*b->tracks*b->length*2+(b->tracks*player->line+i)*2+1];
      int midichannel=-1;

      /* Check which MIDI channel the command will affect */
      if(instrument!=0)
	/* Instrument number defines MIDI channel */
	midichannel=song->instruments[instrument-1]->midichannel;
      else
	/* Note playing defines MIDI channel */
	midichannel=player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL];

      /* Check for previous command if any */
      if(command==COMMAND_PREVIOUS_COMMAND_VALUE) {
	if(value!=0)
	  command=player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_PREVIOUS_COMMAND];
      } else
	player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_PREVIOUS_COMMAND]=command;
      
      switch(command) {
      case COMMAND_PITCH_WHEEL:
	/* Pitch wheel can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(player->tick==0) {
	      midi_pitch_wheel(player->midi, midichannel, value);
	      player->midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]=value;
	    }
	  } else {
	    if(player->tick<song->ticksperline-1) {
	      int delta=(value-0x80-player->midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL])/((int)song->ticksperline-1);

	      midi_pitch_wheel(player->midi, midichannel, player->midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]+player->tick*delta);
	    } else {
	      midi_pitch_wheel(player->midi, midichannel, value-0x80);
	      player->midicontrollervalues[midichannel*VALUES+VALUES_PITCH_WHEEL]=value-0x80;
	    }
	  }
	}
	break;
      case COMMAND_END_BLOCK:
	/* Only on last tick */
	if(player->tick==song->ticksperline-1) {
	  postcommand=COMMAND_END_BLOCK;
	  postvalue=value;
	}
	break;
      case COMMAND_PLAYSEQ_POSITION:
	/* Only on last tick */
	if(player->tick==song->ticksperline-1) {
	  postcommand=COMMAND_PLAYSEQ_POSITION;
	  postvalue=value;
	}
	break;
      case COMMAND_STOP:
	/* Only on last tick */
	if(player->tick==song->ticksperline-1)
	  postcommand=COMMAND_STOP;
	break;
      case COMMAND_SEND_SYSEX:
	/* Only on first tick */
	if(player->tick==0 && value<song->numsysexes)
	  midi_system_exclusive(player->midi,
				song->sysexes[value]->data,
				song->sysexes[value]->length);
	break;
      case COMMAND_HOLD:
	hold=value;
	break;
      case COMMAND_RETRIGGER:
	delay=-value;
	break;
      case COMMAND_DELAY:
	delay=value;
	break;
      case COMMAND_VELOCITY:
	if(note!=0) {
	  volume=value;
	  if(midichannel!=-1)
	    player->midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value;
	} else {
	  /* Note playing defines MIDI channel */
	  midichannel=player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL];

	  if(midichannel!=-1) {
	    if(value<0x80) {
	      if(player->tick==0) {
		if(value>0) {
		  midi_aftertouch(player->midi, midichannel, player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE], value);
		  player->midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value;
		} else {
		  midi_note_off(player->midi, midichannel,
				player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE],
				127);
		  player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]=-1;
		}
	      }
	    } else {
	      if(player->tick<song->ticksperline-1) {
		int delta=(value-0x80-player->midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH])/((int)song->ticksperline-1);
		
		midi_aftertouch(player->midi, midichannel, player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE], player->midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]+player->tick*delta);
	      } else {
		midi_aftertouch(player->midi, midichannel, player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE], value-0x80);
		player->midicontrollervalues[midichannel*VALUES+VALUES_AFTERTOUCH]=value-0x80;
	      }
	    }
	  }
	}
	break;
      case COMMAND_CHANNEL_PRESSURE:
	/* Channel pressure can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(player->tick==0) {
	      midi_channel_pressure(player->midi, midichannel, value);
	      player->midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]=value;
	    }
	  } else {
	    if(player->tick<song->ticksperline-1) {
	      int delta=(value-0x80-player->midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE])/((int)song->ticksperline-1);

	      midi_channel_pressure(player->midi, midichannel, player->midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]+player->tick*delta);
	    } else {
	      midi_channel_pressure(player->midi, midichannel, value-0x80);
	      player->midicontrollervalues[midichannel*VALUES+VALUES_CHANNEL_PRESSURE]=value-0x80;
	    }
	  }
	}
	break;
      case COMMAND_TPL:
	song_set_tpl(song, value);
	changetempo=1;
	break;
      case COMMAND_TEMPO:
	song_set_tempo(song, value);
	changetempo=1;
	break;
      }

      /* Handle MIDI controllers */
      if(command>=COMMAND_MIDI_CONTROLLERS) {
	/* MIDI controllers can be set if the MIDI channel is known */
	if(midichannel!=-1) {
	  if(value<0x80) {
	    if(player->tick==0) {
	      midi_controller(player->midi, midichannel, command-COMMAND_MIDI_CONTROLLERS,
			      value);
	      player->midicontrollervalues[midichannel*VALUES+command-COMMAND_MIDI_CONTROLLERS]=value;
	    }
	  } else {
	    if(player->tick<song->ticksperline-1) {
	      int delta=(value-0x80-player->midicontrollervalues[midichannel*VALUES+command-COMMAND_MIDI_CONTROLLERS])/((int)song->ticksperline-1);

	      midi_controller(player->midi, midichannel, command-COMMAND_MIDI_CONTROLLERS,
			      player->midicontrollervalues[midichannel*VALUES+command-COMMAND_MIDI_CONTROLLERS]+player->tick*delta);
	    } else {
	      midi_controller(player->midi, midichannel, command-COMMAND_MIDI_CONTROLLERS, value-0x80);
	      player->midicontrollervalues[midichannel*VALUES+command-COMMAND_MIDI_CONTROLLERS]=value-0x80;
	    }
	  }
	}
      }
    }
    
    if(note!=0 && ((delay>=0 && player->tick==delay) ||
		   (delay<0 && player->tick%(-delay)==0))) {
      note--;

      player_play_note(player, instrument-1, note, volume, i);

      /* If no hold value was defined use the instrument's hold value */
      if(hold==-1 && instrument<=song->numinstruments &&
	 song->instruments[instrument-1]!=NULL)
	hold=song->instruments[b->notes[(b->tracks*player->line+i)*2+1]-1]->hold;

      if(hold==0)
	player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=-1;
      else
	player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=hold;
    }
  }

  /* Decrement hold times of notes and stop notes that should be stopped */
  for(i=0; i<song->maxtracks; i++) {
    if(player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]!=-1) {
      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]--;
      if(player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]==-1) {
	midi_note_off(player->midi,
		      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL],
		      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE],
		      127);
	player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]=-1;
      }
    }
  }

  /* Next tick */
  player->tick++;
  player->tick%=song->ticksperline;

  /* Advance and handle post commands if ticksperline ticks have passed */
  if(player->tick==0) {
    int changeblock=0;

    player->line++;

    switch(postcommand) {
    case COMMAND_END_BLOCK:
      player->line=postvalue;
      player_next_playseq(player);
      changeblock=1;
      break;
    case COMMAND_PLAYSEQ_POSITION:
      player->line=0;
      player->position=postvalue;
      if(player->position>=song->playseqs[player->playseq]->length) {
	player->position=0;
	player_next_section(player);
      }
      changeblock=1;
      break;
    case COMMAND_STOP:
      player_stop(player);
      return;
      break;
    default:
      /* Advance in block */
      if(player->line>=song->blocks[player->block]->length) {
	player->line=0;
	if(player->mode==MODE_PLAY_SONG) {
	  player_next_playseq(player);
	  changeblock=1;
	}
      }
      break;
    }

    if(changeblock==1)
      player_refresh_playseq_and_block(player);

    gettimeofday(&tod, NULL);

    /* Tell the editor about the new position if an editor exists */
    if(player->editor!=NULL)
      editor_set_values(player->editor, player->section, player->playseq,
			player->position, player->block, player->line,
			(int)(player->playedsofar.tv_sec*1000+
			      player->playedsofar.tv_usec/1000+
			      (tod.tv_sec*1000+tod.tv_usec/1000)-
			      (player->playingstarted.tv_sec*1000+
			       player->playingstarted.tv_usec/1000))/1000,
			changetempo);
  }
}

/* Plays a note using given instrument on a given track */
void player_play_note(struct player *player,
		      unsigned int instrument, unsigned char note,
		      unsigned char volume, unsigned char track) {
  struct song *song;

  if(player==NULL) {
    fprintf(stderr, "player_play_note() called with null player\n");
    return;
  }

  song=player->song;

  /* Stop currently playing note */
  if(player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]!=-1) {
    midi_note_off(player->midi,
		  player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL],
		  player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE],
		  127);
  }

  /* Don't play a note if the instrument does not exist */
  if(instrument<song->numinstruments && song->instruments[instrument]!=NULL) {
    /* Apply note transpose */
    note+=song->instruments[instrument]->transpose;

    /* Set new track status (channel, note, volume, time) */
    player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL]=song->instruments[instrument]->midichannel;
    player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]=note;
    player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]=song->instruments[instrument]->defaultvelocity*volume/127*(song->trackvolumes[track]&127)/127*((255-song->trackvolumes[track])&128)/128*song->mastervolume/127;
    if(song->instruments[instrument]->hold>0)
      player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=song->instruments[instrument]->hold;
    else
      player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=-1;

    /* Make sure the volume isn't too large */
    if(player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]<0)
      player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]=127;
    
    /* Play note */
    midi_note_on(player->midi,
		 player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL],
		 player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE],
		 player->trackstatus[track*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]);
  }
}

/* Stops notes playing at the moment */
void player_stop_notes(struct player *player) {
  int i;

  if(player==NULL) {
    fprintf(stderr, "player_stop_notes() called with null player\n");
    return;
  }

  for(i=0; i<player->song->maxtracks; i++) {
    if(player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]!=-1) {
      midi_note_off(player->midi,
		    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL], player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE], 127);
      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL]=-1;
      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]=-1;
      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]=-1;
      player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=-1;
    }
  }
}

/* Stops all notes playing at the moment */
void player_stop_all_notes(struct player *player) {
  int i, j;

  if(player==NULL) {
    fprintf(stderr, "player_stop_all_notes() called with null player\n");
    return;
  }

  for(i=0; i<16; i++)
    for(j=0; j<128; j++)
      midi_note_off(player->midi, i, j, 127);
}

/* Resets the pitch wheel on all channels */
void player_reset_pitch(struct player *player) {
  int i;

  if(player==NULL) {
    fprintf(stderr, "player_reset_pitch() called with null player\n");
    return;
  }

  for(i=0; i<16; i++)
    midi_pitch_wheel(player->midi, i, 64);
}

/* Player thread */
static void player_thread(struct player *player) {
#ifdef USE_NANOSLEEP
  struct timespec req, rem;
#endif
  struct timeval next, now;

  if(player==NULL) {
    fprintf(stderr, "player_thread() called with null player\n");
    return;
  }

  player->tick=0;
  next.tv_sec=player->playingstarted.tv_sec;
  next.tv_usec=player->playingstarted.tv_usec;

  while(1) {
    /* Calculate time of next tick (tick every 1000000/((BPM/60)*24) usecs) */
    next.tv_sec+=(25/player->song->tempo)/10;
    next.tv_usec+=((2500000/player->song->tempo)%1000000);
    while(next.tv_usec>1000000) {
      next.tv_sec++;
      next.tv_usec-=1000000;
    }

    /* Calculate difference between now and the next tick */
#ifdef USE_NANOSLEEP
    gettimeofday(&now, NULL);
    req.tv_sec=next.tv_sec-now.tv_sec;
    req.tv_nsec=(next.tv_usec-now.tv_usec)*1000;
    while(req.tv_nsec<0) {
      req.tv_sec--;
      req.tv_nsec+=1000000000;
    }

    /* Sleep until the next tick if necessary */
    if(req.tv_sec>=0)
      while(nanosleep(&req, &rem)==-1);
#else
    gettimeofday(&now, NULL);
    while(now.tv_sec<next.tv_sec ||
          (now.tv_sec==next.tv_sec && now.tv_usec<next.tv_usec)) {
      sched_yield();
      gettimeofday(&now, NULL);
    }
#endif

    /* Handle this tick */
    player_next_tick(player);

    /* Check whether this thread should be killed */
    pthread_testcancel();
  }
}

/* Creates a player for a song */
struct player *player_open(struct song *song, struct editor *editor) {
  struct player *player=(struct player *)calloc(1, sizeof(struct player));
  int i;

  player->song=song;
  player->editor=editor;
  player->midi=midi_open("/dev/midi");
  player->threadstatus=-1;

  /* Initialize notes playing array */
  player_trackstatus_create(player, 0);

  /* Send SysEx messages to be autosent */
  for(i=0; i<player->song->numsysexes; i++) {
    if(player->song->sysexes[i]->autosend)
      midi_system_exclusive(player->midi,
			    player->song->sysexes[i]->data,
			    player->song->sysexes[i]->length);
  }

  /* Reset MIDI controller values */
  for(i=0; i<16*VALUES; i++)
    player->midicontrollervalues[i]=0;

  return player;
}

/* Closes a player for a song */
void player_close(struct player *player) {
  if(player==NULL) {
    fprintf(stderr, "player_close() called with null player\n");
    return;
  }

  player_stop(player);

  /* Free the track status array */
  if(player->trackstatus!=NULL)
    free(player->trackstatus);

  /* Close MIDI */
  if(player->midi!=NULL)
    midi_close(player->midi);

  free(player);
}

/* Starts the player thread */
void player_start(struct player *player, unsigned int mode,
		  int section, int position, int block, int cont) {
  if(player==NULL) {
    fprintf(stderr, "player_start() called with null player\n");
    return;
  }

  player_stop(player);

  player->mode=mode;
  player->tick=0;

  switch(mode) {
  case MODE_PLAY_SONG:
    if(cont) {
      player->section=section;
      player->position=position;
    } else {
      player->section=0;
      player->position=0;
      player->line=0;
    }
    player_refresh_playseq_and_block(player);
    break;
  case MODE_PLAY_BLOCK:
    player->block=block;
    if(!cont)
      player->line=0;
    break;
  }

  /* Get the starting time */
  gettimeofday(&player->playingstarted, NULL);
  if(!cont) {
    player->playedsofar.tv_sec=0;
    player->playedsofar.tv_usec=0;
  }

  /* Create a new thread only if there isn't one already */
  if(player->threadstatus==-1)
    player->threadstatus=pthread_create(&player->thread, NULL,
					(void *)player_thread, player);
}

/* Kills the player thread */
void player_stop(struct player *player) {
  if(player==NULL) {
    fprintf(stderr, "player_stop() called with null player\n");
    return;
  }

  player_stop_notes(player);

  if(player->mode!=MODE_IDLE)
    player->mode=MODE_IDLE;

  if(player->threadstatus==0) {
    pthread_cancel(player->thread);
    pthread_join(player->thread, NULL);
    player->threadstatus=-1;
  }
}

/* Reallocate track status array */
void player_trackstatus_create(struct player *player, int oldmax) {
  int i;

  if(player==NULL) {
    fprintf(stderr, "player_trackstatus_create() called with null player\n");
    return;
  }

  player->trackstatus=(char *)realloc(player->trackstatus,
				      TRACK_STATUS_NUMVALUES*player->song->maxtracks*sizeof(char));
  
  /* Set new tracks to -1 */
  for(i=oldmax; i<player->song->maxtracks; i++) {
    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_MIDI_CHANNEL]=-1;
    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_NOTE]=-1;
    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_VOLUME]=-1;
    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_HOLD]=-1;
    player->trackstatus[i*TRACK_STATUS_NUMVALUES+TRACK_STATUS_PREVIOUS_COMMAND]=0;
  }
}
