/* song.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <libxml/tree.h>
#include "song.h"

/* Allocates a new song structure and initializes it to default values */
struct song *song_alloc(void) {
  struct song *song;
  int i;

  song=(struct song *)calloc(1, sizeof(struct song));
  song->name=(char *)strdup("Untitled");
  song->tempo=130;
  song->ticksperline=6;
  song->numsections=1;
  song->sections=(unsigned int *)calloc(1, sizeof(unsigned int));
  song->numplayseqs=1;
  song->playseqs=(struct playseq **)calloc(1, sizeof(struct playseq *));
  song->playseqs[0]=playseq_alloc();
  song->numblocks=1;
  song->blocks=(struct block **)calloc(1, sizeof(struct block *));
  song->blocks[0]=block_alloc(4, 64, 1);
  song->numinstruments=1;
  song->instruments=(struct instrument **)calloc(1, sizeof(struct instrument *));
  song->instruments[0]=instrument_alloc();
  song->maxtracks=4;
  song->trackvolumes=(unsigned int *)calloc(4, sizeof(unsigned int));
  song->mastervolume=127;
  for(i=0; i<song->maxtracks; i++)
    song->trackvolumes[i]=127;

  return(song);
}

/* Frees a song structure and its contents */
void song_free(struct song *song) {
  if(song==NULL) {
    fprintf(stderr, "song_free() called with null song\n");
    return;
  }

  if(song->name)
    free(song->name);
    
  if(song->numsections>0)
    free(song->sections);
    
  if(song->numplayseqs>0) {
    int i;
      
    for(i=0; i<song->numplayseqs; i++)
      playseq_free(song->playseqs[i]);
      
    free(song->playseqs);
  }
    
  if(song->numblocks>0) {
    int i;
      
    for(i=0; i<song->numblocks; i++)
      block_free(song->blocks[i]);
      
    free(song->blocks);
  }
    
  if(song->numinstruments>0) {
    int i;
      
    for(i=0; i<song->numinstruments; i++)
      if(song->instruments[i]!=NULL)
	instrument_free(song->instruments[i]);
      
    free(song->instruments);
  }
    
  if(song->maxtracks>0)
    free(song->trackvolumes);
    
  free(song);
}

/* Inserts a new block in the block array in the given position */
void song_insert_block(struct song *song, int pos, int current) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_insert_block() called with null song\n");
    return;
  }

  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numblocks)
    pos=song->numblocks;

  song->numblocks++;
  /* Reallocate the block array */
  song->blocks=(struct block **)realloc(song->blocks, song->numblocks*sizeof(struct block *));

 /* Move rest of the blocks onwards */
  for(i=song->numblocks-1; i>pos; i--)
    song->blocks[i]=song->blocks[i-1];

  /* Insert a new block similar to the current block */
  song->blocks[pos]=block_alloc(song->blocks[current]->tracks,
				song->blocks[current]->length,
				song->blocks[current]->commandpages);
  
  /* Update playing sequences */
  for(i=0; i<song->numplayseqs; i++) {
    int j;

    for(j=0; j<song->playseqs[i]->length; j++)
      if(song->playseqs[i]->blocknumbers[j]>=pos)
	song->playseqs[i]->blocknumbers[j]++;
  }
}

/* Deletes a block from the given position of the block array */
void song_delete_block(struct song *song, int pos) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_delete_block() called with null song\n");
    return;
  }
  
  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numblocks)
    pos=song->numblocks-1;

  /* Don't delete the last block */
  if(song->numblocks>1) {
    /* Free the block in question */
    block_free(song->blocks[pos]);
    song->numblocks--;
    
    /* Move rest of the blocks backwards */
    for(i=pos; i<song->numblocks; i++)
      song->blocks[i]=song->blocks[i+1];
    
    /* Reallocate the block array */
    song->blocks=(struct block **)realloc(song->blocks, song->numblocks*sizeof(struct block *));
    
    /* Update playing sequences */
    for(i=0; i<song->numplayseqs; i++) {
      int j;
      
      for(j=0; j<song->playseqs[i]->length; j++)
	if(song->playseqs[i]->blocknumbers[j]>=pos &&
	   song->playseqs[i]->blocknumbers[j]>0)
	  song->playseqs[i]->blocknumbers[j]--;
    }
  }
}

/* Inserts a new playseq in the playseq array in the given position */
void song_insert_playseq(struct song *song, int pos) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_insert_playseq() called with null song\n");
    return;
  }

  /* Check playseq existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numplayseqs)
    pos=song->numplayseqs;

  song->numplayseqs++;
  /* Reallocate the playseqs array */
  song->playseqs=(struct playseq **)realloc(song->playseqs, song->numplayseqs*sizeof(struct playseq *));

  /* Move rest of the playseqs onwards */
  for(i=song->numplayseqs-1; i>pos; i--)
    song->playseqs[i]=song->playseqs[i-1];

  /* Insert a new playing sequence */
  song->playseqs[pos]=playseq_alloc();
  
  /* Update sections */
  for(i=0; i<song->numsections; i++) {
    if(song->sections[i]>=pos)
	song->sections[i]++;
  }
}

/* Deletes a playseq from the given position of the playseq array */
void song_delete_playseq(struct song *song, int pos) {
  int i;
  
  if(song==NULL) {
    fprintf(stderr, "song_delete_playseq() called with null song\n");
    return;
  }

  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numplayseqs)
    pos=song->numplayseqs-1;

  /* Don't delete the last playseq */
  if(song->numplayseqs>1) {
    /* Free the playseq to be deleted */
    playseq_free(song->playseqs[pos]);
    song->numplayseqs--;
    
    /* Move rest of the playseqs backwards */
    for(i=pos; i<song->numplayseqs; i++)
      song->playseqs[i]=song->playseqs[i+1];
    
    /* Reallocate the playseqs array */
    song->playseqs=(struct playseq **)realloc(song->playseqs, song->numplayseqs*sizeof(struct playseq *));

    /* Update section lists */
    for(i=0; i<song->numsections; i++) {
      if(song->sections[i]>=pos && song->sections[i]>0)
	  song->sections[i]--;
    }
  }
}


/* Inserts a new section in the section array in the given position */
void song_insert_section(struct song *song, int pos) {
  int i;
  unsigned int old;

  if(song==NULL) {
    fprintf(stderr, "song_insert_section() called with null song\n");
    return;
  }

  /* Check that the value is possible */
  if(pos<0)
    pos=0;
  else if(pos>song->numsections)
    pos=song->numsections;

  /* Which playing sequence number to insert */
  if(pos<song->numsections)
    old=song->sections[pos];
  else
    old=song->sections[song->numsections-1];

  song->numsections++;
  /* Reallocate the sections array */
  song->sections=(unsigned int *)realloc(song->sections, song->numsections*sizeof(unsigned int));

  /* Move rest of the sections onwards */
  for(i=song->numsections-1; i>pos; i--)
    song->sections[i]=song->sections[i-1];

  song->sections[pos]=old;
}

/* Deletes a section from the given position of the section array */
void song_delete_section(struct song *song, int pos) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_delete_section() called with null song\n");
    return;
  }
  
  /* Check block existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numsections)
    pos=song->numsections-1;

  /* Don't delete the last section */
  if(song->numsections>1) {
    song->numsections--;
    
    /* Move rest of the sections backwards */
    for(i=pos; i<song->numsections; i++)
      song->sections[i]=song->sections[i+1];
    
    /* Reallocate the sections array */
    song->sections=(unsigned int *)realloc(song->sections, song->numsections*sizeof(unsigned int));
  }
}

/* Inserts a new sysex in the sysex array in the given position */
void song_insert_sysex(struct song *song, int pos) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_insert_sysex() called with null song\n");
    return;
  }

  /* Check sysex existence */
  if(pos<0)
    pos=0;
  else if(pos>song->numsysexes)
    pos=song->numsysexes;

  song->numsysexes++;
  /* Reallocate the sysex array */
  song->sysexes=(struct sysex **)realloc(song->sysexes, song->numsysexes*sizeof(struct sysex *));

  /* Move rest of the sysexes onwards */
  for(i=song->numsysexes-1; i>pos; i--)
    song->sysexes[i]=song->sysexes[i-1];

  /* Insert a new sysex similar to the current sysex */
  song->sysexes[pos]=(struct sysex *)calloc(1, sizeof(struct sysex));
}

/* Deletes a sysex from the given position of the sysex array */
void song_delete_sysex(struct song *song, int pos) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_delete_sysex() called with null song\n");
    return;
  }

  /* Check sysex existence */
  if(pos<0)
    pos=0;
  else if(pos>=song->numsysexes)
    pos=song->numsysexes-1;

  /* Don't delete inexisting sysexes */
  if(song->numsysexes>0) {
    /* Free the sysex in question */
    if(song->sysexes[pos]->data!=NULL)
      free(song->sysexes[pos]->data);
    if(song->sysexes[pos]->name!=NULL)
      free(song->sysexes[pos]->name);
    free(song->sysexes[pos]);
    song->numsysexes--;
    
    /* Move rest of the sysexes backwards */
    for(i=pos; i<song->numsysexes; i++)
      song->sysexes[i]=song->sysexes[i+1];
    
    /* Reallocate the sysex array */
    song->sysexes=(struct sysex **)realloc(song->sysexes, song->numsysexes*sizeof(struct sysex *));
  }
}

/* Sets a section in the given position to point somewhere */
void song_set_section(struct song *song, int pos, int playseq) {
  if(song==NULL) {
    fprintf(stderr, "song_set_section() called with null song\n");
    return;
  }

  if(playseq>=0 && playseq<song->numplayseqs)
    song->sections[pos]=playseq;
}

/* Sets the number of ticks per line for a song */
void song_set_tpl(struct song *song, unsigned int ticksperline) {
  if(song==NULL) {
    fprintf(stderr, "song_set_tpl() called with null song\n");
    return;
  }

  song->ticksperline=ticksperline;
}

/* Sets the tempo of a song */
void song_set_tempo(struct song *song, unsigned int tempo) {
  if(song==NULL) {
    fprintf(stderr, "song_set_tempo() called with null song\n");
    return;
  }

  song->tempo=tempo;
}

/* If the maximum number of tracks has changed recreate the track volumes */
int song_check_maxtracks(struct song *song) {
  int i, oldmax, max=0;

  if(song==NULL) {
    fprintf(stderr, "song_check_maxtracks() called with null song\n");
    return 0;
  }

  oldmax=song->maxtracks;

  /* Check the maximum number of tracks; */
  for(i=0; i<song->numblocks; i++)
    if(song->blocks[i]->tracks>max)
      max=song->blocks[i]->tracks;
  
  /* If the maximum number of tracks as changed... */
  if(oldmax!=max) {
    /* Reallocate volume array */
    song->trackvolumes=(unsigned int *)realloc(song->trackvolumes,
					       max*sizeof(unsigned int));

    /* Set new tracks to volume 127 */
    for(i=song->maxtracks; i<max; i++)
      song->trackvolumes[i]=127;

    song->maxtracks=max;
    return 1;
  } else
    return 0;
}

/* Make sure the instrument exists; add instruments if necessary */
void song_check_instrument(struct song *song, int instrument) {
  if(song==NULL) {
    fprintf(stderr, "song_check_instrument() called with null song\n");
    return;
  }

  if(instrument>=song->numinstruments || song->instruments[instrument]==NULL) {
    if(instrument>=song->numinstruments) {
      int i;

      song->instruments=realloc(song->instruments,
				(instrument+1)*sizeof(struct instrument *));
      for(i=song->numinstruments; i<(instrument+1); i++)
	song->instruments[i]=instrument_alloc();

      song->numinstruments=instrument+1;
    }
  }
}

/* Converts an MMD2 module to a song */
struct song *song_convert_MMD2(struct MMD2 *mmd) {
  struct song *song;
  int i, j, k, l;
  unsigned char commandconversion[256];

  if(mmd==NULL)
    return NULL;

  /* Create a table for converting MMD commands; values need special handling */
  for(i=0; i<256; i++)
    commandconversion[i]=i;
  commandconversion[0]=COMMAND_PREVIOUS_COMMAND_VALUE;
  commandconversion[1]=COMMAND_PITCH_WHEEL;
  commandconversion[2]=COMMAND_PITCH_WHEEL;
  commandconversion[3]=COMMAND_PITCH_WHEEL;
  commandconversion[4]=COMMAND_MIDI_CONTROLLERS+1;
  commandconversion[5]=COMMAND_NOT_DEFINED;
  commandconversion[6]=COMMAND_NOT_DEFINED;
  commandconversion[7]=COMMAND_NOT_DEFINED;
  commandconversion[8]=COMMAND_HOLD;
  commandconversion[9]=COMMAND_TPL;
  commandconversion[10]=COMMAND_VELOCITY;
  commandconversion[11]=COMMAND_PLAYSEQ_POSITION;
  commandconversion[12]=COMMAND_VELOCITY;
  commandconversion[13]=COMMAND_CHANNEL_PRESSURE;
  commandconversion[14]=COMMAND_MIDI_CONTROLLERS+10;
  commandconversion[15]=COMMAND_TEMPO;
  commandconversion[16]=COMMAND_SEND_SYSEX;
  commandconversion[17]=COMMAND_NOT_DEFINED;
  commandconversion[18]=COMMAND_NOT_DEFINED;
  commandconversion[19]=COMMAND_NOT_DEFINED;
  commandconversion[20]=COMMAND_NOT_DEFINED;
  commandconversion[21]=COMMAND_NOT_DEFINED;
  commandconversion[22]=COMMAND_MIDI_CONTROLLERS+7;
  commandconversion[23]=COMMAND_NOT_DEFINED;
  commandconversion[24]=COMMAND_NOT_DEFINED;
  commandconversion[25]=COMMAND_NOT_DEFINED;
  commandconversion[26]=COMMAND_NOT_DEFINED;
  commandconversion[27]=COMMAND_NOT_DEFINED;
  commandconversion[28]=COMMAND_NOT_DEFINED;
  commandconversion[29]=COMMAND_END_BLOCK;
  commandconversion[30]=COMMAND_NOT_DEFINED;
  commandconversion[31]=COMMAND_DELAY;

  song=(struct song *)calloc(1, sizeof(struct song));
  if(mmd->expdata && mmd->expdata->songname)
    song->name=(char *)strdup(mmd->expdata->songname);
  else
    song->name="Untitled";
  if(mmd->song) {
    /* Tempo */
    song->tempo=mmd->song->deftempo;
    song->ticksperline=mmd->song->tempo2;
    /* Sections */
    song->numsections=mmd->song->songlen;
    song->sections=(unsigned int *)calloc(song->numsections,
					    sizeof(unsigned int));
    for(i=0; i<song->numsections; i++) {
      /* FIX: Where to get these? */
    }
    /* Playing sequences */
    song->numplayseqs=mmd->song->numpseqs;
    song->playseqs=(struct playseq **)calloc(song->numplayseqs,
					     sizeof(struct playseq *));
    for(i=0; i<song->numplayseqs; i++) {
      struct playseq *playseq=(struct playseq *)calloc(1,
						       sizeof(struct playseq));
      struct PlaySeq *PlaySeq=mmd->song->playseqtable[i];
      song->playseqs[i]=playseq;
      if(PlaySeq->name)
	playseq->name=(char *)strdup(PlaySeq->name);
      playseq->length=PlaySeq->length;
      playseq->blocknumbers=(unsigned int *)calloc(playseq->length,
						     sizeof(unsigned int));
      for(j=0; j<song->playseqs[i]->length; j++)
	playseq->blocknumbers[j]=PlaySeq->seq[j];
    }
    /* Blocks */
    song->numblocks=mmd->song->numblocks;
    song->blocks=(struct block **)calloc(song->numblocks,
					 sizeof(struct block *));
    for(i=0; i<song->numblocks; i++) {
      struct block *block;
      struct MMD1Block *MMD1Block=mmd->blockarr[i];
      unsigned char *notes=(unsigned char *)MMD1Block+sizeof(struct MMD1Block);
      if(MMD1Block->info->pagetable)
	block=block_alloc(MMD1Block->numtracks, MMD1Block->lines+1,
			  MMD1Block->info->pagetable->num_pages+1);
      else
	block=block_alloc(MMD1Block->numtracks, MMD1Block->lines+1, 1);

      song->blocks[i]=block;
      for(j=0; j<block->length; j++)
	for(k=0; k<block->tracks; k++) {
	  block->notes[(j*block->tracks+k)*2]=notes[(j*block->tracks+k)*4];
	  block->notes[(j*block->tracks+k)*2+1]=notes[(j*block->tracks+k)*4+1];
	  block->commands[(j*block->tracks+k)*2]=commandconversion[notes[(j*block->tracks+k)*4+2]];
	  block->commands[(j*block->tracks+k)*2+1]=notes[(j*block->tracks+k)*4+3];
	  /* Command values may need special handling */
	  song_convert_commands(&(block->commands[(j*block->tracks+k)*2]),
				mmd->expdata->mmdcmd3x);
	}

      if(MMD1Block->info->pagetable) {
	for(j=0; j<MMD1Block->info->pagetable->num_pages; j++) {
	  for(k=0; k<(MMD1Block->lines+1); k++)
	    for(l=0; l<MMD1Block->numtracks; l++) {
	      block->commands[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2]=commandconversion[MMD1Block->info->pagetable->page[j][(k*MMD1Block->numtracks+l)*2]];
	      block->commands[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2+1]=MMD1Block->info->pagetable->page[j][(k*MMD1Block->numtracks+l)*2+1];

	      /* Command values may need special handling */
	      song_convert_commands(&(block->commands[(j+1)*2*block->tracks*block->length+(k*block->tracks+l)*2]), mmd->expdata->mmdcmd3x);
	    }
	}
      }
    }
    /* Instruments */
    song->numinstruments=mmd->song->numsamples;
    song->instruments=(struct instrument **)calloc(song->numinstruments,
						   sizeof(struct instrument *));
    for(i=0; i<song->numinstruments; i++) {
      song->instruments[i]=instrument_alloc();
      if(mmd->expdata) {
	song->instruments[i]->name=(char *)strdup(mmd->expdata->iinfo[i].name);
	song->instruments[i]->hold=mmd->expdata->exp_smp[i].hold;
      }
      song->instruments[i]->midichannel=mmd->song->sample[i].midich-1;
      song->instruments[i]->midipreset=mmd->song->sample[i].midipreset;
      song->instruments[i]->transpose=mmd->song->sample[i].strans;
      if(mmd->song->sample[i].svol==64)
	song->instruments[i]->defaultvelocity=127;
      else
	song->instruments[i]->defaultvelocity=mmd->song->sample[i].svol*2;
    }

    /* Track volumes */
    song->maxtracks=mmd->song->numtracks;
    song->trackvolumes=(unsigned int *)calloc(song->maxtracks,
					      sizeof(unsigned int));
    for(i=0; i<song->maxtracks; i++) {
      if(mmd->song->trackvols[i]<64)
	song->trackvolumes[i]=mmd->song->trackvols[i]*2;
      else
	song->trackvolumes[i]=127;
    }
    if(mmd->song->mastervol<64)
      song->mastervolume=mmd->song->mastervol*2;
    else
      song->mastervolume=127;

    /* SysEX dumps */
    if(mmd->expdata && mmd->expdata->dumps) {
      /* Is this really the only way to do this? I know, this is horrible but
       * this is what the lame MMD spec says */
      struct MMDDump **dumps=(struct MMDDump **)((char *)mmd->expdata->dumps+
						 sizeof(struct MMDDumpData));

      song->numsysexes=mmd->expdata->dumps->numdumps;
      song->sysexes=(struct sysex **)calloc(song->numsysexes,
						   sizeof(struct sysex *));
      for(i=0; i<song->numsysexes; i++) {
	song->sysexes[i]=(struct sysex *)calloc(1, sizeof(struct sysex));
	song->sysexes[i]->length=dumps[i]->length;
	song->sysexes[i]->data=(unsigned char *)calloc(song->sysexes[i]->length, sizeof(unsigned char));
	memcpy(song->sysexes[i]->data, dumps[i]->data,
	       song->sysexes[i]->length);

	if(dumps[i]->ext_len>=20)
	  song->sysexes[i]->name=strdup(dumps[i]->name);
      }
    }
  } else {
    /* Nothing can be done in this case... should never happen though */
    free(song);
    song=song_alloc();
  }

  return song;
}

/* Transposes a song */
void song_transpose(struct song *song, int instruments, int halfnotes) {
  int i;

  if(song==NULL) {
    fprintf(stderr, "song_transpose() called with null song\n");
    return;
  }

  for(i=0; i<song->numblocks; i++)
    block_transpose(song->blocks[i], instruments, halfnotes, 0, 0,
		    song->blocks[i]->tracks-1, song->blocks[i]->length-1);
}

/* Converts MMD command values at given location */
void song_convert_commands(unsigned char *command,
			   struct MMDMIDICmd3x *mmdcmd3x) {
  switch(command[0]) {
  case COMMAND_VELOCITY:
    /* Fix volume range to 0..127 */
    if(command[1]==64)
      command[1]=127;
    else
      command[1]*=2;
    break;
  case COMMAND_DELAY:
    /* Command 1F in MMD is both note delay and retrigger */
    if(command[1]>15)
      command[1]=command[1]>>4;
    else
      command[0]=COMMAND_RETRIGGER;
    break;
  case COMMAND_TEMPO:
    /* Values >0xf0 have special meanings in OctaMED */
    if(command[1]>0xf0) {
      switch(command[1]) {
      case 0xf1:
	/* Makes a single note play twice */
	command[0]=COMMAND_RETRIGGER;
	command[1]=3;
	break;
      case 0xf2:
	/* Delays the start of a note by half a line */
	command[0]=COMMAND_DELAY;
	command[1]=3;
	break;
      case 0xf3:
	/* Works like 0FF1 except the note is played three times */
	command[0]=COMMAND_RETRIGGER;
	command[1]=2;
	break;
      case 0xf4:
	/* Delays the note one-third of a line */
	command[0]=COMMAND_DELAY;
	command[1]=2;
	break;
      case 0xf5:
	/* Delays the note two thirds of a line */
	command[0]=COMMAND_DELAY;
	command[1]=4;
	break;
      case 0xfe:
	/* Stops the song playing */
	command[0]=COMMAND_STOP;
	command[1]=0;
	break;
      case 0xff:
	/* Stops the note on the current track */
	command[0]=COMMAND_VELOCITY;
	command[1]=0;
	break;
      default:
	command[0]=0;
	command[1]=0;
	break;
      }
    }
    break;
  case 0x31:
  case 0x32:
  case 0x33:
  case 0x34:
  case 0x35:
  case 0x36:
  case 0x37:
  case 0x38:
  case 0x39:
  case 0x3a:
  case 0x3b:
  case 0x3c:
  case 0x3d:
  case 0x3e:
  case 0x3f:
    if(mmdcmd3x->ctrlr_types[command[0]-0x31]==MCS_TYPE_STD_MSB)
      command[0]=mmdcmd3x->ctrlr_numbers[command[0]-0x31];
    break;
  default:
    break;
  }
}

/* Loads a song from an XML file */
struct song *song_load(char *filename) {
  /* Check file existence and type first */
  FILE *file;

  if((file=fopen(filename, "r"))!=NULL) {
    unsigned char data[4];

    fread(&data, sizeof(char), 4, file);
    fclose(file);

    /* Is it an MMD2 file? */
    if(data[0]==0x4d && data[1]==0x4d && data[2]==0x44 && data[3]==0x32) {
      struct MMD2 *mmd;
      struct song *song;
      mmd=MMD2_load(filename);
      song=song_convert_MMD2(mmd);
      MMD2_free(mmd);
      /* If it didn't succeed either just allocate a new song */
      if(song!=NULL)
	return song;
      else
	return song_alloc();
    } else {
      /* Try loading it as XML */     
      xmlDocPtr doc;
      xmlNodePtr cur;
  
      /* Build an XML tree from a the file */
      if(!(doc=xmlParseFile(filename)))
	return song_alloc();

      /* Go ahead and parse the document */
      cur=xmlDocGetRootElement(doc);
      return song_parse(doc, NULL, cur);
    }
  } else
    return song_alloc();
}

/* Parses a song element in an XML file */
struct song *song_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur) {
  struct song *song=NULL;
  
  if((!xmlStrcmp(cur->name, "song")) && (cur->ns==ns)) {
    char *prop;
    /* Allocate song */
    song=(struct song *)calloc(1, sizeof(struct song));
    song->name=xmlGetProp(cur, "name");
    prop=xmlGetProp(cur, "tempo");
    if(prop!=NULL)
      song->tempo=atoi(prop);
    prop=xmlGetProp(cur, "ticksperline");
    if(prop!=NULL)
      song->ticksperline=atoi(prop);
    prop=xmlGetProp(cur, "mastervolume");
    if(prop!=NULL)
      song->mastervolume=atoi(prop);
    prop=xmlGetProp(cur, "sendsync");
    if(prop!=NULL)
      song->sendsync=atoi(prop);

    cur=cur->xmlChildrenNode;
    while(cur!=NULL) {
      if((!xmlStrcmp(cur->name, "blocks")) && (cur->ns==ns)) {
	/* Parse and add all block elements */
	xmlNodePtr temp=cur->xmlChildrenNode;

	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int number=-1;
	    struct block *block=block_parse(doc, ns, temp);

	    if(block!=NULL) {
	      prop=xmlGetProp(temp, "number");
	      if(prop!=NULL)
		number=atoi(prop);
	      
	      if(number>=song->numblocks) {
		int i;
		
		song->blocks=realloc(song->blocks,
				     (number+1)*sizeof(struct block *));
		for(i=song->numblocks; i<=number; i++)
		  song->blocks[i]=NULL;
	      }
	      if(song->blocks[number]!=NULL)
		block_free(song->blocks[number]);
	      song->blocks[number]=block;
	      song->numblocks=number+1;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "sections")) && (cur->ns==ns)) {
	/* Parse and add all section elements */
	xmlNodePtr temp=cur->xmlChildrenNode;

	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int number=-1;

	    /* The section number is required */
	    if((!xmlStrcmp(temp->name, "section")) && (temp->ns==ns)) {
	      xmlNodePtr temp2;
	      prop=xmlGetProp(temp, "number");
	      if(prop!=NULL)
		number=atoi(prop);

	      if(number>=song->numsections) {
		int i;
		
		song->sections=realloc(song->sections,
				       (number+1)*sizeof(unsigned int));
		for(i=song->numsections; i<=number; i++)
		  song->sections[i]=0;
	      }

	      /* Get playing sequence */
	      temp2=temp->xmlChildrenNode;
	      if(temp2!=NULL)
		song->sections[number]=atoi(temp2->content);
	      song->numsections=number+1;
	    } else
	      fprintf(stderr, "XML error: expected section, got %s\n",
		      temp->name);
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "playingsequences")) && (cur->ns==ns)) {
	/* Parse and add all playingsequence elements */

	xmlNodePtr temp=cur->xmlChildrenNode;
	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int number=-1;
	    struct playseq *playseq=playseq_parse(doc, ns, temp);

	    if(playseq!=NULL) {
	      prop=xmlGetProp(temp, "number");
	      if(prop!=NULL)
		number=atoi(prop);
	      
	      if(number>=song->numplayseqs) {
		int i;
		
		song->playseqs=realloc(song->playseqs,
				       (number+1)*sizeof(struct playseq *));
		for(i=song->numplayseqs; i<=number; i++)
		  song->playseqs[i]=NULL;
	      }
	      if(song->playseqs[number]!=NULL)
		playseq_free(song->playseqs[number]);
	      song->playseqs[number]=playseq;
	      song->numplayseqs=number+1;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "instruments")) && (cur->ns==ns)) {
	/* Parse and add all instrument elements */
	xmlNodePtr temp=cur->xmlChildrenNode;

	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int number=-1;
	    struct instrument *instrument=instrument_parse(doc, ns, temp);

	    if(instrument!=NULL) {
	      prop=xmlGetProp(temp, "number");
	      if(prop!=NULL)
		number=atoi(prop);
	      
	      if(number>=song->numinstruments) {
		int i;
		
		song->instruments=realloc(song->instruments,
				       (number+1)*sizeof(struct instrument *));
		for(i=song->numinstruments; i<=number; i++)
		  song->instruments[i]=NULL;
	      }
	      if(song->instruments[number]!=NULL)
		instrument_free(song->instruments[number]);
	      song->instruments[number]=instrument;
	      song->numinstruments=number+1;
	    }
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "trackvolumes")) && (cur->ns==ns)) {
	/* Parse and add all track volume elements */
	xmlNodePtr temp=cur->xmlChildrenNode;

	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int track=-1;

	    /* The track number is required */
	    if((!xmlStrcmp(temp->name, "trackvolume")) && (temp->ns==ns)) {
	      xmlNodePtr temp2;
	      prop=xmlGetProp(temp, "track");
	      if(prop!=NULL)
		track=atoi(prop);

	      if(track>=song->maxtracks) {
		int i;
		
		song->trackvolumes=realloc(song->trackvolumes,
					   (track+1)*sizeof(unsigned int));
		for(i=song->maxtracks; i<=track; i++)
		  song->trackvolumes[i]=0;
	      }

	      /* Get the volume */
	      temp2=temp->xmlChildrenNode;
	      if(temp2!=NULL)
		song->trackvolumes[track]=atoi(temp2->content);
	      song->maxtracks=track+1;
	    } else 
	      fprintf(stderr, "XML error: expected section, got %s\n",
		      temp->name);
	  }
	  
	  temp=temp->next;
	}
      } else if((!xmlStrcmp(cur->name, "sysexes")) && (cur->ns==ns)) {
	/* Parse and add all SysEx elements */
	xmlNodePtr temp=cur->xmlChildrenNode;

	while(temp!=NULL) {
	  if(!(xmlIsBlankNode(temp))) {
	    int number=-1;
	    struct sysex *sysex=sysex_parse(doc, ns, temp);

	    if(sysex!=NULL) {
	      prop=xmlGetProp(temp, "number");
	      if(prop!=NULL)
		number=atoi(prop);
	      
	      if(number>=song->numsysexes) {
		int i;
		
		song->sysexes=realloc(song->sysexes,
				       (number+1)*sizeof(struct sysex *));
		for(i=song->numsysexes; i<=number; i++)
		  song->sysexes[i]=NULL;
	      }
	      if(song->sysexes[number]!=NULL) {
		free(song->sysexes[number]->data);
		free(song->sysexes[number]);
	      }
	      song->sysexes[number]=sysex;
	      song->numsysexes=number+1;
	    }
	  }
	  
	  temp=temp->next;
	}
      }
      cur=cur->next;
    }
  } else
    fprintf(stderr, "XML error: expected song, got %s\n", cur->name);

  return song;
}

/* Saves a song to an XML file */
void song_save(struct song *song, char *filename) {
  xmlDocPtr doc;
  xmlNodePtr node, subnode;
  int i;
  char c[10];

  if(song==NULL) {
    fprintf(stderr, "song_save() called with null song\n");
    return;
  }

  doc=xmlNewDoc("1.0");
  doc->xmlRootNode=xmlNewDocNode(doc, NULL, "song", NULL);
  xmlSetProp(doc->xmlRootNode, "name", song->name);
  snprintf(c, 10, "%d", song->tempo);
  xmlSetProp(doc->xmlRootNode, "tempo", c);
  snprintf(c, 10, "%d", song->ticksperline);
  xmlSetProp(doc->xmlRootNode, "ticksperline", c);
  snprintf(c, 10, "%d", song->mastervolume);
  xmlSetProp(doc->xmlRootNode, "mastervolume", c);
  snprintf(c, 10, "%d", song->sendsync);
  xmlSetProp(doc->xmlRootNode, "sendsync", c);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "blocks", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all blocks */
  for(i=0; i<song->numblocks; i++)
    block_save(song->blocks[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "sections", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all sections */
  for(i=0; i<song->numsections; i++) {
    snprintf(c, 10, "%d", song->sections[i]);
    subnode=xmlNewChild(node, NULL, "section", c);
    snprintf(c, 10, "%d", i);
    xmlSetProp(subnode, "number", c);
    xmlAddChild(node, xmlNewText("\n"));
  }
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "playingsequences", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all playing sequences */
  for(i=0; i<song->numplayseqs; i++)
    playseq_save(song->playseqs[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "instruments", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all instruments */
  for(i=0; i<song->numinstruments; i++)
    instrument_save(song->instruments[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "trackvolumes", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all track volumes */
  for(i=0; i<song->maxtracks; i++) {
    snprintf(c, 10, "%d", song->trackvolumes[i]);
    subnode=xmlNewChild(node, NULL, "trackvolume", c);
    snprintf(c, 10, "%d", i);
    xmlSetProp(subnode, "track", c);
    xmlAddChild(node, xmlNewText("\n"));
  }
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node=xmlNewChild(doc->xmlRootNode, NULL, "sysexes", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all SysEx messages */
  for(i=0; i<song->numsysexes; i++)
    sysex_save(song->sysexes[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  /* Use stdout if no filename is specified */
  if(filename==NULL)
    filename="-";

  xmlSaveFile(filename, doc);
  xmlFreeDoc(doc);
}
